/* 
 * diskdevice-model - simulating an HP97560 disk devices, and the bus.
 *    THIS FILE IS INCLUDED BY DISKDEVICE.C.
 *
 * exported functions:
 *   DiskDeviceInit
 *   DiskDeviceDone
 *   DiskDeviceTransfer
 *   DiskDeviceStart
 *   DiskDeviceFinish
 *   DiskDeviceSync
 *   DiskDeviceShape
 *
 * David Kotz 1993-94, Song Bac Toh 1994
 *
 *  There is a Dartmouth College Technical Report that describes this code.
 * The number is PCS-TR94-220.  It can be found for ftp at cs.dartmouth.edu.
 */

/* $Id: diskdevice-model.c,v 2.0 94/07/17 18:34:22 dfk Exp $ */

#ifdef DFK
# include "diskdevice-dfk.h"  /* only used when in context of DFK's program */
#else
# include "modularize.h"      /* miscellaneous stuff */
#endif DFK

#include "diskevent.h"	      /* link to diskevent.c */
#include "diskdevice.h"	      /* defines interface to this module */
#include "diskdevices.h"      /* defines details about different devices */

/* debugging */
/* you probably want to turn on tracing in diskevent.c too */
/* #define DEBUG */
#ifdef DEBUG
# define DEBUG_TRACE(x) {CYCLE_COUNTING_OFF; printf x; CYCLE_COUNTING_ON;}
#else
# define DEBUG_TRACE(x) {}
#endif DEBUG

static boolean first_init = TRUE;


/* @SUBTITLE "Per-disk status structure" */
/* structure that contains information about status of the disk */
typedef struct diskstatus
{
    boolean done;	      /* is the device done being used? */

    /* when the current request started and finished */
    TICS startTime;	      /* when request started (NEVER if none) */    
    TICS finishTime;	      /* when previous request finished */

    void *Pointer;            /* Added by Yiming Hu, 1996.
                               * A general purpose pointer.
                               * Can be used to the event handler to decide
                               * the caller of the disk
                               */
    TICS LastStartTime;       /* Added by Yiming Hu, 1996
			       * Same as startTime, but will not be set
			       * to NEVER when disk is done. Used to 
			       * report disk start/finish time
			       */
    boolean reading;	      /* TRUE if reading, FALSE if writing */

    ulong      SavedSector;   /* Added by Yiming Hu */
    ulong      SavedNSectors; /*  Since the DiskDeviceTransfer() function */
    UserData   SavedBuffer;   /*  is broken into 2 functions, some interface */
                              /*  variables have to be saved so the 2nd */
                              /*  function can use them */ 

    /* a way to communicate a new request from DiskDeviceXfer to Controller */
    ulong newSector;	      /* the first sector to transfer  */
    ulong newLastSector;      /* the last sector to transfer */
    boolean newReading;	      /* TRUE if reading, FALSE if writing */
    boolean newHaveBuffer;    /* does the host have the buffer yet? */

    /* state of the disk - which track are heads at now? */
    boolean track;	      /* current track */
    /* There is no need to keep track of the disk rotational position since
     * we are assuming that disk head is at position 0 at time 0.
     * The range for disk head position is 0~1 with (position 0 == position 1)
     */ 

    boolean activeBusXfer;    /* is this bus actively using the bus? */
    boolean activeRead;	      /* is a read diskxfer happening now? */
    boolean cancelPrefetch;   /* should we cancel an outstanding prefetch? */
    DDEvent *prefetchReq;     /* a cancellable EndDiskXfer request */
    boolean needMove;	      /* do we need DiskMove before restarting xfers */
    TICS    wantBus;	      /* after what time would we like to use bus */
    boolean haveBuffer;	      /* do we have the buffer at host end of bus? */
    boolean wantSync;	      /* host wants to know when write-behind done */
    TICS    restartController; /* when should Controller() be run again */

/* Each variable below contains a logical sector number.
 *  firstCachedSector always contains the sector of the lowest-numbered
 * sector in the cache.
 *  nextDiskXfer and nextBusXfer always contain the number of the
 * sector to be transferred next.
 *  currentDiskXfer and currentBusXfer contains the number of the
 * sector currently being transferred.
 *  
 * Some invariants:
 *  When reading, 
 *    FCS <= CBX <= NBX <= CDX <= NDX
 *    FCS is not actually in the cache unless FCS < NBX
 *  When writing, 
 *    FCS <= CDX <= NDX <= CBX <= NBX
 *    FCS is not actually in the cache unless FCS < NDX
 *  NBX-CBX == 0 or 1
 *    if 0, no bus transfer is active
 *    if 1, sector CBX is going across the bus
 *  NDX-CDX == 0 or 1
 *    if 0, no disk transfer is active
 *    if 1, sector CBX is going across the bus
 */
    ulong firstCachedSector;  /* sector at head of cache */
    ulong currentDiskXfer;    /* sector currently being transferred */
    ulong nextDiskXfer;	      /* sector to be transferred next */
    ulong currentBusXfer;     /* sector currently being transferred */
    ulong nextBusXfer;	      /* sector to be transferred next */

/* macros to find state of disk cache  */
#define CACHE_SLOTS_USED(disk) ((dstatus[disk].reading ?     \
				dstatus[disk].nextDiskXfer \
				:dstatus[disk].nextBusXfer) \
    	    	    	        - dstatus[disk].firstCachedSector)
#define CACHE_EMPTY(disk)      (CACHE_SLOTS_USED(disk) <= 0)
#define CACHE_FULL(disk)       (CACHE_SLOTS_USED(disk) >= CACHE_SLOTS)

    ulong firstSector;	      /* first sector involved in current request */
    ulong lastSector;	      /* last sector involved in current request */
    int wait_thread;	      /* what thread (STID) is waiting on request */
    int busId;		      /* which bus do we use (for tracing)? */
    int *busOwner;	      /* (*busOwner == BUS_FREE_ ==> bus is free */
    NCQUEUE *busq;	      /* place to queue disks waiting for bus */

    FILE *datafile;	      /* Unix file to read/write the data from/to */

} diskstatus;

/* data structure keeping the status of the disks */
PRIVATE struct diskstatus dstatus[NO_OF_DISKS];


/* Records to go on the busq; 
 * basically, an event to schedule when bus is free 
 */
typedef struct busitem {
  int disk;
  REQUESTCODE event;
} busItem;


/* Added by Yiming Hu */
TICS GetDiskFinishedTime(int disk)
{
    return dstatus[disk].finishTime;
}


/* @SUBTITLE "Utility functions" */
PRIVATE TICS MoveTime_t(int disk, ulong lsector, TICS eventTime);
PRIVATE TICS MoveNextTime_t(int disk, ulong lsector, TICS eventTime); 
PRIVATE ulong Logical2Physical(ulong lsector);
PRIVATE TICS RotDelay_t(int disk, ulong lsector, TICS eventTime);
PRIVATE TICS SeekTime_t(ulong dcylinders, ulong dtracks, TICS eventTime);
PRIVATE void SanityCheck(int disk, ulong lsector, ulong nsectors);
PRIVATE boolean GrabBus(int disk, TICS *grabTime, TICS eventTime);
PRIVATE void WaitForBus(int disk, REQUESTCODE event, TICS eventTime);
PRIVATE void PassBus(int disk, TICS eventTime);

/* Print out the requests scheduled (in diskevent.c) */
extern void DDPrintRequests(void);

/* @SUBTITLE "Event-handler functions" */
PRIVATE void DiskMove(int disk, TICS now);
PRIVATE void StartDiskXfer(int disk, TICS now);
PRIVATE void EndDiskXfer(int disk, TICS now);
PRIVATE void ConsiderBusXfer(int disk, TICS now);
PRIVATE void StartBusXfer(int disk, TICS now);
PRIVATE void EndBusXfer(int disk, TICS now);
PRIVATE void SendCommand(int disk, TICS now);
PRIVATE void EndCommand(int disk, TICS now);
PRIVATE void Controller(int disk, TICS now);
PRIVATE void SendSyncCommand(int disk, TICS now);
PRIVATE void EndSyncCommand(int disk, TICS now);
PRIVATE void SendBufferMesg(int disk, TICS eventTime);
PRIVATE void EndBufferMesg(int disk, TICS eventTime);
PRIVATE void SendDoneMessage(int disk, TICS now);
PRIVATE void EndDoneMessage(int disk, TICS now);

/* possible types of requests */
struct dd_handlers DDhandlers[] = 
{
    { NULL, "NOP" },
    { DiskMove, "DiskMove" },
    { StartDiskXfer, "StartDiskXfer" },
    { EndDiskXfer, "EndDiskXfer"},
    { ConsiderBusXfer, "ConsiderBusXfer" },
    { StartBusXfer, "StartBusXfer" },
    { EndBusXfer, "EndBusXfer" },
    { SendCommand, "SendCommand" },
    { EndCommand, "EndCommand" },
    { Controller, "Controller" },
    { SendSyncCommand, "SendSyncCommand" },
    { EndSyncCommand, "EndSyncCommand" },
    { SendBufferMesg, "SendBufferMesg" },
    { EndBufferMesg, "EndBufferMesg" },
    { SendDoneMessage, "SendDoneMessage" },
    { EndDoneMessage, "EndDoneMessage" }
};

/* ======================================================================== */

/* @SUBTITLE "DiskDeviceInit: initialize a disk" */
/* The caller must provide a pointer to a boolean, which  
 * we use to tell when that bus is free.  We also need a NCQUEUE, with room 
 * for as many disks as are attached to that bus, where disks can wait
 * for service on the bus.   Presumably, several disks share
 * the same values; thus, this is how we handle bus contention.
 * It must not be deallocated until after DiskDeviceDone is called.
 * The busId is used for debugging and tracing, nothing functional.
 */
void
DiskDeviceInit(int disk, int busId, int *busOwner, NCQUEUE *busWaitq,
	       char *diskFileName)
{
    struct diskstatus *ds = &(dstatus[disk]);
  
    if (first_init) {
	/* initialize the disk event subsystem, once only */
	DDEventInit();
	first_init = FALSE;
    }

    /* initialize the disk status */
    ds->track = 0;	      /* physical track of the disk */
    ds->activeRead = FALSE;
    ds->activeBusXfer =  FALSE;
    ds->done = FALSE;
    ds->wantBus = NEVER;
    ds->needMove = FALSE;
    ds->cancelPrefetch = FALSE;
    ds->firstSector   = 0;
    ds->lastSector   = 0;
    ds->reading = TRUE;            
    ds->busId = busId;
    ds->busq = busWaitq;
    ds->busOwner = busOwner;
    ds->firstCachedSector = 0; /* Initially cache is empty */
    ds->currentDiskXfer = 0;  /* Initially the two numbers are equal - */
    ds->nextDiskXfer = 0;     /* no disk transfer. */
    ds->currentBusXfer = 0;   /* Initially the two numbers are equal - */
    ds->nextBusXfer = 0;      /* no bus transfer. */
    ds->haveBuffer = FALSE;
    ds->wantSync = FALSE;
    ds->prefetchReq = NULL;

    /* use these to pass parameters to Controller() */
    ds->newSector = 0;	      /* no pending transfer or move request */
    ds->newLastSector = 0;    /* no request is being processed */
    ds->newReading = FALSE;
    ds->newHaveBuffer = FALSE;

    /* use this flag to determine whether we are the first time through
       the controller (NEVER == first time through) */
    ds->restartController = NEVER;

    /* use this flag to check whether a DiskDevice* is finished */
    ds->startTime = NEVER;
    ds->LastStartTime = 0;    /* Added by Yiming Hu, 1996 */
    ds->Pointer = NULL;       /* Added by Yiming Hu, 1996 */
    ds->finishTime = 0;

    /* decide whether to use some Unix file as a pretend "disk" */
    if (diskFileName == NULL)
      ds->datafile = NULL;    /* do not use any disk file; dummy data only */
    else {
	/* try to open the given file as a place for this "disk"s data */
	/* upon error in fopen(), ds->datafile will also be assigned NULL */
	ds->datafile = fopen(diskFileName, "r+");
	/* file does not exist, try to create the file */
	if (ds->datafile == NULL) {
	    ds->datafile = fopen(diskFileName, "w+");
	    /* can't even create it, fail it */
	    INVARIANT3(ds->datafile != NULL,
		       "DiskDeviceInit: disk file %s cannot be opened\n",
		       diskFileName);
	}
    }

    INDEX_TIME_EVENT(EV_DISKSTATE, disk, DISKSTATE_IDLE, 0);
}



/* Added by Yiming Hu, 04/11/96 */
boolean
DiskIsBusy(int disk)
{
    return (dstatus[disk].startTime != NEVER);
}


/* Added by Yiming Hu, 08/09/96 */
boolean
DiskWaitingSync(int disk)
{
    return dstatus[disk].wantSync ||
           (dstatus[disk].currentDiskXfer < dstatus[disk].currentBusXfer);
}


/* Added by Yiming Hu, 08/15/96 */
boolean
bImmediateReportDisk(void)
{
#ifdef IMMEDIATE_REPORT
    return TRUE;
#else
    return FALSE;
#endif
}


void GetDiskTime(int disk, TICS *StartTime, TICS *FinishTime)
{
    *StartTime  = dstatus[disk].LastStartTime;
    *FinishTime = dstatus[disk].finishTime;   
}

void *GetDiskPointer(int disk)
{
    return dstatus[disk].Pointer;
}


/* @SUBTITLE "DiskDeviceDone: cleanup" */
void
DiskDeviceDone(int disk)
{
    struct diskstatus *ds = &(dstatus[disk]);
  
    INVARIANT3(!ds->activeBusXfer, 
	       "DiskDeviceDone: disk %d bus transfer still active\n", disk);
    INVARIANT3(ds->wantBus == NEVER, 
	       "DiskDeviceDone: disk %d still wants the bus\n", disk);
    INVARIANT3(ds->startTime == NEVER,
	       "DiskDeviceDone %d: previous DiskDeviceStart not finished!\n",
	       disk);
    INVARIANT3(ds->currentDiskXfer >= ds->currentBusXfer,
	       "DiskDeviceDone %d: needs Sync!\n", disk);

    DEBUG_TRACE(("DiskDeviceDone \t%d\t@%s\n", disk, time_print(GetTime())));

    ds->firstCachedSector = 0;  /* reset the cache */
    ds->currentDiskXfer = ds->nextDiskXfer = 0;   
    ds->currentBusXfer = ds->nextBusXfer = 0;

    if (ds->datafile != NULL) 
      fclose(ds->datafile);		/* close the datafile */

    ds->done = TRUE;

    /* record the final idle period */
    if (ds->finishTime > 0)   /* this disk was used at some point */
      SUM_ARRAY_METRIC(stats.diskidle, disk, 
		       (GetTime() - ds->finishTime) * MSECperTIC);
    INDEX_TIME_EVENT(EV_DISKIDLE, disk, 0, ds->finishTime);
    INDEX_TIME_EVENT(EV_DISKSTATE, disk, DISKSTATE_IDLE, ds->finishTime);
}

/* @SUBTITLE "DiskDeviceStart: start a disk transaction" */
/* Like DiskDeviceTransfer, but split into two parts: the first part
 * starts the disk moving to the appropriate location, and the second supplies
 * the buffer needed to complete the transaction. Must be followed by a 
 * corresponding Finish.
 */ 
void
DiskDeviceStart(int disk, ulong sector, ulong nsectors, boolean write)
{    
    TICS eventTime;
    struct diskstatus *ds = &(dstatus[disk]);

    /* check whether the caller of this function is insane */
    SanityCheck(disk, sector, nsectors);
  
    /* record the preceding idle period */
    if (ds->finishTime > 0)   /* this disk has been used before */
      SUM_ARRAY_METRIC(stats.diskidle, disk, 
		       (GetTime() - ds->finishTime) * MSECperTIC);
    INDEX_TIME_EVENT(EV_DISKIDLE, disk, 0, ds->finishTime);

    /* record the event of reading or writing */
    INDEX_EVENT(write ? EV_DISKWRITE : EV_DISKREAD, disk, sector);
    SUM_ARRAY_METRIC(write ? stats.diskwrite : stats.diskread, disk, 1.0);
    INDEX_EVENT(EV_DISKBUSY, disk, 0);
  
    eventTime = GetTime();
    ds->startTime = eventTime;     /* start the disk */
    ds->LastStartTime = eventTime; /* Added by Yiming Hu, 1996 */
  
    DEBUG_TRACE(("DiskDeviceStart \t%d\t@%s: sector:%lu  nsectors:%lu  R/W:%d\n",
		 disk, time_print(eventTime), sector, nsectors, write));

    ds->newReading = !write;	
    ds->newSector = sector;
    /* go through all sectors */
    ds->newLastSector = sector + nsectors - 1;

    ds->newHaveBuffer = FALSE; /* we don't have the buffer yet */

    DEBUG_TRACE(("\t"));
    SendCommand(disk, eventTime); /* schedules EndCommand */

    DEBUG_TRACE(("\tDiskDeviceStart \t%d\t@%s  suspending %d/%d\n",
		 disk, time_print(eventTime), CURR_PROCESSOR, MY_TID));

    /* now wait for EndCommand to wake us up */
    ds->wait_thread = MY_STID;
    thread_suspend(MY_TID);
}

/* @SUBTITLE "DiskDeviceFinish: finish a disk transaction" */
/* See DiskDeviceStart.  This, the second call, 
 * BLOCKs until the transfer is complete.  There should not have been
 * any other requests since the corresponding Start.  Ideally, we should
 * not be late in calling DiskDeviceFinish, ie, not after the disk movement
 * has finished.
 */ 
void
DiskDeviceFinish(int disk, UserData buffer)
{    
    struct diskstatus *ds = &(dstatus[disk]);
    TICS eventTime = GetTime();
    ulong xferSize = ds->lastSector - ds->firstSector + 1;

    INVARIANT3(ds->startTime != NEVER,
	       "DiskDeviceFinish %d: no corresponding DiskDeviceStart\n",
	       disk);
    DEBUG_TRACE(("DiskDeviceFinish \t%d\n", disk));

    /* process the request */
    ds->newHaveBuffer = TRUE; /* EndBufferMesg set hasBuffer */

    DEBUG_TRACE(("\t"));
    SendBufferMesg(disk, eventTime);

    DEBUG_TRACE(("\tDiskDeviceFinish \t%d\t@%s  suspending %d/%d\n",
		 disk, time_print(eventTime), CURR_PROCESSOR, MY_TID));

    ds->wait_thread = MY_STID;
    thread_suspend(MY_TID);   /* send the thread to sleep */

    ds->finishTime = GetTime();

    AVG_ARRAY_METRIC(stats.diskwait, disk, 
		     (ds->finishTime - ds->startTime) * MSECperTIC);
    STDDEV_ARRAY_METRIC(stats.diskwait_sd, stats.diskwait, disk, 
			(ds->finishTime - ds->startTime) * MSECperTIC);

    ds->startTime = NEVER;    /* reset the disk */

    /* actually read and write the data to vdisk */
    if (ds->datafile != NULL) {
	fseek(ds->datafile, (long)(ds->firstSector * SECTOR_SIZE), SEEK_SET);

	if (ds->reading)
	  fread(buffer, (size_t)SECTOR_SIZE, xferSize, ds->datafile);
	else {
	    INVARIANT2(fwrite(buffer, (size_t)SECTOR_SIZE, xferSize, 
			      ds->datafile) == xferSize,
		       "DiskDeviceFinish: error while writing to data file\n");
	}
    }
}

/* @SUBTITLE "DiskDeviceShape: return the shape of the disk" */
/* Return the shape of the disk. */
void 
DiskDeviceShape(int disk, ulong *nSectors, ulong *sectorSize,
		ulong *nTracks, ulong *sectorsPerTrack)
{
    *nSectors = TOTAL_NSECTORS;
    *sectorSize = SECTOR_SIZE;
    *nTracks = NTRACKS;
    *sectorsPerTrack = NSECTORS;
}



/* ======================================================================= */
/* @SUBTITLE "DiskDeviceTransfer: a complete disk transaction" */
/* Do a disk operation. We give it everything it needs to know: the disk
 * number, sector number on that disk, length in sectors,
 * a read/write flag, and the buffer for the data.   
 * This BLOCKS until the time when the transfer would be complete.
 */
void
DiskDeviceTransfer(int disk, ulong sector, ulong nsectors,
		   boolean write, UserData buffer, void *Pointer)
{
    TICS eventTime;
    struct diskstatus *ds = &(dstatus[disk]);

  
    /* check whether the caller of this function is insane */
    SanityCheck(disk, sector, nsectors);
  
    /* Save paramaters so the next call (DiskDeviceTransferEnd) can use them
     * to read/write data from/to the data file.
     * Added by Yiming Hu, Jan 1997
     */
    ds->SavedSector   = sector;
    ds->SavedNSectors = nsectors;
    ds->SavedBuffer   = buffer;


    /* record the preceding idle period */
    if (ds->finishTime > 0) { /* this disk has been used before */
	SUM_ARRAY_METRIC(stats.diskidle, disk, 
			 (GetTime() - ds->finishTime) * MSECperTIC);
	INDEX_TIME_EVENT(EV_DISKIDLE, disk, 0, ds->finishTime);
    }

    /* record the event of reading or writing */
    INDEX_EVENT(write ? EV_DISKWRITE : EV_DISKREAD, disk, sector);
    SUM_ARRAY_METRIC(write ? stats.diskwrite : stats.diskread, disk, 1.0);
    INDEX_EVENT(EV_DISKBUSY, disk, 0);
  
    eventTime = GetTime();
    ds->startTime = eventTime; /* start the disk */
    ds->LastStartTime = eventTime; /* Added by Yiming Hu, 1996 */
    ds->Pointer = Pointer;         /* Added by Yiming Hu, 1996 */
 
    DEBUG_TRACE(("DiskDeviceXfer \t%d\t@%s: sector:%lu  nsectors:%lu  R/W:%d\n",
		 disk, time_print(eventTime), sector, nsectors, write));

    /* only Controller() can change the values */
    ds->newReading = !write;	
    ds->newSector = sector;
    /* go through all sectors */
    ds->newLastSector = sector + nsectors - 1;
    ds->newHaveBuffer = TRUE;

    /* ask the controller to process the request */
    DEBUG_TRACE(("\t"));
    SendCommand(disk, eventTime);

    DEBUG_TRACE(("\tDiskDeviceTransfer \t%d\t@%s  suspending %d/%d\n",
		 disk, time_print(eventTime), CURR_PROCESSOR, MY_TID));

/*
    ds->wait_thread = MY_STID;
    thread_suspend(MY_TID); */  /* send the thread to sleep */

}


/*
 *    The DiskDeviceTransfer() function is broken into two parts.
 *    The first part (see above) registers some parameters and 
 *    returns to the caller immediately to avoid blocking. A system
 *    event handler then processes all events, including the disk
 *    events. When the request is fisished, this following function
 *    is called by the event handler.
 *                                                Yiming Hu
 */
void
DiskDeviceTransferEnd(int disk)
{
    TICS eventTime;
    struct diskstatus *ds = &(dstatus[disk]);

    ds->finishTime = GetTime();

    AVG_ARRAY_METRIC(stats.diskwait, disk, 
		     (ds->finishTime - ds->startTime) * MSECperTIC);
    STDDEV_ARRAY_METRIC(stats.diskwait_sd, stats.diskwait, disk, 
			(ds->finishTime - ds->startTime) * MSECperTIC);

    ds->startTime = NEVER;    /* reset the disk */
    ds->Pointer   = NULL;     /* Added by Yiming Hu, 1996 */

    /* actually read and write the data to vdisk */
    /* Some parameters are changed by Yiming Hu */
    if (ds->datafile != NULL) {
	fseek(ds->datafile, ((long) ds->SavedSector * SECTOR_SIZE), SEEK_SET);

	if (ds->reading) {
	  fread(ds->SavedBuffer, (size_t)SECTOR_SIZE, ds->SavedNSectors, ds->datafile);

	} else {
	  INVARIANT2(fwrite(ds->SavedBuffer, (size_t)SECTOR_SIZE, ds->SavedNSectors, 
			    ds->datafile) == ds->SavedNSectors,
		     "DiskDeviceTransfer: error while writing to data file\n");
        }
    }
}


/* @SUBTITLE "DiskDeviceSync: Synchronizing the disk." */
/* All the data in the disk cache is written to the disk before returning.
 */
void 
DiskDeviceSync(int disk)
{
    TICS eventTime;
    struct diskstatus *ds = &(dstatus[disk]);

    INVARIANT3(ds->startTime == NEVER,
	       "DiskDeviceSync %d: previous DiskDeviceStart not finished!\n",
	       disk);
    DEBUG_TRACE(("DiskDeviceSync \t%d\n", disk));
  
    eventTime = GetTime();
    SendSyncCommand(disk, eventTime);

      /* send the thread to sleep, SendDoneMessage */
/*    ds->wait_thread = MY_STID; */
/*    thread_suspend(MY_TID);    */
}


/* ======================================================================= */
/* @SUBTITLE "SendCommand: send a command to the disk" */
/* Called by DiskDeviceTransfer to send a command to the disk. 
 * It schedules an EndCommand() call in the future. 
 */
PRIVATE void
SendCommand(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 
    TICS grabTime = 0;	      /* time it took to grab the bus */

    DEBUG_TRACE(("SendCommand\t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));
  
    if (GrabBus(disk, &grabTime, eventTime))
      {
	  DDEventRequest(ENDCOMMAND, (eventTime + grabTime + COMMANDTIME_t), disk);

	  if (grabTime > 0)
	    INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_GRAB, eventTime);
	  INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_COMMAND, 
			   eventTime + grabTime);
      }
    else
      {
	  DEBUG_TRACE(("SendCommand:  Waiting for bus\n"));
	  WaitForBus(disk, SENDCOMMAND, eventTime);
      }    
}

/* @SUBTITLE "EndCommand: end command xfer, start controller" */
/* Scheduled by SendCommand() to end a command 
 * and schedule a Controller() event. 
*/
PRIVATE void
EndCommand(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 
  
    DEBUG_TRACE(("EndCommand\t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));
  
    /* give the bus to the next event on the bus queue */
    PassBus(disk, eventTime);

    /* if command was sent by DiskDeviceStart, wake it up */
    /* we can tell DiskDeviceStart sent command, if there is no buffer */
    if (!ds->newHaveBuffer) {
      printf("Do wakeup..., in EndCommand\n");
      DDEventWakeup(ds->wait_thread, eventTime);
    }
    /* start the controller */
    Controller(disk, eventTime);
}


/* @SUBTITLE "Controller: examine new request and get disk going." */
/* Scheduled by EndCommand() to start a disk
 * read or write operation and then go to sleep.  
 * Note that this function may be called twice in some requests; if the
 * previous request is a write that was immediately reported, and this
 * request does not extend it, we pause until that previous write completes
 * to disk; then the Controller is "restarted" to consider the new request.
 */

PRIVATE void
Controller(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]);   
    ulong sector;
    TICS endCtrllerTime;
    boolean extendWrite = FALSE; /* can we extend the current write request? */

    DEBUG_TRACE(("Controller\t%d\t@%s: nextDiskXfer = %lu  currentBusXfer:%lu\n",
		 disk, time_print(eventTime), 
		 ds->nextDiskXfer, ds->currentBusXfer));

    INDEX_TIME_EVENT(EV_DISKSTATE, disk, DISKSTATE_CONTROL, eventTime);

    if (ds->restartController == NEVER)
      {
	  if (ds->newReading)
	    endCtrllerTime = eventTime + CTRLOVHDREAD_t;
	  else
	    endCtrllerTime = eventTime + CTRLOVHDWRITE_t;
      }
    else
      endCtrllerTime = eventTime;

    /* Check to see whether the current bus transfer is ahead of the
       current disk transfer.  If TRUE, previous operation was an
       immediate reported write.
       If the incoming operation is a read or a non-connecting write, we
       simply return and let the second call to Controller() deal with
       it.  If the next operation is a sequential write, we extend the
       write and continue.  */
    if (ds->currentDiskXfer < ds->currentBusXfer)
      {
	  if (ds->newReading || (ds->newSector != ds->lastSector + 1))
	    {
		ds->restartController = endCtrllerTime;
		DEBUG_TRACE(("\trestart Controller %d after @%s and EndDiskXfer\n", 
			     disk, time_print(endCtrllerTime)))
		  return;
	    }
	  else		      /* need to extend an immediate-reporting write */
	    {
		ds->restartController = NEVER;
		extendWrite = TRUE;
	    }
      } 
    else		      /* no need to restart */
      ds->restartController = NEVER;
  
    sector = ds->firstSector = ds->newSector;
    ds->lastSector = ds->newLastSector;
    ds->reading = ds->newReading;	
    ds->haveBuffer = ds->newHaveBuffer;
  
    if (ds->reading)	      /* read */
      {
	  /* cache hit? */
	  if (sector >= ds->firstCachedSector && sector < ds->nextDiskXfer)
	    {		      /* cache hit */
		SUM_ARRAY_METRIC(stats.diskhit, disk, 1.0);
		ds->wantBus = endCtrllerTime; /* want to start transferring */
		if (!ds->activeRead) {
		    /* get prefetching started */
		    DDEventRequest(DISKMOVE, endCtrllerTime, disk);
		    ds->needMove = FALSE;
		} else	      /* transfers continue in the background */
		  INDEX_TIME_EVENT(EV_DISKSTATE, disk, 
				   DISKSTATE_XFER, endCtrllerTime);
	  
		/* flush the cache up to that point */
		ds->firstCachedSector = sector;
		ds->nextBusXfer = ds->currentBusXfer = sector;
		/* cache hit on ready item, can start transferring */
		if (sector < ds->currentDiskXfer) 
		  DDEventRequest(CONSIDERBUSXFER, endCtrllerTime, disk);
	    }
	  else
	    {		      /* cache miss */
		SUM_ARRAY_METRIC(stats.diskmiss, disk, 1.0);
		/* reset (flush) the cache */
		ds->firstCachedSector = sector;
		ds->nextDiskXfer = ds->currentDiskXfer = sector;
		ds->nextBusXfer = ds->currentBusXfer = sector;	  
		ds->wantBus = endCtrllerTime;
	  
		if (ds->activeRead)
		  {
		      if (ds->prefetchReq == NULL)
			ds->cancelPrefetch = TRUE;
		      else
			{
			    DDEventCancel(ds->prefetchReq);
			    ds->cancelPrefetch = FALSE;
			    ds->activeRead = FALSE;
			    ds->prefetchReq = NULL;
			    /* move to new request */
			    DDEventRequest(DISKMOVE, endCtrllerTime, disk); 
			    ds->needMove = FALSE;
			}
		  }
		else
		  {
		      DDEventRequest(DISKMOVE, endCtrllerTime, disk); 
		      ds->needMove = FALSE;
		  }
	    }
      }
    else		      /* write */
      {
	  if (!extendWrite)
	    {
		SUM_ARRAY_METRIC(stats.diskmiss, disk, 1.0);
		/* discard all cache only when we are not extending a write */
		ds->firstCachedSector = sector;
		ds->nextDiskXfer = ds->currentDiskXfer = sector;
		ds->nextBusXfer = ds->currentBusXfer = sector;	  
	  
		if (ds->activeRead)
		  {
		      if (ds->prefetchReq == NULL)
			ds->cancelPrefetch = TRUE;
		      else
			{
			    DDEventCancel(ds->prefetchReq);
			    ds->cancelPrefetch = FALSE;
			    ds->activeRead = FALSE;
			    ds->prefetchReq = NULL;
			    /* move to new request */
			    DDEventRequest(DISKMOVE, endCtrllerTime, disk); 
			    ds->needMove = FALSE;
			}
		  }
		else
		  {
		      DDEventRequest(DISKMOVE, endCtrllerTime, disk); 
		      ds->needMove = FALSE;
		  }
	    }
	  else {
	      /* the transfer continues in the background */
	      SUM_ARRAY_METRIC(stats.diskhit, disk, 1.0);
	      /* We want to update firstCachedSector to be the same 
	       * as the new request's firstSector, but we can't until the 
	       * current request finishes.  To allow the bus xfer to go, 
	       * we will allow firstCachedSector to track currentDiskXfer
	       * until it hits firstSector (see also EndDiskXfer).
	       */
	      ds->firstCachedSector = 
		min(ds->currentDiskXfer, ds->firstSector);
	      INDEX_TIME_EVENT(EV_DISKSTATE, disk, 
			       DISKSTATE_XFER, endCtrllerTime);
	  }

	  ds->wantBus = endCtrllerTime;
	  DDEventRequest(CONSIDERBUSXFER, endCtrllerTime, disk);
      }
}


/* @SUBTITLE "SendBufferMesg: Give a buffer to the Controller()" */
/* Give a buffer to the Controller() so that disk transfers can take place */

PRIVATE void
SendBufferMesg(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 
    TICS grabTime = 0;	      /* time it took to grab the bus */
  
    DEBUG_TRACE(("SendBufferMesg\t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));
  
    if (GrabBus(disk, &grabTime, eventTime))
      {
	  DDEventRequest(ENDBUFFERMESG, (eventTime + grabTime + COMMANDTIME_t), 
			 disk);

	  if (grabTime > 0)
	    INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_GRAB, eventTime);
	  INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_COMMAND, 
			   eventTime + grabTime);
      }
    else
      {
	  DEBUG_TRACE(("SendBufferMesg:  Waiting for bus\n"));
	  WaitForBus(disk, SENDBUFFERMESG, eventTime);
      }    
}



/* @SUBTITLE "EndBufferMesg:  Finish sending the buffer" */ 
/* Finish sending the buffer and consider a bus transfer. */

PRIVATE void
EndBufferMesg(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 
  
    DEBUG_TRACE(("EndBufferMesg\t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));
  
    /* give the bus to the next event on the bus queue */
    PassBus(disk, eventTime);

    ds->haveBuffer = TRUE;
  
    /* If the controller is not waiting for disk transfer to finish,
       we can now consider a bus transfer */
    if (ds->restartController == NEVER)
      {
	  DEBUG_TRACE(("\t"));
	  ConsiderBusXfer(disk, eventTime); 
      }
}

/* @SUBTITLE "SendSyncCommand:  ask disk to sync" */
/* Scheduled by DiskDriverSync() to synchronize the disk. */

PRIVATE void
SendSyncCommand(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 
    TICS grabTime = 0;	      /* time it took to grab the bus */
  
    DEBUG_TRACE(("SendSyncCommand\t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));

    if (GrabBus(disk, &grabTime, eventTime))
      {
	  DDEventRequest(ENDSYNCCOMMAND,
			 (eventTime + grabTime + COMMANDTIME_t), disk);

	  if (grabTime > 0)
	    INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_GRAB, eventTime);
	  INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_COMMAND, 
			   eventTime + grabTime);
      }
    else
      {
	  DEBUG_TRACE(("SendSyncCommand:  Waiting for bus\n"));
	  WaitForBus(disk, SENDSYNCCOMMAND, eventTime);
      }    
}



/* @SUBTITLE "EndSyncCommand: set flag to wakeup when disk done" */
/* Scheduled by SendSyncCommand() to synchronize
 * the disk by setting a flag so EndDiskXfer can wake
 * the DiskDriverSync up. 
 */

PRIVATE void
EndSyncCommand(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 

    DEBUG_TRACE(("EndSyncCommand\t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));
  
    /* give the bus to the next event on the bus queue */
    PassBus(disk, eventTime);

    /* only set sync flag when we have outstanding disk operations */
    if (ds->currentDiskXfer < ds-> currentBusXfer)
      {
	  ds->wantSync = TRUE;
	  DEBUG_TRACE(("\tdisk %d want last EndDiskXfer to SendDoneMessage\n", 
		       disk));
      }
    else
      {
	  ds->wantSync = FALSE;
	  /* done syncing coz we didn't have to sync in the first place */
	  DDEventRequest(SENDDONEMESSAGE, eventTime, disk); 
      }
}


/* @SUBTITLE "DiskMove: move disk head before disk xfer" */
/* Compute the time to move the disk head, then schedule StartDiskXfer at
 * the move-completion time.
 */

PRIVATE void
DiskMove(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]);  
    TICS endMoveTime;
  
    endMoveTime = MoveTime_t(disk, ds->nextDiskXfer, eventTime);
  
    ds->activeRead = ds->reading; /* so we know a StartDiskXfer is coming */

    DEBUG_TRACE(("DiskMove \t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));
    DDEventRequest(STARTDISKXFER, endMoveTime, disk);

    INDEX_TIME_EVENT(EV_DISKSTATE, disk, DISKSTATE_MOVE, eventTime);
    SUM_ARRAY_METRIC(stats.diskmoves, disk, 1.0);
}


/* @SUBTITLE "StartDiskXfer: start one-sector disk xfer" */
/* 
 * Start a disk transfer event given the disk and the called time.
 * The disk is at the beginning of nextSector at the given eventTime.
 *
 * !! StartDiskXfer() should only advance ds->nextDiskXfer.  It should
 * !! by no means change ds->currentDiskXfer.
 */

PRIVATE void
StartDiskXfer(int disk, TICS eventTime)
{
    TICS abstime;
    struct diskstatus *ds = &(dstatus[disk]);
  
    DEBUG_TRACE(("StartDiskXfer \t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));
  
    if (ds->done) 
      {
	  INDEX_TIME_EVENT(EV_DISKSTATE, disk, DISKSTATE_IDLE, eventTime);
	  return;
      }
  
    if (ds->cancelPrefetch)
      {
	  ds->cancelPrefetch = FALSE;
	  ds->activeRead = FALSE;
	  DDEventRequest(DISKMOVE, eventTime, disk); /* move to new request */
      }
    else
      {
	  if (ds->reading)    /* reading */
	    {
		/* try to enqueue the next sector onto the queue */
		if (!CACHE_FULL(disk)) 
		  {
		      ds->nextDiskXfer++;
		      abstime = DSECTORXFER_t(eventTime);
		      ds->prefetchReq = DDEventRequest(ENDDISKXFER, 
						       abstime, disk);
		      INDEX_TIME_EVENT(EV_DISKSTATE, disk,
				       DISKSTATE_XFER, eventTime);
		  }
		else
		  {
		      /* cache is full */
		      ds->activeRead = FALSE;
		      ds->needMove = TRUE;
		      INDEX_TIME_EVENT(EV_DISKSTATE, disk, 
				       DISKSTATE_IDLE, eventTime);
		  }
	    }    
	  else		      /* writing */
	    {
		/* there is nothing to be written onto the disk */
		if (ds->nextDiskXfer == ds->currentBusXfer) 
		  {
		      ds->needMove = TRUE; /* not yet */
		      INDEX_TIME_EVENT(EV_DISKSTATE, disk, 
				       DISKSTATE_IDLE, eventTime);
		  } 
		else
		  {
		      ds->nextDiskXfer++;
		      abstime = DSECTORXFER_t(eventTime);
		      ds->prefetchReq = DDEventRequest(ENDDISKXFER, 
						       abstime, disk);
		      INDEX_TIME_EVENT(EV_DISKSTATE, disk,
				       DISKSTATE_XFER, eventTime);
		  }
	    }
      }
}

/* @SUBTITLE "EndDiskXfer: end of one sector; start another" */
/*  
 *  Just finished transferring nextSector to/from queue. So, consider
 *  starting another sector xfer, and possibly a bus transfer.
 *
 * !! EndDiskXfer() should only advance ds->currentDiskXfer.  It should
 * !! by no means change ds->nextDiskXfer.
 */

PRIVATE void
EndDiskXfer(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]);
  
    DEBUG_TRACE(("EndDiskXfer \t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));
  
    ds->prefetchReq = NULL;   /* EndDiskXfer has been pulled off the heap */

    if (ds->done) 
      {
	  INDEX_TIME_EVENT(EV_DISKSTATE, disk, DISKSTATE_IDLE, eventTime);
	  return;
      }
  
    if (ds->cancelPrefetch)
      {
	  ds->cancelPrefetch = FALSE;
	  ds->activeRead = FALSE;
	  DDEventRequest(DISKMOVE, eventTime, disk); /* move to new request */
      }
    else
      {
	  if (ds->reading)    /* reading */
	    {
		ds->currentDiskXfer++;
		if (!ds->activeBusXfer)  
		  {
		      DEBUG_TRACE(("\t"));
		      ConsiderBusXfer(disk, eventTime);
		  }

		/* start next sector */
		DDEventRequest(STARTDISKXFER, 
			       MoveNextTime_t(disk, ds->nextDiskXfer, 
					      eventTime), disk);
	    }
	  else		      /* writing */
	    {
		ds->currentDiskXfer++;
		/* when writes get extended, sometimes fCS gets behind */
		/* (see end of controller() for explanation) */
		ds->firstCachedSector = 
		  min(ds->currentDiskXfer, ds->firstSector);

		if (!ds->activeBusXfer)
		  {
		      DEBUG_TRACE(("\t"));
		      ConsiderBusXfer(disk, eventTime);
		  }
	  
		/* start disk transfer again if we are not done */
		if (ds->nextDiskXfer <= ds->lastSector)
		  DDEventRequest(STARTDISKXFER, 
				 MoveNextTime_t(disk, ds->nextDiskXfer, 
						eventTime), disk);
		else
		  {
		      INDEX_TIME_EVENT(EV_DISKSTATE, disk, DISKSTATE_IDLE, 
				       eventTime);
#ifdef IMMEDIATE_REPORT
		      if (ds->restartController != NEVER)
			DDEventRequest(CONTROLLER, 
				       max(ds->restartController, eventTime), 
				       disk);
		      else if (ds->wantSync)
			DDEventRequest(SENDDONEMESSAGE, eventTime, disk);	
#else
		      DDEventRequest(SENDDONEMESSAGE, eventTime, disk);
#endif IMMEDIATE_REPORT
		  }
	    }
      }
}


/* @SUBTITLE "ConsiderBusXfer: consider starting a bus xfer" */

PRIVATE void
ConsiderBusXfer(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]);
    ulong readySectors;	      /* sectors that are ready */
    boolean lastSectorReady = FALSE;
    boolean start = FALSE;
    TICS grabTime = 0;	      /* time it took to grab the bus */

    DEBUG_TRACE(("ConsiderBusXfer\t%d\t@%s: nextBusXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextBusXfer));
  
    if (eventTime >= ds->wantBus && !ds->activeBusXfer && ds->haveBuffer)
      { 
	  if (ds->reading)
	    {
		lastSectorReady = (ds->currentDiskXfer > ds->lastSector);
		readySectors = (ds->currentDiskXfer - ds->nextBusXfer);
	  
		start = ((readySectors >= READFENCE) || lastSectorReady || 
			 CACHE_FULL(disk));
	    }
	  else
	    {
		lastSectorReady = (ds->nextBusXfer > ds->lastSector);
		readySectors = (ds->currentBusXfer - ds->nextDiskXfer);
	  
		start = ((readySectors <= WRITEFENCE) && !lastSectorReady &&
			 !CACHE_FULL(disk));
	    }
      
	  if (start)
	    {
		ds->activeBusXfer = TRUE; /* somehow, we WILL start it */

		if (GrabBus(disk, &grabTime, eventTime))
		  {
		      DDEventRequest(STARTBUSXFER, (eventTime + grabTime), 
				     disk);
		      if (grabTime > 0)
			INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_GRAB, 
					 eventTime);
		  }
		else
		  {
		      DEBUG_TRACE(("ConsiderBusXfer: waiting for bus\n"));
		      WaitForBus(disk, STARTBUSXFER, eventTime);
		  }
	    }
      }  
}

/* @SUBTITLE "StartBusXfer: start one-sector bus xfer" */
/*  
 *  Start a bus transfer at the given eventTime.
 *
 *  !! StartBusXfer() should only advance ds->nextBusXfer.  It should by
 *  !! no means change ds->currentBusXfer.
 */
PRIVATE void
StartBusXfer(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 
    TICS finish;

    INVARIANT4(disk == *(ds->busOwner),
	       "StartBusXfer: I (disk %d) do not own the bus, disk %d does\n",
	       disk, *(ds->busOwner));

    DEBUG_TRACE(("StartBusXfer \t%d\t@%s: nextBusXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextBusXfer));
  
    INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_DATA, eventTime);

    finish = BSECTORXFER_t(eventTime); /* find the finish time of the xfer */
    ds->nextBusXfer++;
    DDEventRequest(ENDBUSXFER, finish, disk);
}


/* @SUBTITLE "EndBusXfer: end one-sector bus xfer, start another" */
/*
 *  End a bus transfer at the given eventTime.
 *
 *  !! EndBusXfer() should only advance ds->currentBusXfer.  It should by
 *  !! no means change ds->nextBusXfer.
 */

PRIVATE void
EndBusXfer(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 
    boolean finished;
  
    DEBUG_TRACE(("EndBusXfer \t%d\t@%s: currentBusXfer = %lu\n", disk, 
		 time_print(eventTime), ds->currentBusXfer));
  
    ds->currentBusXfer++;
    if ((ds->nextBusXfer <= ds->lastSector) &&
	((ds->reading && ds->nextBusXfer < ds->currentDiskXfer)
	 || (!ds->reading && !CACHE_FULL(disk))))
      DDEventRequest(STARTBUSXFER, eventTime, disk);
    else
      {
	  finished = (ds->nextBusXfer > ds->lastSector);
      
	  ds->activeBusXfer = FALSE;
	  ds->wantBus = (finished ? NEVER : eventTime);
      
#ifdef IMMEDIATE_REPORT
	  if (finished)
	    DDEventRequest(SENDDONEMESSAGE, eventTime, disk);
#else
	  if (finished)
	    if (ds->reading)
	      DDEventRequest(SENDDONEMESSAGE, eventTime, disk);
	    else
	      /* give the bus to the next event on the bus queue */
	      PassBus(disk, eventTime);
#endif IMMEDIATE_REPORT
      }

    if (ds->needMove)
      {
	  ds->needMove = FALSE;	/* reset the value */
	  /* move to appropriate location */
	  DDEventRequest(DISKMOVE, eventTime, disk); 
      }
}

/* @SUBTITLE "SendDoneMessage: send a 'done' message to host" */

PRIVATE void
SendDoneMessage(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 
    TICS grabTime = 0;	      /* time it took to grab the bus */

    DEBUG_TRACE(("SendDoneMessage\t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));

    if (GrabBus(disk, &grabTime, eventTime))
      {
	  DDEventRequest(ENDDONEMESSAGE,
			 (eventTime + grabTime +DONETIME_t), disk);

	  if (grabTime > 0)
	    INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_GRAB, eventTime);
	  INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_COMMAND, 
			   eventTime + grabTime);
      }
    else
      {
	  DEBUG_TRACE(("SendDoneMessage:  Waiting for bus\n"));
	  WaitForBus(disk, SENDDONEMESSAGE, eventTime);
      }    
}

/* @SUBTITLE "EndDoneMessage: 'done' message reaches host; wakeup thread" */

PRIVATE void
EndDoneMessage(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 

    DEBUG_TRACE(("EndDoneMessage \t%d\t@%s: nextDiskXfer = %lu\n", disk, 
		 time_print(eventTime), ds->nextDiskXfer));

    /* give the bus to the next event on the bus queue */
    PassBus(disk, eventTime);
    DDEventWakeup(ds->wait_thread, eventTime);

    /* reset the sync flag */
    if (ds->wantSync) {
       ds->wantSync = FALSE;
    } else {
        /* Added by Yiming Hu */
        DEBUG0 printf("Disk %d EndDoneMessage\n", disk);
        ScheduleEvent(disk, EVENT_DISKDONE, eventTime + 0);
    }

}

/* == Utility Functions ================================================= */

/* @SUBTITLE "MoveTime_t: return the time when the move is completed" */
/* Calculate and return the time taken to move from one disk 
 * sector to another.
 *
 * Given the destination sector of the disk and the starting time, 
 * calculate the time when the disk head motion is finished.
 */
TICS
MoveTime_t(int disk, ulong lsector, TICS eventTime)
{
    TICS abstime;	      /* amount of time needed */
    ulong dcylinders, dtracks;	       
    ulong oldtrack = dstatus[disk].track;  
    ulong sector = Logical2Physical(lsector);
    ulong track = SECTOR2TRACK(sector);
    TICS oldtime;
  
    /* calculate the differences in cylinders */
    dcylinders = UlongAbsDiff(TRACK2CYLINDER(oldtrack), 
			      TRACK2CYLINDER(track));
  
    dtracks = UlongAbsDiff(oldtrack, track);
    oldtime = eventTime;
    abstime = SeekTime_t(dcylinders, dtracks, eventTime);

    /* ----------- seek time is calculated, next rotational delay ---------- */
  
    /* include the rotation delay */
    oldtime = abstime;
    abstime = RotDelay_t(disk, sector, abstime); 
  
    dstatus[disk].track = track; /* remember the new position */
    return(abstime);
}



/* @SUBTITLE "MoveNextTime_t: time needed to move to next logical sector" */
/*  
 *  Enters with:  Disk number, Logical sector number to move the disk
 *                head to and Time the move begins.
 *  Returns:      Time the move completes.
 */
PRIVATE TICS
MoveNextTime_t(int disk, ulong lsector, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]); 
    ulong trackswitch, headswitch;
    ulong oldtrack = SECTOR2TRACK(Logical2Physical(lsector - 1));
    ulong newtrack = SECTOR2TRACK(Logical2Physical(lsector));

    trackswitch = UlongAbsDiff(TRACK2CYLINDER(oldtrack), 
			       TRACK2CYLINDER(newtrack));
    headswitch = UlongAbsDiff(oldtrack, newtrack);

    /* more than one cylinder, eg when crossing a sparing region */
    if (trackswitch > 1)		
      {
	  /* crossing a sparing region */
	  return(MoveTime_t(disk, lsector, eventTime));
      }
    else
      {
	  ds->track = newtrack;

	  if (trackswitch == 1)	/* takes precedence over headswitch */
	    return(eventTime + TRACK_SWITCH_t);

	  /* only 1 headswitch within a cylinder counts */
	  else if (headswitch >= 1)	
	    return(eventTime + HEAD_SWITCH_t);
	  else
	    return(eventTime); /* no switch occurred */
      }
}

/* @SUBTITLE "RotDelay_t: time to rotate to the desired sector" */
/*
 * Enters with: Number of the disk of significance and the seektime.
 *              Parameter requested is the physical sector number.
 * Returns:     The time the sector requested is reached.
 */
PRIVATE TICS
RotDelay_t(int disk, ulong sector, TICS endSeekTime)
{
    double initRotPos = 0;    /* initial position is 0 at time 0 */
    double dtime;
    double endSeekRotPos;     /* rotational position after the seek */
    double destPosition, dPosition;
    ulong tracksector;

    endSeekRotPos = initRotPos + (double)(endSeekTime - 0) / ROT_TIME_t; 

    /* find the position in fractions */
    endSeekRotPos = endSeekRotPos - (unsigned long) endSeekRotPos; 

    tracksector = sector % NSECTORS;
    destPosition = ((double) (tracksector +
			      SECTOR2CYLINDER(sector) * CYLINDERSKEW +
			      (SECTOR2TRACK(sector) - SECTOR2CYLINDER(sector))
			      * TRACKSKEW) / NSECTORS);

    dPosition = 1 + destPosition - endSeekRotPos;
    dPosition = dPosition - (unsigned long) dPosition;
    dtime = dPosition *  ROT_TIME_t;

    return ((TICS)dtime + endSeekTime);
}

/* @SUBTITLE "SanityCheck: check parameters for errors" */
/*  
 *  This function checks whether the parameters given to the function
 *  is valid.  If they are not, the program will be aborted and an error
 *  message printed.
 */
PRIVATE void
SanityCheck(int disk, ulong lsector, ulong nsectors)
{

    INVARIANT3((disk >= 0 && disk < NO_OF_DISKS),
	       "SanityCheck %d: invalid disk number\n",
	       disk);
    INVARIANT3(dstatus[disk].startTime == NEVER,
	       "SanityCheck %d: previous DiskDeviceStart not finished!\n",
	       disk);
    INVARIANT3((lsector < TOTAL_NSECTORS),
	       "SanityCheck %d: starting sector not in disk\n",
	       disk);
    INVARIANT3((nsectors > 0),
	       "SanityCheck %d: transfer size too small\n",
	       disk);
    INVARIANT3(((lsector + nsectors - 1) < TOTAL_NSECTORS),
	       "SanityCheck %d: transfer past end of disk\n",
	       disk);

    /* The real disk may not be limited to this, but my implementation is.
     * That's ok, because I never make big requests like this anyway.
     */
    INVARIANT4((nsectors <= CACHE_SLOTS),
	       "SanityCheck %d: transfer size (%lu sectors) too large\n",
	       disk, nsectors);

    INVARIANT3((dstatus[disk].wantBus == NEVER),
	       "%d: previous request not finished\n", disk);
    INVARIANT3((!dstatus[disk].activeBusXfer), "%d: bus is still busy\n", disk);
}

/* @SUBTITLE "SeekTime_t: compute the seek-time component of disk move" */
/*  
 *  This function returns the time when a seek from one sector to the
 *  other will complete.
 *  The number returned is in TICS.
 */
PRIVATE TICS
SeekTime_t(ulong dcylinders, ulong dtracks, TICS eventTime)
{
    double dtime;
  
    /* as given in pg 14 (3.4.2) of Modelling Disks HPL-93-68 revision 1
       if (dcylinders < 616)
       dtime = (3.45 + 0.597 * sqrt((double)dcylinders));
       else
       dtime = (10.8 + 0.012 * dcylinders);
       */
  
    /* as given in pg 17, facts for HP 97560 */
    if (dcylinders == 0)
      {
	  if (dtracks != 0)
	    /* time for head switch instead of seek (convert 
	       HEAD_SWITCH_t back from TICS to MSEC) */
	    dtime = (double) HEAD_SWITCH_t / TICSperMSEC; /* just 1 switch */
	  else
	    dtime = 0;
      }
    else if (dcylinders < 383)
      dtime = (3.24 + 0.400 * sqrt((double)dcylinders));
    else
      dtime = (8.00 + 0.008 * dcylinders);
  
    /* convert dtime from msec to TICS */
    return ((TICS) (dtime * TICSperMSEC) + eventTime);
}

/* @SUBTITLE "Logical2Physical: convert logical sector number to physical" */
/*  Convert a logical sector number to a physical sector number. */

PRIVATE ulong
Logical2Physical(ulong lsector)
{
    ulong region, psector;
  
    for (region = 0; region < NUM_REGIONS; region++)
      if (lsector <= Regions[region].logicalend)
	break;		      /* find the region in which the lsector
				 is situated */

    /* calculate the corresponding physical sector number */
    psector = Regions[region].physicalstart + 
      (lsector - Regions[region].logicalstart);

    return (psector);
}

/* @SUBTITLE "GrabBus: try to grab the bus for ourselves" */
/*  Try to grab control of the bus. */

PRIVATE boolean		      /* TRUE iff we can use the bus now */
GrabBus(int disk, TICS *grabTime, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]);
    int *busOwner = ds->busOwner;
    
    if (*busOwner == BUS_FREE) { /* bus is available */
	*busOwner = disk;     /* grab it */
	*grabTime = BUSGRABTIME_t;
	INDEX_TIME_EVENT(EV_BUSCONTEND, ds->busId, 1, eventTime);
	return(TRUE);
    } else if (*busOwner == disk) { /* we already have the bus */
	*grabTime = 0;
	return(TRUE);	      /* ok */
    } else {
	return (FALSE);	      /* bus not available */
    }
}

/* @SUBTITLE "WaitForBus: wait for our bus to be free" */
/*
 *  Wait for the given bus, which is busy.
 */
PRIVATE void
WaitForBus(int disk, REQUESTCODE event, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]);
    NCQUEUE *busq = ds->busq;
    busItem *itemptr = (busItem *) malloc(sizeof(busItem));
    INVARIANT2((itemptr != NULL), "WaitForBus: malloc failed\n");
	
    itemptr->disk = disk;     /* record who we are and what we want */
    itemptr->event = event;
    Enqueue_noncyc(busq, itemptr); /* put us into the queue waiting for bus */

    INDEX_TIME_EVENT(EV_BUSCONTEND, ds->busId, 
		     InQueue_noncyc(busq)+1, eventTime);
}

/* @SUBTITLE "PassBus: pass our bus to a waiter, if any" */
/*
 *  Pass the given bus to another disk waiting to use that bus.
 *  'disk' should be the disk currently holding the bus.
 */
PRIVATE void
PassBus(int disk, TICS eventTime)
{
    struct diskstatus *ds = &(dstatus[disk]);
    TICS requestTime;

    /* is there anyone waiting for this bus? */
    if (!EmptyQueue_noncyc(ds->busq)) {
	busItem *itemptr;
    
	/* pull off the first disk waiting for this bus */
	Dequeue_noncyc(ds->busq, (void **)&itemptr);

	DEBUG_TRACE(("\tPass bus from %d to %d @%s\n", 
		     disk, itemptr->disk, time_print(eventTime)));

	INVARIANT4(dstatus[itemptr->disk].busOwner == ds->busOwner,
		   "PassBus: disks %d and %d don't share a bus!\n",
		   itemptr->disk, ds - dstatus);

	INVARIANT3(*(ds->busOwner) == disk,
		   "PassBus: disk %d doesn't own its bus!\n", disk);

	/* special-case when CONSIDERBUSXFER put STARTBUSXFER on queue */
	if (itemptr->event == STARTBUSXFER) {
	    INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_GRAB, eventTime);
	    /* it actually starts a little later */
	    requestTime = eventTime + BUSGRABTIME_t;
	} else
	  requestTime = eventTime;

	/* Give the bus to them right now */
	*(ds->busOwner) = itemptr->disk;

	/* and schedule the event they wanted */
	DDEventRequest(itemptr->event, requestTime, itemptr->disk);

	INDEX_TIME_EVENT(EV_BUSCONTEND, ds->busId, 
			 InQueue_noncyc(ds->busq)+1, eventTime);

	free(itemptr);
    } else {
	*(ds->busOwner) = BUS_FREE;
	INDEX_TIME_EVENT(EV_BUSSTATE, ds->busId, BUSSTATE_IDLE, eventTime);
	INDEX_TIME_EVENT(EV_BUSCONTEND, ds->busId, 0, eventTime);
    }
}
