/* 
 *  modularize.c
 *
 *  This file is supposed to include all the typedefs, macros
 *  and function headers that allows diskdevice.c to compile
 *  independently of proteus.
 *
 * $Id: modularize.c,v 2.0 94/07/17 18:34:55 dfk Exp $
 */


#include "modularize.h"
#include "heap.h"
#include "diskevent.h"


/* =======================The Timing Stuff ================================ */
/* keep the time struct within this file, i.e. modularize.c */
static struct timestruct system_timer;


/* returns the current time */
TICS GetTime(void)
{
  return (system_timer.currenttime);
}


/* returns the greater of timea and timeb;
   if (timea == timeb) return timeb */
TICS max(TICS timea, TICS timeb)
{
  return ((timea > timeb) ? timea : timeb);
}


/* returns the lesser of timea and timeb;
   if (timea == timeb) return timeb */
TICS min(TICS timea, TICS timeb)
{
  return ((timea < timeb) ? timea : timeb);
}


/* wait till system time is TICS, process the
   events if they happen before TICS */
void WaitTime(TICS donetime)
{
  extern Heap heap;
  HeapKey Hpkey;
  HeapData Hpdat;
  /*  FuncPtr handler; */
  void (*handler)(int, TICS);  /* By Yiming */
  
  while (TopHeap(heap, &Hpdat, &Hpkey) != FALSE && (Hpkey < donetime))
    {
/*
      INVARIANT2((RemHeap(heap, &Hpdat, &Hpkey)),"WaitTime: heap empty!\n");
      system_timer.currenttime = Hpkey;
      handler = DDhandlers[Hpdat->req_code].handler;
      
      if (Hpdat->req_code != NOP)
	(*handler) (Hpdat->disk, Hpkey);
      free(Hpdat);*/		/* free up space allocated during AddHeap() */

      ProcessAllEvents();   /* Added by Yiming Hu */
    }
  system_timer.currenttime = donetime;
}



void WaitDuration(TICS duration)
{
  WaitTime(system_timer.currenttime + duration);
}



TICS InitTime(void)
{
  system_timer.currenttime = 0;
  DDEventInit();		/* initialize the event heap */
  return(system_timer.currenttime);
}




void ProcessDiskEvent(HeapData Hpdat, HeapKey Hpkey)
{
  extern Heap heap;
  void (*handler)(int, TICS);  /* By Yiming */

  DEBUG0 printf("->Get one event %s \t for disk  %d \t time = %s code = %d\n",
                DDhandlers[Hpdat->req_code].name,
                Hpdat->disk,
                time_print(Hpkey), 
                Hpdat->req_code);
  
  handler = DDhandlers[Hpdat->req_code].handler;
  if (Hpdat->req_code != NOP)
     (*handler) (Hpdat->disk, Hpkey);
  else
     {DEBUG0 printf("ProcessDiskEvent: removing NOP\n");}
}




/*------------------------------------------------------
 *
 *    User Event Process
 *
 *------------------------------------------------------
 */





boolean DefaultUserEventProcedure(int disk, int EventCode, TICS now)
{
    if (EventCode == EVENT_QUIT)
       return FALSE;
    else
       return TRUE;
}


/* User Event Procedure Pointer */
static tEventProcedure UserEventProcedure = DefaultUserEventProcedure;


/*  ScheduleEvent
 *
 *  Put an event request into the event heap.
 */
void ScheduleEvent(int disk, int event, TICS eventTime)
{
  extern Heap heap;              /* defined in modularize.h, initialized in
                                    InitTime() because WaitTime() needs it. */
  heapdata *hpdat;

  INVARIANT2((eventTime < NEVER), "User Event: occuring at NEVER\n");

  hpdat = (heapdata *) malloc (sizeof(heapdata));
  if (hpdat == NULL)
    {
      fprintf(stderr, "ScheduleEvent: malloc failed\n");
      exit(-1);
    }

  hpdat->req_code  = (REQUESTCODE) event;
  hpdat->eventTime = eventTime;
  hpdat->disk = disk;
  AddHeap(heap, hpdat, (hpdat->eventTime));
}



tEventProcedure RegisterUserEventProcedure(tEventProcedure UserProcedure)
{
  INVARIANT2((UserProcedure != NULL), "UserEventProcedure is NULL\n");  
  UserEventProcedure = UserProcedure;

  return DefaultUserEventProcedure;
}



/* Return FALSE if get END_EVENT */
boolean ProcessAllEvents(void)
{
  extern   Heap heap;
  HeapKey  Hpkey;
  HeapData Hpdat;
  int      EventCode;
  int      ReturnCode;

  if (GetHeapNum(heap) == 0) {
     return TRUE;
  }

  if (RemHeap(heap, &Hpdat, &Hpkey) != FALSE) {
     EventCode = Hpdat->req_code;
     system_timer.currenttime = Hpkey;
     if (EventCode >= USER_EVENT) {
         ReturnCode =  (*UserEventProcedure)(Hpdat->disk, EventCode, Hpkey);
         free(Hpdat);            /* free up space allocated during AddHeap() */
         return ReturnCode;
     } else {
         ProcessDiskEvent(Hpdat, Hpkey);

         /* actually the event has already been processed. We call the user event
          * procedure anyway to give the user handle a chance to run. It may
          * want to insert some new envets or check the system idel time.
          */ 
         (*UserEventProcedure)(Hpdat->disk, EventCode, Hpkey); 
         free(Hpdat);
         return TRUE;
     }
  }
  
  return TRUE;    
}


/* void thread_sleep(int tid) */
void thread_suspend(int tid)
{
  extern int wakeup;            /* declared in modularize.h */
  extern Heap heap;
  HeapKey Hpkey;
  HeapData Hpdat;
/*  FuncPtr handler; */
  void (*handler)(int, TICS);  /* By Yiming */

  wakeup = -1;
  do
    {
      INVARIANT2((RemHeap(heap, &Hpdat, &Hpkey) != FALSE),
                 "thread_suspend: can't remove from empty heap\n");
/*
      DEBUG0   printf("--->Get one event %s from %d \t time = %s code = %d\n",
                 DDhandlers[Hpdat->req_code].name,
                 heap->numheap,
                 time_print(Hpkey),
                 Hpdat->req_code);
*/
      system_timer.currenttime = Hpkey;
      handler = DDhandlers[Hpdat->req_code].handler;
      if (Hpdat->req_code != NOP)
        (*handler) (Hpdat->disk, Hpkey);
      else
        {DEBUG0 printf("thread_suspend: removing NOP\n");}
      free(Hpdat);              /* free up space allocated during AddHeap() */
    }
  while (wakeup != tid);
}





/* ======================================================================== */
/*  time_print
 *
 *  Function to return the system time in a string.
 */
char* time_print(TICS t) {
  /* List of static buffers */
  static char buffers[100][21]; /* 20 digits holds a 64-bit number */
  static int next_buffer = 0;
  char* buffer;
  
  buffer = buffers[next_buffer];
  next_buffer = (next_buffer + 1) % 100;
  
#ifdef LONG_LONG_TIME
  /* Print out seven digits at a time */
  {
    unsigned long part1, part2, part3;
    
    
    part3 = (unsigned long) (t % 10000000);
    t = t / 10000000;
    
    part2 = (unsigned long) (t % 10000000);
    t = t / 10000000;
    
    part1 = (unsigned long) t;
    
    if (part1 > 0) {
      sprintf(buffer, "%lu%07lu%07lu", part1, part2, part3);
    }
    else if (part2 > 0) {
      sprintf(buffer, "%lu%07lu", part2, part3);
    }
    else {
      sprintf(buffer, "%lu", part3);
    }
  }
#else
  sprintf(buffer, "%lu", t);
#endif
  
  return buffer;
}


/* timep_print(Time *t)
 *    gdb can't pass unsigned long longs correctly, but it can pass 
 * *pointers* correctly.  So this is a front end to time_print()
 * for debugging.
 */
char *timep_print(TICS *t) {  return(time_print(*t)); }
