/*****************************************************************************
 *                                                                           *
 *      File Name:        tracedata.C                                        *
 *                                                                           *
 *      Function:         Defines functions to parse the trace data          *
 *                        and fill the request packet.                       *
 *                                                                           *
 *      Author:           Yiming Hu                                          *
 *                        hu@ele.uri.edu                                     *
 *                        Department of Electrical & Computer Engineering    *
 *                        University of Rhode Island                         *
 *                        Kingston, RI 02881                                 *
 *                                                                           *
 *****************************************************************************/

//Do not change next line. It is for SCCS version control
//SCCS  @(#)tracedata.c	V2.2    08/16/96  (in DCD Simulator) (C) Yiming Hu


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "general.H"
#include "tracedata.H"


static const int BUFFERSIZE = 200;


static FILE      *inputfile;
static char      sInputBuffer[BUFFERSIZE];
static unsigned  nLineCount;
static unsigned  long RequestID = 0;


//--------------------------------------------------------
// Private routine to display error information
//--------------------------------------------------------
static void FormatError(char *info)
{
    fprintf(stderr, "%s in trace data file, line %d:\n %s \n",
            info, nLineCount, sInputBuffer);
    exit(FATAL);
}



//--------------------------------------------------------
// Just open the file
//--------------------------------------------------------
void OpenTraceData(char *filename)
{
    if ((inputfile = fopen(filename,"r")) == NULL) {
       perror(filename);
       exit(FATAL);
    }
}


//--------------------------------------------------------
// Parse the user trace data and put the data into 
// the Request package.
// Return FALSE at EOF
//--------------------------------------------------------
BOOLEAN bGetTraceData(tRequestPacket *Request)
{
    int n,i;
    float t;
    char c[BUFFERSIZE];

    do {
       if (NULL == fgets(sInputBuffer, BUFFERSIZE, inputfile) )
          return FALSE;    /* End of trace data */

       nLineCount++;
       // skip leading space
       for (i=0; isspace( sInputBuffer[i] ); i++);

       if (sInputBuffer[i] == '@') {
          // Echo debug information
          printf("%s", &sInputBuffer[i+1]);
       }

    } while ( (sInputBuffer[i] == '%') || (sInputBuffer[i] == 0) ||
              (sInputBuffer[i] == '@') || (sInputBuffer[i] == '\n'));
  
    n = sscanf( sInputBuffer,"%f%s%lu%lu%d",
                &t, &c, &(Request->DiskNo), &(Request->LBA), &(Request->SectorCount) );

    if (n != 5) {
       FormatError("INVALID input line");
    }

    Request->RequestTime = GetTime() + (TICS)(TICSperMSEC * t);

    c[0] = toupper(c[0]);
    switch (c[0]) {
        case '0':
        case 'R':
             Request->Command = cmdRead;
             break;
        case '1':
        case 'W':
             Request->Command = cmdWrite;
             break;
        case 'S':
             Request->Command = cmdSeek;
             break;
        case 'V':
             Request->Command = cmdVerify; 
             break;
        case 'D':
             Request->Command = cmdDelay;
             break;
        default :
             FormatError("Unknown Request command");
             break;
    }

    Request->RequestNo = RequestID++;

    Request->CallBack  = NULL;  
    return TRUE;
}



