/* @TITLE "diskdevice-trivial.c: Simulate a disk device"*/
/* 
 * diskdevice-trivial - a constant-time simulation of a disk drive
 *    This is handy for debugging other parts of the overall simulator.
 *    THIS FILE IS INCLUDED BY DISKDEVICE.C.
 *
 * functions:
 *   DiskDeviceInit
 *   DiskDeviceTransfer
 *   DiskDeviceStart
 *   DiskDeviceFinish
 *   DiskDeviceSync
 *   DiskDeviceShape
 *
 * David Kotz 1994
 */

/* $Id: diskdevice-trivial.c,v 2.0 94/07/17 18:34:24 dfk Exp $ */

#ifdef DFK
# include "dmcache.h"
# include "userdata.h"
# include "util.h"
# include "stats.h"
# include "disk.h"
# include "diskmodel.param"
#else
# include "modularize.h"
#endif

#include "diskevent.h"
#include "diskdevice.h"
#include "diskdevices.h"

struct dd_handlers DDhandlers[1]; /* dummy */

/* debugging */
/*
#define DEBUG_TRACE(x) {CYCLE_COUNTING_OFF; printf x; CYCLE_COUNTING_ON;}
 */
#define DEBUG_TRACE(x) {}

PRIVATE void SanityCheck(int disk, ulong sector, ulong nsectors);

/* When each disk started and ended the current transaction. */
PRIVATE int  BusID[NO_OF_DISKS];
PRIVATE FILE *datafiles[NO_OF_DISKS];
/* below are used by DiskDeviceStart/Finish  */
PRIVATE TICS StartTime[NO_OF_DISKS];
PRIVATE TICS FinishTime[NO_OF_DISKS];
PRIVATE ulong firstSector[NO_OF_DISKS];
PRIVATE ulong nsectors[NO_OF_DISKS];
PRIVATE boolean reading[NO_OF_DISKS];

/* @SUBTITLE "DiskDeviceInit: initialize all disks" */
/* The caller must provide a pointer to a boolean, which  
 * we use to tell when that bus is free.  We also need a QUEUE, with room 
 * for as many disks as are attached to that bus, where disks can wait
 * for service on the bus.   Presumably, several disks share
 * the same values; thus, this is how we handle bus contention.
 * It must not be deallocated until after DiskDeviceDone is called.
 * The busId is used for debugging and tracing, nothing functional.
 */
void DiskDeviceInit(int disk, int busId, int *busOwner, NCQUEUE *busWaitq, 
		    char *diskFileName)
{
    DEBUG_TRACE(("DiskDeviceInit \t%d\n", disk));

    INVARIANT3(disk >= 0 && disk < NO_OF_DISKS,
	       "DiskDeviceInit: disk %d is invalid\n", disk);

    StartTime[disk] = 0;
    FinishTime[disk] = 0;
    BusID[disk] = busId;

    /* decide whether to use some Unix file as a pretend "disk" */
    if (diskFileName == NULL)
      datafiles[disk] = NULL; /* do not use any disk file; dummy data only */
    else {
	/* try to open the given file as a place for this "disk"s data */
	/* upon error in fopen(), datafiles[disk] will also be assigned NULL */
	datafiles[disk] = fopen(diskFileName, "r+");
	/* file does not exist, try to create the file */
	if (datafiles[disk] == NULL) {
	    datafiles[disk] = fopen(diskFileName, "w+");
	    /* can't even create it, fail it */
	    INVARIANT3(datafiles[disk] != NULL,
		       "DiskDeviceInit: disk file %s cannot be opened\n",
		       diskFileName);
	}
    }
}

/* @SUBTITLE "DiskDeviceDone: cleanup" */
void
DiskDeviceDone(int disk)
{
    DEBUG_TRACE(("DiskDeviceDone \t%d\n", disk));

    /* record the final idle period */
    if (FinishTime[disk] > 0) /* not at the beginning */
      SUM_ARRAY_METRIC(stats.diskidle, disk, 
		       (GetTime() - FinishTime[disk]) * MSECperTIC);
    INDEX_TIME_EVENT(EV_DISKIDLE, disk, 0, FinishTime[disk]);
    INDEX_TIME_EVENT(EV_BUSSTATE, BusID[disk],BUSSTATE_IDLE, FinishTime[disk]);
    INDEX_TIME_EVENT(EV_DISKSTATE, disk, DISKSTATE_IDLE, FinishTime[disk]);

    if (datafiles[disk] != NULL)
      fclose(datafiles[disk]);		/* close the datafile */
}



/* @SUBTITLE "DiskDeviceStart: start a disk transaction" */
/* Like DiskDeviceTransfer, but split into two parts: the first part
 * starts the disk moving to the appropriate location, and the second supplies
 * the buffer needed to complete the transaction. Must be followed by a 
 * corresponding Finish.
 */ 
void
DiskDeviceStart(int disk, ulong sector, ulong num_sectors, boolean write)
{    
    /* check whether the caller of this function is insane */  
    SanityCheck(disk, sector, num_sectors);
  
    /* record the preceding idle period */
    if (FinishTime[disk] > 0) /* not at the beginning */
      SUM_ARRAY_METRIC(stats.diskidle, disk, 
		       (GetTime() - FinishTime[disk]) * MSECperTIC);
    INDEX_TIME_EVENT(EV_DISKIDLE, disk, 0, FinishTime[disk]);
    INDEX_TIME_EVENT(EV_BUSSTATE, BusID[disk],BUSSTATE_IDLE, FinishTime[disk]);
    INDEX_TIME_EVENT(EV_DISKSTATE, disk, DISKSTATE_IDLE, FinishTime[disk]);
  
    /* record the event of reading or writing */
    INDEX_EVENT(write ? EV_DISKWRITE : EV_DISKREAD, disk, sector);
    SUM_ARRAY_METRIC(write ? stats.diskwrite : stats.diskread, disk, 1.0);
    INDEX_EVENT(EV_DISKBUSY, disk, 0);
  
    StartTime[disk] = GetTime();

    /* record the location so DiskDeviceFinish will know */
    firstSector[disk] = sector;
    nsectors[disk] = num_sectors;
    reading[disk] = !write;

    INDEX_TIME_EVENT(EV_DISKSTATE, disk, 
		     DISKSTATE_XFER, StartTime[disk]);
    INDEX_TIME_EVENT(EV_BUSSTATE, BusID[disk], BUSSTATE_DATA, StartTime[disk]);

    FinishTime[disk] = StartTime[disk] + 
      (write ? DISK_WRITE_TIME : DISK_READ_TIME) * TICSperMSEC;

    DEBUG_TRACE(("DiskDeviceStart %d sector %lu start %s finish %s\n", 
		 disk, sector, 
		 time_print(StartTime[disk]), 
		 time_print(FinishTime[disk])));
}


/* @SUBTITLE "DiskDeviceFinish: finish a disk transaction" */
/* See DiskDeviceStart.  This, the second call, 
 * BLOCKs until the transfer is complete.  There should not have been
 * any other requests since the corresponding Start.  Ideally, we should
 * not be late in calling DiskDeviceFinish, ie, not after the disk movement
 * has finished.
 */ 
void
DiskDeviceFinish(int disk, UserData buffer)
{    
    INVARIANT3(StartTime[disk] != 0,
	       "DiskDeviceFinish %d: no corresponding DiskDeviceStart!\n",
	       disk);

    DEBUG_TRACE(("DiskDeviceFinish %d start %s finish %s\n", 
		 disk, 
		 time_print(StartTime[disk]), 
		 time_print(FinishTime[disk])));

    WaitTime(FinishTime[disk]);

    /* actually read and write the data to vdisk */
    if (datafiles[disk] != NULL) {
	fseek(datafiles[disk], (long)(firstSector[disk] * SECTOR_SIZE), SEEK_SET);
	if (reading[disk])
	  fread(buffer, (size_t)SECTOR_SIZE, nsectors[disk], datafiles[disk]);
	else {
	    INVARIANT2(fwrite(buffer, (size_t)SECTOR_SIZE, nsectors[disk],
			      datafiles[disk]) == nsectors[disk],
		       "DiskDeviceFinish: error while writing to data file\n");
	}
    }

    AVG_ARRAY_METRIC(stats.diskwait, disk, 
                     (FinishTime[disk] - StartTime[disk]) * MSECperTIC);
    STDDEV_ARRAY_METRIC(stats.diskwait_sd, stats.diskwait, disk, 
                        (FinishTime[disk] - StartTime[disk]) * MSECperTIC);

    StartTime[disk] = 0;      /* reset the disk */
}

/* @SUBTITLE "DiskDeviceTransfer: a complete disk transaction" */
/* Do a disk operation. We give it everything it needs to know: the disk
 * number, sector number on that disk, length in sectors,
 * a read/write flag, and the buffer for the data.   
 * This BLOCKS until the time when the transfer would be complete.
 */
void
DiskDeviceTransfer(int disk, ulong sector, ulong nsectors,
		   boolean write, UserData buffer)
{
    SanityCheck(disk, sector, nsectors);

    /* record the preceding idle period */
    if (FinishTime[disk] > 0) /* not at the beginning */
      SUM_ARRAY_METRIC(stats.diskidle, disk, 
		       (GetTime() - FinishTime[disk]) * MSECperTIC);
    INDEX_TIME_EVENT(EV_DISKIDLE, disk, 0, FinishTime[disk]);
    INDEX_TIME_EVENT(EV_BUSSTATE, BusID[disk],BUSSTATE_IDLE, FinishTime[disk]);
    INDEX_TIME_EVENT(EV_DISKSTATE, disk, DISKSTATE_IDLE, FinishTime[disk]);
    
    /* record the event of reading or writing */
    INDEX_EVENT(write ? EV_DISKWRITE : EV_DISKREAD, disk, sector);
    SUM_ARRAY_METRIC(write ? stats.diskwrite : stats.diskread, disk, 1.0);
    INDEX_EVENT(EV_DISKBUSY, disk, 0);

    StartTime[disk] = GetTime(); /* start the disk */
    INDEX_TIME_EVENT(EV_DISKSTATE, disk, 
		     DISKSTATE_XFER, StartTime[disk]);
    INDEX_TIME_EVENT(EV_BUSSTATE, BusID[disk], BUSSTATE_DATA, StartTime[disk]);

    /* compute the finish time, and wait until that time */

    FinishTime[disk] = GetTime() + 
      (write ? DISK_WRITE_TIME : DISK_READ_TIME) * TICSperMSEC;

    DEBUG_TRACE(("DiskDeviceTransfer %d sector %lu start %s finish %s\n", 
		 disk, sector, 
		 time_print(StartTime[disk]), 
		 time_print(FinishTime[disk])));

    WaitTime(FinishTime[disk]);

    /* actually read and write the data to vdisk */
    if (datafiles[disk] != NULL) {
	fseek(datafiles[disk], ((long) sector * SECTOR_SIZE), SEEK_SET);
	if (!write)
	  fread(buffer, (size_t)SECTOR_SIZE, nsectors, datafiles[disk]);
	else
	  INVARIANT4(fwrite(buffer, (size_t)SECTOR_SIZE, nsectors, 
			    datafiles[disk]) == nsectors,
		     "DiskDeviceTransfer: error while writing %lu sectors at sector %lu to data file\n", nsectors, sector);
    }
    
    AVG_ARRAY_METRIC(stats.diskwait, disk, 
                     (FinishTime[disk] - StartTime[disk]) * MSECperTIC);
    STDDEV_ARRAY_METRIC(stats.diskwait_sd, stats.diskwait, disk, 
                        (FinishTime[disk] - StartTime[disk]) * MSECperTIC);

    StartTime[disk] = 0;      /* reset the disk */
}


/* @SUBTITLE "DiskDeviceSync: Synchronizing the disk." */
/* All the data in the disk cache is written to the disk before returning.
 */
void 
DiskDeviceSync(int disk)
{
    DEBUG_TRACE(("DiskDeviceSync \t%d\n", disk));

    INVARIANT3(StartTime[disk] == 0,
	       "DiskDeviceSync %d: previous DiskDeviceStart not finished!\n",
	       disk);
}

/* ======================================================================= */
/* @SUBTITLE "DiskDeviceShape: return the shape of the disk" */
/* Return the shape of the disk. */
void 
DiskDeviceShape(int disk, ulong *nSectors, ulong *sectorSize,
		ulong *nTracks, ulong *sectorsPerTrack)
{
    *nSectors = TOTAL_NSECTORS;
    *sectorSize = SECTOR_SIZE;
    *nTracks = NTRACKS;
    *sectorsPerTrack = NSECTORS;
}


/* @SUBTITLE "SanityCheck: check parameters for sanity" */
/*  SanityCheck
 *
 *  This function checks whether the parameters given to the function
 *  is valid.  If they are not, the program will be aborted and an error
 *  message printed.
 */
PRIVATE void SanityCheck(int disk, ulong sector, ulong nsectors)
{

    INVARIANT4((disk >= 0 && disk < NO_OF_DISKS),
	       "SanityCheck disk %d: disk %d does not exist\n",
	       disk, disk);
    INVARIANT3(StartTime[disk] == 0,
	       "SanityCheck disk %d: previous DiskDeviceStart not finished!\n",
	       disk);
    INVARIANT4((nsectors > 0),
	       "SanityCheck disk %d: transfer size (%lu sectors) too small\n",
	       disk, nsectors);

    INVARIANT4((sector < TOTAL_NSECTORS),
	       "SanityCheck disk %d: starting sector %lu not in disk\n",
	       disk, sector);
    INVARIANT4(((sector + nsectors - 1 ) < TOTAL_NSECTORS),
	       "SanityCheck disk %d: transfer past end of disk to sector %lu\n",
	       disk, (sector + nsectors - 1));

}
