/*****************************************************************************
 *                                                                           *
 *      File Name:        diskmodel.C                                        *
 *                                                                           *
 *                        Defines a C++ wrapper for the disk simulator.      *
 *                                                                           *
 *                        hu@ele.uri.edu                                     *
 *                        Department of Electrical & Computer Engineering    *
 *                        University of Rhode Island                         *
 *                        Kingston, RI 02881                                 *
 *                                                                           *
 *****************************************************************************/

//Do not change the next line. It is for SCCS version control
//SCCS  @(#)diskqueue.c V3.2    02/07/97  (in DCD Simulator) (C) Yiming Hu


#include <iostream.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>

#include "diskmodel.H"


/************************************************************************
 *                                                                      *
 *              Global Variable                                         *
 *                                                                      *
 ************************************************************************/


Heap heap;                      // The event heap. We define it
                                // here so we don't have to define
                                // it in the application code


/************************************************************************
 *                                                                      *
 *              Member Functions for tUserEventHandler              *
 *                                                                      *
 ************************************************************************/


tUserEventHandler *tUserEventHandler::HandlerPointer = NULL;


//--------------------------------------------------------
//--------------------------------------------------------
void
tUserEventHandler::init(void)
{
    if (HandlerPointer != NULL) {
       cerr << "Only one user event handler is allowed!\n";
       exit (FATAL);
    }

    HandlerPointer = this;

    DefaultHandler = RegisterUserEventProcedure(theHandler);
}



//--------------------------------------------------------
//--------------------------------------------------------
boolean
tUserEventHandler::theHandler(int disk, int EventCode, TICS now)
{
    return HandlerPointer->Handler(disk, EventCode, now);
}



//--------------------------------------------------------
//--------------------------------------------------------
boolean 
tUserEventHandler::Handler(int disk, int EventCode, TICS now)
{
  tDisk *DiskPointer;
  switch (EventCode) {
      case EVENT_DISKDONE:
        DiskPointer = (tDisk*)GetDiskPointer(disk);
        if (DiskPointer)
           DiskPointer->endRequest();
        else {
           cerr << "NULL Disk Pointer\n";
           exit(FATAL);
        }
        return FALSE;
      default:
        return (*DefaultHandler)(disk, EventCode, now);
  }
}



//--------------------------------------------------------
//--------------------------------------------------------
boolean 
tUserEventHandler::topHeap(HeapData *data, TICS *Time)
{
    return ::TopHeap(heap, data, Time);
}




//--------------------------------------------------------
//--------------------------------------------------------
int
tUserEventHandler::getHeapNum()
{
    return ::GetHeapNum(heap);
}



/************************************************************************
 *                                                                      *
 *              Member Functions for tAbstractDisk                          *
 *                                                                      *
 ************************************************************************/


static const int MAX_BUS_NUMBER = 10;
static const int BQSIZE         = 15;

struct tBusPool 
{
  int      Used;
  int      BusOwner;
  NCQUEUE  *BusWaitq;
  char     BusName[20];

  tBusPool(void) {Used = -1; BusOwner = BUS_FREE; BusWaitq = NULL;};
  ~tBusPool(void) {if (BusWaitq) free(BusWaitq);};
};



// Static member must be defined explicitly.
int tDisk::NextDiskID = 1;


//--------------------------------------------------------
//--------------------------------------------------------
tDisk::tDisk(char *Name)
    : tAbstractDisk(Name)
{
    DiskID = NextDiskID++;

    TotalReadNumber  = 0;
    TotalWriteNumber = 0;
}



//--------------------------------------------------------
//--------------------------------------------------------
tDisk::tDisk(char *Name, int busId, char *diskFileName)
    : tAbstractDisk(Name)
{
    // Note: I put the BusPool as a static array instead of a global
    // because I want it be initialized before being called by the
    // constructor.
    static tBusPool BusPool[MAX_BUS_NUMBER];

    DiskID = NextDiskID++;

    if (busId >= MAX_BUS_NUMBER) {
       cerr << "tAbstractDisk: busId number too large\n";
       exit(FATAL);
    }

    if (BusPool[busId].Used == -1) {
       sprintf(BusPool[busId].BusName, "bus queue%d", busId);
       BusPool[busId].BusWaitq = MakeQueue_noncyc(BQSIZE, BusPool[busId].BusName);

       if (BusPool[busId].BusWaitq == NULL) {
          cerr << "tAbstractDisk: can not allocate memory for bus queue\n";
          exit(FATAL);
       }

       BusPool[busId].Used = 1;
    }

    init(busId, &BusPool[busId].BusOwner, BusPool[busId].BusWaitq, diskFileName);
}



//--------------------------------------------------------
//--------------------------------------------------------
tDisk::tDisk(char *Name, int busId, int *busOwner,
                     NCQUEUE *busWaitq, char *diskFileName)
    : tAbstractDisk(Name)
{
    DiskID = NextDiskID++;
    init(busId, busOwner, busWaitq, diskFileName);
}



//--------------------------------------------------------
//--------------------------------------------------------
void 
tDisk::init(int busId, int *busOwner, NCQUEUE *busWaitq, char *diskFileName)
{
    DiskDeviceInit(DiskID, busId, busOwner, busWaitq, diskFileName);

    TotalReadNumber  = 0;
    TotalWriteNumber = 0;
}


//--------------------------------------------------------
//     Get disk request package. Setup parameters, put disk
//     in seek state and start seek.
//--------------------------------------------------------
void 
tDisk::sendRequest(tRequestPacket *Request)
{
    int    write;

    CurrentRequest = *Request;

    if ( Request->Command == cmdWrite ) {
       TotalWriteNumber++;
    } else if ( Request->Command == cmdRead ) {
       TotalReadNumber++;
    }

    write = (Request->Command == cmdWrite) ? 1 : 0;

    DiskDeviceTransfer(DiskID, Request->LBA, Request->SectorCount,
                       write, (char*)Request->Data, (void *)this);

}



//--------------------------------------------------------
//--------------------------------------------------------
void 
tDisk::endRequest(void)
{
    tRequestPacket DiskReturnData;

    DiskDeviceTransferEnd(DiskID);


    if (NULL != CurrentRequest.CallBack) { 

        DiskReturnData = CurrentRequest;

        // Get disk access start and stop time
        GetDiskTime(DiskID, &DiskReturnData.StartTime, &DiskReturnData.FinishTime);
        
        (*(CurrentRequest.CallBack))( DiskReturnData );
    }
}



//--------------------------------------------------------
//--------------------------------------------------------
BOOLEAN 
tDisk::bIsBusy(void)
{
    return DiskIsBusy(DiskID);
}



//--------------------------------------------------------
//--------------------------------------------------------
void 
tDisk::done()
{
    while (DiskIsBusy(DiskID)) 
        ProcessAllEvents();
    DiskDeviceDone(DiskID);
}



//--------------------------------------------------------
//--------------------------------------------------------
void 
tDisk::sync()
{
    // Wait for previous request finished
    while (DiskIsBusy(DiskID))
        ProcessAllEvents();

    DiskDeviceSync(DiskID);

   // Wait for sync finished
   while (DiskWaitingSync(DiskID))
        ProcessAllEvents();

}




//--------------------------------------------------------
//--------------------------------------------------------
void 
tDisk::deviceShape(ulong *nSectors, ulong *sectorSize,
                   ulong *nTracks, ulong *sectorsPerTrack) const
{
    DiskDeviceShape(DiskID, nSectors, sectorSize, nTracks, sectorsPerTrack);
}



//--------------------------------------------------------
//--------------------------------------------------------
void
tDisk::getAccessNumber(int& TotalRead, int& TotalWrite) const
{
    TotalRead  = TotalReadNumber;
    TotalWrite = TotalWriteNumber;
}





/************************************************************************
 *                                                                      *
 *              Member Functions for tQueuedDisk                     *
 *                                                                      *
 ************************************************************************/


//--------------------------------------------------------
//--------------------------------------------------------
tQueuedDisk::tQueuedDisk(char *Name)
   : tDisk(Name)
{
}



//--------------------------------------------------------
//--------------------------------------------------------
tQueuedDisk::tQueuedDisk(char *Name, int busId, char *diskFileName)
   : tDisk(Name, busId, diskFileName)
{
}



//--------------------------------------------------------
//--------------------------------------------------------
tQueuedDisk::tQueuedDisk(char *Name, int busId, int *busOwner,
                         NCQUEUE *busWaitq, char *diskFileName)
   : tDisk(Name, busId, busOwner, busWaitq, diskFileName)
{
}



//--------------------------------------------------------
//--------------------------------------------------------
void
tQueuedDisk::sendRequest(tRequestPacket *Request)
{
    if (bIsBusy() == NO) {
       tDisk::sendRequest(Request);
    } else  {
       Queue.In(Request);
    }
}




//--------------------------------------------------------
//--------------------------------------------------------
void
tQueuedDisk::endRequest(void)
{
    tDisk::endRequest();

    if (Queue.bIsEmpty() == NO) {

       tRequestPacket  Item;

       Queue.Out(&Item);
       tDisk::sendRequest( &Item );
    }
}


//--------------------------------------------------------
//--------------------------------------------------------
BOOLEAN
tQueuedDisk::bIsBusy(void)
{
    if (Queue.bIsEmpty() == NO) {
       return YES;
    } else {
       return tDisk::bIsBusy();
    }
}

