/************************************************************************* 

     This program is used to show how to simulate multiple disks.
     Here two disks are simulated. The program reads requests from the 
     trace file, sends them to the target disks at the specified time.
     The time intervals between requests are defined in the trace files.

     The disks have built-in queues (using tQueuedDisk class) so 
     they can accept requests while they are busying for previous requests.
     When a request finishes, the disk automatically checks the queue and 
     dequeues a request to the disk if the queue is not empty.

     To run the program, type
          test2disk input2.dat

     Author:           Yiming Hu                                         
                       hu@ele.uri.edu                                 
                       Department of Electrical & Computer Engineering
                       University of Rhode Island                    
                       Kingston, RI 02881                           

*************************************************************************/

//Do not change the next line. It is for SCCS version control
//SCCS  @(#)diskqueue.c V3.2    02/07/97  (in DCD Simulator) (C) Yiming Hu


#include <iostream.h>
#include <stream.h>
#include <iomanip.h>

#include "diskmodel.H"
#include "tracedata.H"
#include "general.H"


//--------------------------------------------------------
//  Constants 
//--------------------------------------------------------

const int    NUMBER_OF_DISK    = 2;
const int    EVENT_SENDREQUEST = 0x2000;

const double MSECperUSEC  =   ( 1.0 / 1000.0 );
const double TICSperUSEC  =   (TICSperMSEC * MSECperUSEC);


//--------------------------------------------------------
//  Global data
//--------------------------------------------------------

void SendNextRequest();

tQueuedDisk Disk[NUMBER_OF_DISK] = 
            { 
              tQueuedDisk( "Disk1", 0, NULL ),
              tQueuedDisk( "Disk2", 0, NULL )
            };

tRequestPacket Request;


//--------------------------------------------------------
//  User event handler
//--------------------------------------------------------
class tMyUserEventHandler : public tUserEventHandler
{
    virtual boolean Handler(int disk, int EventCode, TICS now)
    {
        // Important! You must call the base function to handle the events 
        boolean ReturnCode = tUserEventHandler::Handler( disk, EventCode, now );

        // Put your own code here to handle newly defined user events
        switch ( EventCode ) {
            case EVENT_SENDREQUEST:
                 SendNextRequest();
                 return FALSE;
            default:
                 break;
        }
        return ReturnCode;
    }

};


tMyUserEventHandler MyEventHandler;



//--------------------------------------------------------
//  main program
//--------------------------------------------------------
int 
main(int argc, char **argv)
{

    // can only take 1 parameter 
    if ( argc != 2 ) {
        cerr << "syntax: " << argv[0] << "testinput\n";
        exit(FATAL);
    }

    MyEventHandler.init();

    OpenTraceData( argv[1] );
    // get first request 
    if ( ! bGetTraceData( &Request ) ) {
       cerr << "Empty trace file\n";
       exit( 1 ); 
    }

    // Send the request to disk at the specific time
    ScheduleEvent(0, EVENT_SENDREQUEST, Request.RequestTime);

    // The event loop
    while ( 1 ) {
       ProcessAllEvents();
    }
}



// Callback function : when a request finishes, this function 
// will be called. We can report the timing information here.
void CallBack(struct tRequestPacket DiskReturnData)
{
    TICS StartTime = (TICS)(DiskReturnData.RequestTime/TICSperUSEC);
    TICS StopTime  = (TICS)(DiskReturnData.FinishTime/TICSperUSEC);

    cout << "Disk "     << DiskReturnData.DiskNo 
         << "  Req No. "  << setw(3) << DiskReturnData.RequestNo
         << "  LBA = "    << setw(7) << DiskReturnData.LBA
         << "  Start at " << setw(6)  << time_print( StartTime )
         << "  Finish at " << setw(6) << time_print( StopTime )
         << "  Response time = " << time_print( StopTime - StartTime )
         << endl;
}



void SendNextRequest()
{
  int i;

  // Send the request to disk
  //
  Request.CallBack = CallBack;
  Disk[ Request.DiskNo ].sendRequest( &Request );

  // Get next request
  //
  if ( ! bGetTraceData( &Request ) ) {

     /* Done with the trace, reset the info and shut it down */
     for (i = 0; i < NUMBER_OF_DISK; i++ ) {
         Disk[i].sync();  
     }

     for (i = 0; i < NUMBER_OF_DISK; i++ ) {
         Disk[i].done();  
     }

     exit( 0 );
  }

  if ( Request.DiskNo >= NUMBER_OF_DISK ) {
     cerr << " Disk number (" << Request.DiskNo << ") out of range\n";
     exit( 1 );
  }


  // Send next request at the specified time
  //
  ScheduleEvent( 0, EVENT_SENDREQUEST, Request.RequestTime );
}

