/* Agent Tcl
   Bob Gray
   25 January 1995

   tclAgent.h

   This file defines the agent library routines.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _TCL_AGENT_H
#define _TCL_AGENT_H

#include <sys/types.h>
#include <time.h>
#include "agentId.h"
#include "my_sizes.h"
#include "tcl.h"
#include "tclTcpip.h"

typedef int (Agent_AppInitProc) (Tcl_Interp *interp, int server);

#ifdef __cplusplus

class MASK_INT;
 
#define EXTERN extern "C"

  /* dummy fd's for messages, meetings and events */

const MEETING_FD = -1;
const MESSAGE_FD = -2;
const EVENT_FD   = -3;

  /* blocking and nonblocking */

const BLOCKING    = 0;
const NONBLOCKING = 1;

  /* item available and item unavailable */

const GET_NONE    = 0;
const GET_SUCCESS = 1;

  /* meetings */

const MEET_REQUEST = 0;
const MEET_ACCEPT  = 1;
const MEET_CONNECT = 2;
const MEET_REFUSE  = 3;

  /* a mask can accept no, some or all incoming messages, meetings or events */

const MASK_ACCEPT_NONE = 0;
const MASK_ACCEPT_SOME = 1;
const MASK_ACCEPT_ALL  = 2;

  /* a mask handle is 0 or greater so -1 is used to indicate ALL masks */

const ALL_MASKS	= -1;

  /* default number of seconds to wait when connecting */

const DEFAULT_CONNECT_SECONDS	= 20;

#else

#define EXTERN extern

  /* dummy fd's for meetings, messages and events */

#define MEETING_FD	-1
#define MESSAGE_FD 	-2
#define EVENT_FD	-3

  /* blocking and nonblocking */

#define BLOCKING	0
#define NONBLOCKING	1

  /* item available and item unavailable */

#define GET_NONE	0
#define GET_SUCCESS	1

  /* meetings */

#define MEET_REQUEST	0
#define MEET_ACCEPT	1
#define MEET_CONNECT	2
#define MEET_REFUSE	3

  /* a mask can accept no, some or all incoming messages, meetings or events */

#define	MASK_ACCEPT_NONE	0
#define	MASK_ACCEPT_SOME	1
#define MASK_ACCEPT_ALL		2

  /* a mask handle is 0 or greater so -1 is used to indicate ALL masks */

#define ALL_MASKS	-1

  /* default number of seconds to wait when connecting */

#define DEFAULT_CONNECT_SECONDS		20

#endif

#define ANY_TAG		NULL

struct MASK_ENTRY           /* external representation of a mask */
{
  struct AGENT_ID *id;      /* identification of source agent    */
  char *tag;                /* tag for EVENTS                    */
  char *handler;            /* name of the handler               */

#ifdef __cplusplus

  MASK_ENTRY (const MASK_ENTRY& mask);
  MASK_ENTRY (MASK_ENTRY *mask);
  MASK_ENTRY (void);
 ~MASK_ENTRY ();

#endif
};

struct MASK        
{
  int type;                   /* MASK_ACCEPT_{NONE,SOME,ALL}   */
  int handle;                 /* the mask handle               */
  int count;                  /* number of entries in the mask */
  struct MASK_ENTRY **masks;  /* the entries                   */
  char *handler;              /* default event handler         */

#ifdef __cplusplus

  MASK (int type, int handle, int count, char *handler);
  MASK (MASK_INT *mask);
 ~MASK ();

#endif
};
 
  /* locations */

EXTERN int is_agent_registered    
	(Tcl_Interp *interp);

EXTERN int is_agent_via_server
	(Tcl_Interp *interp);

EXTERN int is_agent_root
	(Tcl_Interp *interp);

EXTERN struct AGENT_ID *get_root_id
	(Tcl_Interp *interp);

EXTERN struct AGENT_ID *get_local_id
	(Tcl_Interp *interp);

EXTERN struct MACHINE_ID *get_actual_location
	(Tcl_Interp *interp);

EXTERN char *get_language
	(Tcl_Interp *interp);

  /* start, name, end and forcibly terminate an agent */

EXTERN struct AGENT_ID *Agent_Begin 
	(Tcl_Interp *interp, double seconds, struct MACHINE_ID *id);

EXTERN struct AGENT_ID *Agent_Name
	(Tcl_Interp *interp, double seconds, char *name);
 
EXTERN int Agent_End 
	(Tcl_Interp *interp, double seconds);
 
EXTERN int Agent_Force 
	(Tcl_Interp *interp, double seconds, 
	struct AGENT_ID *id, struct AGENT_ID **forcedId);
 
  /* turn the agent into a root agent */

EXTERN int Agent_Root
	(Tcl_Interp *interp);

  /* submit and fork the agent */

EXTERN struct AGENT_ID *Agent_Submit 
	(Tcl_Interp *interp, double seconds, 
	 struct MACHINE_ID *id, char *language,
	 int numProcedures, char **procedures, 
         int numVariables, char **variables,
         char *script);

EXTERN struct AGENT_ID *Agent_Fork 
	(Tcl_Interp *interp, double seconds, struct MACHINE_ID *machine_id);

EXTERN int Agent_Jump
	(Tcl_Interp *interp, double seconds, struct MACHINE_ID *machine_id);

  /* save a jump image to disk and transfer a saved jump image to a server */

EXTERN char *Disk_Jump
	(Tcl_Interp *interp);

EXTERN int Agent_Transfer
	(Tcl_Interp *interp, double seconds, struct MACHINE_ID *machine_id,
	 char *filename);

  /* wait for meetings, events, messages and activity on arbitrary sockets */

EXTERN int Agent_Select
	(Tcl_Interp *interp, double seconds, int n, int *fd, int *ready);
 
  /* send and receive results */ 

EXTERN int Agent_Send
	(Tcl_Interp *interp, double seconds, struct AGENT_ID *id, 
	 int code, char *string);

EXTERN int Agent_Receive
	(Tcl_Interp *interp, double seconds, struct MESSAGE_EXT **message);

  /* send and receive events */

EXTERN int Agent_Event
	(Tcl_Interp *interp, double seconds, struct AGENT_ID *id, 
	 char *tag, char *string);

EXTERN int Agent_GetEvent
	(Tcl_Interp *interp, double seconds, struct EVENT_EXT **event);

  /* request and accept meetings */

EXTERN int Agent_Req
	(Tcl_Interp *interp,  double seconds, struct AGENT_ID *id, 
	 UINT_8 meet_status, UINT_32 port);

EXTERN int Agent_GetReq
	(Tcl_Interp *interp, double seconds, struct MEETING_EXT **meeting);

  /* sleep for a certain number of seconds */

EXTERN int Agent_Sleep
	(Tcl_Interp *interp, double seconds);

EXTERN double Agent_Elapsed
	(Tcl_Interp *interp);

  /* send the result of script execution to the root agent */

EXTERN int Agent_SendResult
	(Tcl_Interp *interp, int retries, int maxSleep, UINT_32 code);

  /* get information about agents */

EXTERN struct ID_LIST *Agent_InfoIds
	(Tcl_Interp *interp, double seconds, struct MACHINE_ID *id);

EXTERN struct NAME_LIST *Agent_InfoNames
	(Tcl_Interp *interp, double seconds, struct MACHINE_ID *id);
 
EXTERN struct AGENT_INFO *Agent_Info
        (Tcl_Interp *interp, double seconds, struct AGENT_ID *id);

  /* message, meeting and event masks */

EXTERN int Agent_MaskNew 
	(Tcl_Interp *interp);

EXTERN int Agent_MaskDelete 
	(Tcl_Interp *interp, int handle);

EXTERN int Agent_MaskFill 
	(Tcl_Interp *interp, int handle);

EXTERN int Agent_MaskAdd
	(Tcl_Interp *interp, int handle, int count, struct MASK_ENTRY **masks);

EXTERN int Agent_MaskEmpty
	(Tcl_Interp *interp, int handle);

EXTERN int Agent_MaskRemove
	(Tcl_Interp *interp, int handle, int count, struct MASK_ENTRY **masks);
	
EXTERN struct MASK *Agent_MaskContents
	(Tcl_Interp *interp, int handle);

EXTERN struct MASK_ENTRY *Agent_SplitMask 
	(Tcl_Interp *interp, char *string);

EXTERN Tcl_DString *Agent_MaskToString 
	(Tcl_Interp *interp, struct MASK_ENTRY *entry);

EXTERN int Agent_MaskSetDefault 
	(Tcl_Interp *interp, int handle, char *handleName);

EXTERN int Agent_MaskGetDefault
	(Tcl_Interp *interp, int handle, char **handleName);

  /* initialization and main routine */

EXTERN int Agent_Init 
	(Tcl_Interp *interp, char *language, int server);

EXTERN void Agent_Main 
	(int argc, char **argv, Agent_AppInitProc *appInitProc);

  /* support routines */

EXTERN struct AGENT_ID *Agent_SplitId
	(Tcl_Interp *interp, char *string);

EXTERN Tcl_DString *Agent_IdToString
	(Tcl_Interp *interp, struct AGENT_ID *id);

EXTERN Tcl_DString *Agent_MachineToString
        (Tcl_Interp *interp, struct MACHINE_ID *id);

EXTERN Tcl_DString *Agent_MeetingToString
	(Tcl_Interp *interp, UINT_8 meeting_status);

#endif
