/* Agent Tcl
   Bob Gray
   15 July 1995

   tclAgentInt.cc

   This file implements the internal routines that are called from more than
   one source file.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#ifndef COSMO
#include <unistd.h>
#else
#include </usr/local/lib/gcc-lib/rs6000-ibm-aix3.2.5/2.6.3/include/unistd.h>
#endif
#include "tclAgentInt.h"
#include "tclMask.h"
#include "message.h"
#include "my_alloc.h"
#include "timers.h"
#include "transmit.h"

  /* send a REQ_END message and get the response */

int agentForceEnd (Tcl_Interp *interp, double seconds, AGENT_ID *id, AGENT_ID **forcedId, UINT_8 end_type)
{
  int sockfd;
  clock_t stop;
  MASK_SET *mask_set;
  RESTRICT *restrict;
  MESSAGE *response = NULL;
  int returnCode = TCL_ERROR;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (id != NULL);
  assert ((end_type == END_NORMAL) || (forcedId != NULL));
  assert ((end_type == END_FORCE) || (forcedId == NULL));
  assert ((end_type == END_NORMAL) || (end_type == END_FORCE));

    /* MASK_SET and RESTRICT */

  mask_set = MASK_SET::get_mask_set (interp);
  restrict = RESTRICT::get_restrict (interp);
  assert (mask_set != NULL);
  assert (restrict != NULL);

    /* turn off the SIGIO interrupt if we are ending our agent */

  if (end_type == END_NORMAL) {
    mask_set -> sigio_off ();
  } else {
    *forcedId = NULL;
  }

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);
 
    /* construct the message */

  MESSAGE message (up_socket.messages[REQ_END]);
  message.elements[0].string = (end_type == END_NORMAL) ? (char *) NULL : id -> name;
  message.elements[1].number = id -> id;
  message.elements[2].number = end_type;

  if ((sockfd = message_conn_and_send (id -> server, id -> ip, message, stop, FALSE)) < 0) {
    Tcl_AppendResult (interp, "unable to send to \"", id -> server, "\"", (char *) NULL);
  } else if ((response = message_receive (sockfd, down_socket)) == NULL) {
    Tcl_AppendResult (interp, "\"", id -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if ((response -> flag != RESP_ID) && (end_type == END_FORCE)) {
    returnCode = TCL_OK;
  } else if (response -> flag != RESP_ID) {
    Tcl_AppendResult (interp, "\"", id -> server, "\" unable to comply (server error)", (char *) NULL);
  } else if (end_type == END_FORCE) {
    char *server = response -> elements[0].string;
    UINT_32 ip   = response -> elements[1].number;
    char *name   = response -> elements[2].string;
    UINT_32 id   = response -> elements[3].number;
    *forcedId    = new AGENT_ID (server, ip, name, id);
    returnCode   = TCL_OK;
  } else {
    returnCode   = TCL_OK;
  }

    /* clean up */

  delete_check (response);
  restrict -> remove ();
  close (sockfd);
  return (returnCode);
}

  /* send a REQ_BEGIN message and get the response */

AGENT_ID *agentForceBegin (Tcl_Interp *interp, double seconds, MACHINE_ID *actual, MACHINE_ID *dest)
{
  int sockfd;
  clock_t stop;
  MASK_SET *mask_set;
  RESTRICT *restrict;
  AGENT_ID *id = NULL; 
  MESSAGE *mesg_from_server;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (actual != NULL);
  assert (dest != NULL);

    /* MASK_SET and RESTRICT */

  mask_set = MASK_SET::get_mask_set (interp);
  restrict = RESTRICT::get_restrict (interp);
  assert (mask_set != NULL);
  assert (restrict != NULL);

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);
 
    /* send the message */

  MESSAGE mesg_to_server (up_socket.messages[REQ_BEGIN]);
  mesg_to_server.elements[0].string = actual -> server;
  mesg_to_server.elements[1].number = actual -> server_ip;
  mesg_to_server.elements[2].number = getpid();

  sockfd = message_conn_and_send (dest -> server, dest -> server_ip, mesg_to_server, stop, FALSE);

    /* remove the restriction and return on error */

  restrict -> remove ();

  if (sockfd < 0) {
    Tcl_AppendResult (interp, "unable to send to \"", dest -> server, "\"", (char *) NULL);
    return NULL;
  }

    /* get the response */

  if ((mesg_from_server = message_receive (sockfd, down_socket)) == NULL) {
    Tcl_AppendResult (interp, "\"", dest -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if (mesg_from_server -> flag != RESP_ID) {
    Tcl_AppendResult (interp, "\"", dest -> server, "\" unable to comply (server error)", (char *) NULL);
  } else {

      /* assemble the agent identification */
 
    char *server       = mesg_from_server -> elements[0].string;
    UINT_32 ip         = mesg_from_server -> elements[1].number;
    UINT_32 numeric_id = mesg_from_server -> elements[3].number;
    id                 = new AGENT_ID (server, ip, NULL, numeric_id);

      /* set up the background handler */

    if (mask_set -> sigio_on (id) < 0) {
      agentForceEnd (interp, seconds, id, NULL, END_NORMAL);
      Tcl_SetResult (interp, "unable to establish background handler", TCL_STATIC);
      delete id;
      id = NULL;  
    }
  }

    /* cleanup */

  delete_check (mesg_from_server);
  close (sockfd);
  return id;
} 
