/* Agent Tcl
   Bob Gray
   27 February 1995

   tclBasic.cc

   This file implements the library routines that handle migration, 
   message passing and meetings.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#ifdef AIX
#include <sys/select.h>
#endif
#include <sys/stat.h>
#include <sys/time.h>
#include <assert.h>
#ifdef SYSV
#ifndef SOLARIS
#include <bstring.h>
#endif
#endif
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "fd_utilities.h"
#include "message.h"
#include "my_alloc.h"
#include "my_strings.h"
#include "redirect.h"
#include "tcl.h"
#include "tclAgent.h"
#include "tclAgentInt.h"
#include "tclLocation.h"
#include "tclMask.h"
#include "tclRestrict.h"
#include "tclTcpip.h"
#include "tcl_utilities.h"
#include "tcpip.h"
#include "transmit.h"
#include "truefalse.h"

  /* buffer size for file to socket transfer */

const BUFFER_SIZE = 1024;

  /* locations */

int is_agent_registered (Tcl_Interp *interp)
{
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);
  return (locations -> registered == AGENT_REGISTERED);
}

int is_agent_via_server (Tcl_Interp *interp)
{
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);
  return (locations -> server);
}

int is_agent_root (Tcl_Interp *interp)
{
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);
  return (locations -> toplevel == AGENT_ROOT);
}

AGENT_ID *get_root_id (Tcl_Interp *interp)
{
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);
  AGENT_ID *id = new AGENT_ID (*locations -> root);
  return id;
}

AGENT_ID *get_local_id (Tcl_Interp *interp)
{
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);
  AGENT_ID *id = new AGENT_ID (*locations -> local);
  return id;
}

MACHINE_ID *get_actual_location (Tcl_Interp *interp)
{
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);
  MACHINE_ID *id = new MACHINE_ID (*locations -> actual);
  return id;
}

char *get_language (Tcl_Interp *interp)
{
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);
  return (my_strcpy (locations -> language)); 
}

/* Agent_Select

   Purpose: Wait for a message, event, meeting request and/or data on 
            arbitrary sockets.  This is the agent equivalent of the Unix
            "select" command.

     Input: interp  = the current interpreter
            seconds = the timeout interval
            n       = number of file descriptors of interest
            fd      = file descriptors of interest
            ready   = ready flags  

            The dummy file descriptors, MEETING_FD, MESSAGE_FD and EVENT_FD,
            are used to indicate interest in messages, events and meeting
            requests.

    Output: The procedure returns TCL_ERROR and sets the interpreter result to
            an appropriate error message.  Otherwise the procedure waits 
            until one or more of the specified file descriptors are ready
            for reading.  ready[i] is set to 1 if fd[i] is ready for reading
            and 0 otherwise.  Then the procedure returns TCL_OK.
*/

int Agent_Select (Tcl_Interp *interp, double seconds, int n, int *fd, int *ready)
{
  int maxFd;                  /* largest file descriptor of interest       */
  int countFd;                /* number of ready file descriptors          */
  clock_t stop;               /* absolute timeout time (clock ticks)       */
  char temp[16];              /* scratch area for building error messages  */
  fd_set readFd;              /* set of file descriptors of interest       */
  fd_set readyFd;             /* set of ready file descriptors             */
  int countTypes;             /* number of ready incoming queues           */
  register int i;             /* loop counter                              */
  INCOMING *item;             /* item in one of the queues                 */
  register int nq;            /* number of items in a queue                */
  MASK_SET *masks;            /* the masks associated with the agent       */
  sigset_t oldMask;           /* old signal mask                           */
  sigset_t newMask;           /* new signal mask                           */
  RESTRICT *restrict;         /* time restrictions                         */
  register int which;         /* loop counter                              */
  int rc = TCL_ERROR;         /* return code from this procedure           */
  struct timeval timeout;     /* timeout for the "select" command          */
  struct timeval *timeptr;    /* pointer to the timeout                    */ 
  int readTypes[NUM_ITEMS];   /* incoming queues of interest               */
  int readyTypes[NUM_ITEMS];  /* ready incoming queues                     */
  int indexTypes[NUM_ITEMS];  /* index of last examined item in each queue */
  AGENT_LOCATION *locations;  /* current agent location                    */
  
    /* assertions on the parameters */

  assert (n > 0);
  assert (fd != NULL);
  assert (ready != NULL);
  assert (interp != NULL);

    /* AGENT_LOCATION, RESTRICT and MASK_SET */

  locations = AGENT_LOCATION::get_location (interp);
  restrict = RESTRICT::get_restrict (interp);
  masks = MASK_SET::get_mask_set (interp);
  assert (locations != NULL);
  assert (restrict != NULL);
  assert (masks != NULL);

    /* make sure that the agent is registetered */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return TCL_ERROR;
  }

    /* load the fd_set and the types array */

  maxFd = 0;
  FD_ZERO (&readFd);
  FD_ZERO (&readyFd);

  for (which = 0; which < NUM_ITEMS; which++) {
    readTypes [which] = 0;
    readyTypes[which] = 0;
    indexTypes[which] = 0;
  }

  for (i = 0; i < n; i++) {
    if (fd[i] == MEETING_FD) {
      readTypes[MEETING_ITEM] = TRUE;
    } else if (fd[i] == MESSAGE_FD) {
      readTypes[MESSAGE_ITEM] = TRUE;
    } else if (fd[i] == EVENT_FD) {
      readTypes[EVENT_ITEM] = TRUE;
    } else if (fd[i] < 0) {
      sprintf (temp, "%d", fd[i]);
      Tcl_AppendResult (interp, "invalid file descriptor \"", temp, "\"", (char *) NULL);
      return TCL_ERROR;
    } else {
      FD_SET (fd[i], &readFd); 
      if (fd[i] > maxFd) {
        maxFd = fd[i];
      }
    }
  }

    /* do a nonblocking select to prime the pump */

  timeout.tv_sec  = 0;
  timeout.tv_usec = 0;
  readyFd         = readFd;

  if ((countFd = select (maxFd + 1, &readyFd, NULL, NULL, &timeout)) < 0) {
    Tcl_AppendResult (interp, "select failed: ", Tcl_PosixError (interp), (char *) NULL);
    return TCL_ERROR; 
  }

    /* add the server socket to the set */

  FD_SET (masks -> sockfd, &readFd);
 
  if (masks -> sockfd > maxFd) {
    maxFd = masks -> sockfd;
  }
 
    /* set the restriction */

  if (seconds >= 0.0) {
    PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
    restrict -> add (&permit);
    restrict -> first_wall (stop);
  }

    /* loop forever (we will break out when something becomes ready) */

  while (1) {

      /* block SIGIO */

    sigemptyset (&oldMask);
    sigemptyset (&newMask);
    sigaddset (&newMask, SIGIO);
    sigprocmask (SIG_BLOCK, &newMask, &oldMask);

      /* receive any incoming items; bail on broken connection */

    if (masks -> receiveIncomingItems() < 0) {
      Tcl_AppendResult (interp, "broken server connection", (char *) NULL);
      break;
    }

      /* check the meeting, message and event queues */

    countTypes = 0;

    for (which = 0; which < NUM_ITEMS; which++) {

      if ((readTypes[which] == TRUE) && (masks -> masks[which] != NULL)) {

        nq = masks -> queues[which].get_count ();

        for (i = indexTypes[which]; i < nq; i++) {

          item = (INCOMING *) masks -> queues[which].peek (i);

          if (masks -> masks[which] -> filter (item)) {
            readyTypes[which] = TRUE;
            countTypes += 1;
            break;
          }
        }

        indexTypes[which] = i;
      }
    }  

      /* return if we have one or more ready things */

    if ((countFd > 0) || (countTypes > 0)) {
      for (i = 0; i < n; i++) {
        if (fd[i] == MESSAGE_FD) {
          ready[i] = readyTypes[MESSAGE_ITEM];
        } else if (fd[i] == MEETING_FD) {
          ready[i] = readyTypes[MEETING_ITEM];
        } else if (fd[i] == EVENT_FD) {
          ready[i] = readyTypes[EVENT_ITEM];
        } else {
          ready[i] = FD_ISSET (fd[i], &readyFd) ? 1 : 0;
        }
      }
    
      sigprocmask (SIG_SETMASK, &oldMask, NULL); 
      rc = TCL_OK;
      break;
    }

      /* do a select that waits forever or until timeout */

    readyFd = readFd;

    do {

        /* get the select timeout */

      if (seconds >= 0.0) {
        timeptr = TIMERS::ticksToTimeval (stop, timeout);
      } else {
        timeptr = NULL;
      }

        /* do the select */
  
      countFd = select (maxFd + 1, &readyFd, NULL, NULL, timeptr);

    } while ((countFd < 0) && (errno == EINTR));

      /* unblock SIGIO */

    sigprocmask (SIG_SETMASK, &oldMask, NULL); 

       /* check for select errors */

    if (countFd < 0) {
      Tcl_AppendResult (interp, "select failed: ", Tcl_PosixError (interp), (char *) NULL);
      break;
    } else if (countFd == 0) { 
      Tcl_SetErrorCode (interp, "TIMEOUT", (char *) NULL);
      Tcl_AppendResult (interp, "timeout", (char *) NULL);
      break;
    }

      /* subtract one if the server socket was ready since the server */
      /* socket is not one of the user-specified sockets              */

    if (FD_ISSET (masks -> sockfd, &readyFd)) {
      countFd -= 1;
    }
  }

    /* cleanup and return */

  if (seconds >= 0.0) {
    restrict -> remove ();
  }

  return (rc);
}

/* Agent_Transfer

   Purpose: Transfer a server message from a disk file to a server

     Input: interp   = the current interpreter
            seconds  = the timeout interval
            machine  = destination server
            filename = name of the file

    Output: The procedure returns TCL_ERROR and sets the interpreter result
            to an appropriate error message on error.  Otherwise the procedure
            returns TCL_OK and transfers the message from the file to the 
            server.
*/

int Agent_Transfer (Tcl_Interp *interp, double seconds, MACHINE_ID *machine, char *filename) 
{
  int fd;
  int nread;
  int sockfd;
  clock_t stop; 
  char *substname;
  MESSAGE *message;
  RESTRICT *restrict;
  Tcl_DString scratch;
  int code = TCL_ERROR;
  char buffer[BUFFER_SIZE];
  AGENT_LOCATION *locations;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (machine != NULL);
  assert (filename != NULL);

    /* AGENT_LOCATION and RESTRICT */

  locations = AGENT_LOCATION::get_location (interp);
  restrict = RESTRICT::get_restrict (interp);
  assert (locations != NULL);
  assert (restrict != NULL);

    /* make sure that the agent is registered */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return TCL_ERROR;
  }

    /* do tilde substitution and open the file */

  if ((substname = Tcl_TildeSubst (interp, filename, &scratch)) == NULL) {
    return TCL_ERROR;
  }

  fd = open (substname, O_RDONLY, 0);
  Tcl_DStringFree (&scratch);

  if (fd < 0) {
    Tcl_AppendResult (interp, "couldn't open \"", filename, "\": ", Tcl_PosixError(interp), (char *) NULL);
     return TCL_ERROR;
  }

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);

    /* connect to the server */

  if ((sockfd = message_conn (machine -> server, machine -> server_ip, stop, FALSE)) < 0) {
    Tcl_AppendResult (interp, "unable to send to \"", machine -> server, "\"", (char *) NULL);
    goto cleanup;
  }   

    /* copy the file onto the socket */

  while ((nread = read (fd, buffer, 1024)) > 0) {
    if (tcpip_writen (sockfd, buffer, nread) < nread) {
      Tcl_AppendResult (interp, "broken connection to \"", machine -> server, "\": ", Tcl_PosixError(interp), (char *) NULL);
      goto cleanup; 
    }
  }

  if (nread < 0) {
    Tcl_AppendResult (interp, "couldn't read from \"", filename, "\": ", Tcl_PosixError (interp), (char *) NULL);
    goto cleanup;
  }

    /* wait for the server response */

  if ((message = message_receive (sockfd, down_socket)) == NULL) {
    Tcl_AppendResult (interp, "\"", machine -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if (message -> flag == RESP_ERROR) {
    Tcl_AppendResult (interp, "\"", machine -> server, "\" unable to comply (server error)", (char *) NULL);
  } else {
    code = TCL_OK;
  } 

cleanup:

  restrict -> remove ();
  close_check (sockfd);
  return (code);
}

/* Agent_Root

   Purpose: Turn the current agent into the root

     Input: interp = the current interpreter
 
    Output: The procedure returns TCL_ERROR and sets the interpreter result
            to an appropriate error message one error.  Otherwise the procedure
            returns TCL_OK.
*/

int Agent_Root (Tcl_Interp *interp)
{
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);

    /* check whether the agent has been registered */

  if (locations -> registered == AGENT_NOT_REGISTERED)
  {
    Tcl_AppendResult (interp, "agent has NOT been registered", (char *) NULL);
    return TCL_ERROR;
  }

    /* reload the agent identification */
 
  locations -> reload (locations -> local, locations -> local, AGENT_REGISTERED, AGENT_ROOT);
  return TCL_OK;
}

/* Agent_Req

   Purpose: Send a meeting request to an agent

     Input: interp  = Tcl interpreter
            seconds = timeout interval
            id      = identification of the destination agent
            status  = meeting status
            port    = meeting TCP/IP port

    Output: The procedure returns TCL_ERROR and sets the interpreter
            result to an error message on error.  Otherwise the procedure
            returns TCL_OK and sends the meeting request.
*/

int Agent_Req (Tcl_Interp *interp, double seconds, AGENT_ID *id, UINT_8 status, UINT_32 port)
{ 
  clock_t stop;
  int sockfd = -1;
  RESTRICT *restrict;
  AGENT_LOCATION *locations;
  int returnCode = TCL_ERROR;
  MESSAGE *mesg_from_server = NULL;
       
    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (id != NULL);
  assert ((status == MEET_CONNECT) || (status == MEET_ACCEPT) || (status == MEET_REFUSE) || (status == MEET_REQUEST));

    /* RESTRICT and AGENT_LOCATION */

  restrict = RESTRICT::get_restrict (interp);
  locations = AGENT_LOCATION::get_location (interp);
  assert (restrict != NULL);
  assert (locations != NULL);
 
    /* check whether the agent is registered */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has NOT been registered", (char *) NULL);
    return TCL_ERROR;
  }

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);

    /* send the message */

  MESSAGE mesg_to_server (up_socket.messages[REQ_MEETING]);
  mesg_to_server.elements[0].string = locations -> local -> server;
  mesg_to_server.elements[1].number = locations -> local -> ip;
  mesg_to_server.elements[2].string = locations -> local -> name;
  mesg_to_server.elements[3].number = locations -> local -> id;
  mesg_to_server.elements[4].string = id -> name;
  mesg_to_server.elements[5].number = id -> id; 
  mesg_to_server.elements[6].number = status;
  mesg_to_server.elements[7].string = locations -> actual -> server;
  mesg_to_server.elements[8].number = locations -> actual -> server_ip;
  mesg_to_server.elements[9].number = port;

  if ((sockfd = message_conn_and_send (id -> server, id -> ip, mesg_to_server, stop, FALSE)) < 0) {
    Tcl_AppendResult (interp, "unable to send to \"", id -> server, "\"", (char *) NULL);
  } else if ((mesg_from_server = message_receive (sockfd, down_socket)) == NULL) {
    Tcl_AppendResult (interp, "\"", id -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if (mesg_from_server -> flag != RESP_OK) {
    Tcl_AppendResult (interp, "\"", id -> server, "\" unable to comply (server error)", (char *) NULL);
  } else {
    returnCode = TCL_OK;
  }

    /* remove the restriction */

  restrict -> remove ();

    /* clean up */

  delete_check (mesg_from_server);
  close_check (sockfd);
  return returnCode;
}

/* Agent_GetReq

      Purpose: Get a meeting request

        Input: interp  = Tcl interpeter
               seconds = timeout interval (0 means poll and -1 means forever)

       Output: The procedure returns TCL_ERROR and sets the intrepreter result
               to an error message if an error occurs.  The procedure returns
               TCL_OK and sets "*meeting" to NULL if no meeting request arrives
               during the timeout interval.  Otherwise the procedure sets
               "*meeting" to a dynamically allocated MEETING_EXT structure
               that contains the meeting request.
*/

int Agent_GetReq (Tcl_Interp *interp, double seconds, MEETING_EXT **meeting)
{
  int maskCode;  
  int returnCode;
  INCOMING_MEETING *inc_meeting;
  MASK_SET *masks = MASK_SET::get_mask_set (interp);
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);
  
    /* check that the agent is registered */
 
  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return TCL_ERROR;
  }

    /* add a time restriction if necessary */

  if (seconds >= 0.0) {
    PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
    Restrict_AddPermit (interp, &permit);
  }
 
    /* check the internal queues for a meeting request */

  *meeting = NULL;

  if ((maskCode = masks -> check_queues (MEETING_ITEM, (INCOMING **) &inc_meeting)) == INVALID_MASK) {
    Tcl_AppendResult (interp, "no meeting mask in the \"mask\" array", (char *) NULL);
    returnCode = TCL_ERROR;
  } else if (maskCode == INVALID_CONNECTION) {
    Tcl_AppendResult (interp, "broken connection to server", (char *) NULL);
    returnCode = TCL_ERROR; 
  } else if (inc_meeting != NULL) {
    *meeting = new MEETING_EXT (inc_meeting);
    delete inc_meeting;
    returnCode = TCL_OK;
  } else {
    returnCode = TCL_OK;
  }

    /* remove the time restriction */

  if (seconds >= 0.0) {
    Restrict_RemovePermit (interp);
  }

  return (returnCode);
} 

/* Agent_Begin

      Purpose: Notify a server that we are now an agent

        Input: interp  = Tcl interpreter
               seconds = seconds before timeout (0 or greater)
               dest    = machine identification
 
       Output: The procedure returns NULL and sets the interpreter result to
               an error message on error.  Otherwise the procedure returns
               a pointer to an AGENT_ID structure that contains the new agent
               identification.  The AGENT_ID structure and its string elements
               are DYNAMICALLY allocated.
*/

AGENT_ID *Agent_Begin (Tcl_Interp *interp, double seconds, MACHINE_ID *dest)
{
  AGENT_ID *id;
  AGENT_LOCATION *locations;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (dest != NULL);

    /* AGENT_LOCATION */

  locations = AGENT_LOCATION::get_location (interp);
  assert (locations != NULL);

    /* use the IP address of the current host if no host is specified */

  if ((dest -> server == NULL) && (dest -> server_ip == UNKNOWN_IP)) {
    dest -> server_ip = locations -> actual -> server_ip;
    dest -> server    = my_strcpy (locations -> actual -> server);
  }

    /* check whether the agent is registered */
  
  if (locations -> registered == AGENT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has been registered", (char *) NULL);
    return NULL;
  }

    /* get the agent identification */

  if ((id = agentForceBegin (interp, seconds, locations -> actual, dest)) != NULL) {
    locations -> reload (id, id, AGENT_REGISTERED, AGENT_ROOT);
  }

  return id; 
} 

/* Agent_Name

      Purpose: Assign a name to an agent

        Input: interp  = the Tcl interpreter
               seconds = timeout interval
               name    = name for the agent

       Output: The procedure returns NULL and sets the interpreter result
               to an error message on error.  Otherwise the procedure returns
               a pointer to an AGENT_ID structure that contains the new
               agent identification.  The AGENT_ID structure and its string
               elements are DYNAMICALLY allocated.
*/

AGENT_ID *Agent_Name (Tcl_Interp *interp, double seconds, char *name)
{
  int num;
  clock_t stop;
  int sockfd = -1;
  RESTRICT *restrict;
  AGENT_LOCATION *locations;
  AGENT_ID *local_id = NULL;
  MESSAGE *mesg_from_server = NULL;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (name != NULL);

    /* RESTRICT and AGENT_LOCATION */

  restrict = RESTRICT::get_restrict (interp);
  locations = AGENT_LOCATION::get_location (interp);
  assert (restrict != NULL);
  assert (locations != NULL);

    /* check whether the agent is registered and whether the name is nonempty */
  
  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has NOT been registered", (char *) NULL);
    return NULL;
  } else if (*name == '\0') {
    Tcl_AppendResult (interp, "name must be nonempty", (char *) NULL);
    return NULL;
  } else if (!strcmp (name, "server")) {
    Tcl_AppendResult (interp, "\"server\" is a reserved name", (char *) NULL);
    return NULL;
  }

    /* check that the name is not an integer */

  if (Tcl_GetInt (interp, name, &num) == TCL_OK) {
    Tcl_AppendResult (interp, "name can not be a number", (char *) NULL);
    return NULL;
  } else {
    Tcl_ResetResult (interp);
  }

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);

    /* send the message */
 
  MESSAGE mesg_to_server (up_socket.messages[REQ_NAME]);
  mesg_to_server.elements[0].number = locations -> local -> id;
  mesg_to_server.elements[1].string = name;

  if ((sockfd = message_conn_and_send (NULL, locations -> local -> ip, mesg_to_server, stop, FALSE)) < 0) {
    Tcl_AppendResult (interp, "unable to send to local server \"", locations -> local -> server, "\"", (char *) NULL);
  } else if ((mesg_from_server = message_receive (sockfd, down_socket)) == NULL) {
    Tcl_AppendResult (interp, "local server \"", locations -> local -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if (mesg_from_server -> flag != RESP_OK) {
    Tcl_AppendResult (interp, "local server \"", locations -> local -> server, "\" unable to comply (server error)", (char *) NULL);
  } else {
    locations -> local -> change_name (name);

    if (locations -> toplevel) {
      locations -> root -> change_name (name);
    }

    locations -> refresh ();
    local_id = new AGENT_ID (*locations -> local);
  }

    /* remove the restriction */

  restrict -> remove ();

    /* clean up */

  delete_check (mesg_from_server);
  close_check (sockfd); 
  return local_id;
}


/* Agent_End

      Purpose: Notify the server that we are no longer an agent

        Input: interp  = the Tcl interpreter
               seconds = number of seconds until timeout (0 or greater)
 
       Output: The procedure returns TCL_ERROR and sets the intrepreter
               result to an error message if an error occurs.  Otherwise
               the procedure returns TCL_OK.
*/

int Agent_End (Tcl_Interp *interp, double seconds)
{
  int returnCode;
  AGENT_LOCATION *locations;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  locations = AGENT_LOCATION::get_location (interp);
  assert (locations != NULL);
 
    /* check whether the agent is registered and whether the agent is toplevel */
 
  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has NOT been registered", (char *) NULL);
    return TCL_ERROR;
  } else if (locations -> toplevel == AGENT_NOT_ROOT) {
    Tcl_AppendResult (interp, "agent is NOT a root agent", (char *) NULL);
    return TCL_ERROR;
  }

    /* send the END request and empty out the locations even on error */

  returnCode = agentForceEnd (interp, seconds, locations -> local, NULL, END_NORMAL);
  locations -> reload (NULL, NULL, AGENT_NOT_REGISTERED, AGENT_NOT_ROOT);
  return (returnCode);
}

/* Agent_Jump

      Purpose: Jump to a remote machine

        Input: interp     = Tcl interpreter
               seconds    = timeout interval
               machine_id = id of the remote machine  
 
       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an error message if it is unable to transport the agent
               to the remote machine.  The procedure returns TCL_OK and
               sets the interpreter result to "SAME" if the remote machine
               is the same as the current machine.  Otherwise the procedure
               calls exit and the agent resumes execution on the remote machine
               with the interpreter result set to "JUMPED".
*/

static MESSAGE *agentCapture (Tcl_Interp *interp, AGENT_LOCATION *locations) 
{
  char *string;
  MESSAGE *message;
  Tcl_DString error;
  Tcl_DString *state;

    /* get the state image */

  Tcl_SetResult (interp, "JUMPED", TCL_STATIC);   
  Tcl_DStringInit (&error);

  if ((state = Tcl_SaveState (interp, &error)) == NULL) { 
    Tcl_DStringResult (interp, &error);
    Tcl_DStringFree (&error);
    return NULL;
  }

  Tcl_DStringFree (&error);
  Tcl_ResetResult (interp);

    /* get the state information out of the Tcl_DString */

  if (state -> string == state -> staticSpace) {
    string = my_strcpy (state -> string);
  } else {
    string                  = state -> string;
    state -> string         = state -> staticSpace;
    state -> staticSpace[0] = '\0';
    state -> length         = 0;
    state -> spaceAvl       = TCL_DSTRING_STATIC_SIZE;
  }

    /* construct the message */

  message = new MESSAGE (up_socket.messages[REQ_STATE]);

  if (!locations -> toplevel) {
    message -> elements[0].string = locations -> root -> server;
    message -> elements[1].number = locations -> root -> ip;
    message -> elements[2].string = locations -> root -> name;
    message -> elements[3].number = locations -> root -> id;
  }

  message -> elements[4].string  = locations -> local -> server;
  message -> elements[5].number  = locations -> local -> ip;
  message -> elements[6].string  = locations -> local -> name;
  message -> elements[7].number  = locations -> local -> id;
  message -> elements[8].string  = locations -> language; 
  message -> elements[9].string  = string;
  message -> elements[9].dealloc = TRUE;

    /* clear out the interpreter result and return the message */

  Tcl_DStringFree (state);
  delete (state);
  return (message);
}

static int agentJump (Tcl_Interp *interp, MACHINE_ID *machine_id, AGENT_LOCATION *locations, clock_t stop)
{
  int sockfd = -1;
  MESSAGE *mesg_to_server = NULL;
  MESSAGE *mesg_from_server = NULL;

    /* get the IP address of the remote machine */

  if (machine_id -> server_ip == UNKNOWN_IP) {
    if (Tcpip_Getip (interp, machine_id -> server, &machine_id -> server_ip) < 0) {
      return TCL_ERROR;
    }
  }

    /* do not jump if the remote machine is the same as the current machine */

  if (machine_id -> server_ip == locations -> actual -> server_ip) {  
    Tcl_SetResult (interp, "SAME", TCL_STATIC);
    return TCL_OK;
  }     

    /* construct the jump message */

  if ((mesg_to_server = agentCapture (interp, locations)) == NULL) {
    return TCL_ERROR;
  }

    /* send the jump message */

  if ((sockfd = message_conn_and_send (machine_id -> server, machine_id -> server_ip, *mesg_to_server, stop, FALSE)) < 0) {
    Tcl_AppendResult (interp, "unable to send to \"", machine_id -> server, "\"", (char *) NULL);
  } else if ((mesg_from_server = message_receive (sockfd, down_socket)) == NULL) { 
    Tcl_AppendResult (interp, "\"", machine_id -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if (mesg_from_server -> flag != RESP_ID) {
    Tcl_AppendResult (interp, "\"", machine_id -> server, "\" unable to comply (server error)", (char *) NULL);
  } else {
    Tcl_Eval (interp, "exit"); 
    exit (0); 
  } 
 
    /* clean up */

  delete_check ((char *) mesg_from_server);
  delete_check ((char *) mesg_to_server);
  close_check (sockfd);
  return TCL_ERROR;
}

int Agent_Jump (Tcl_Interp *interp, double seconds, MACHINE_ID *machine_id)
{
  int code;
  clock_t stop;
  RESTRICT *restrict;
  AGENT_LOCATION *locations;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (machine_id != NULL);

    /* RESTRICT and AGENT_LOCATION */

  restrict = RESTRICT::get_restrict (interp);
  locations = AGENT_LOCATION::get_location (interp);
  assert (restrict != NULL);
  assert (locations != NULL);

    /* check whether the agent is registered */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return TCL_ERROR;
  } else if (interp -> interactive) {
    Tcl_AppendResult (interp, "can not JUMP in interactive mode", (char *) NULL);
    return TCL_ERROR;
  }

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);

    /* jump */

  code = agentJump (interp, machine_id, locations, stop);

    /* remove the restriction */

  restrict -> remove ();
  return (code);
}

/* Disk_Jump

   Purpose: Jump to a remote machine (but save the TRAN_STATE message on disk
            for later transmission rather than sending the TRAN_STATE message
            directly to the remote server)

     Input: interp = the current interpreter

    Output: The procedure returns NULL and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure
            saves the TRAN_STATE message to a temporary file and returns the
            name of the file.  The name is dynamically allocated.
*/

char *Disk_Jump (Tcl_Interp *interp)
{
  int fd;
  int code;
  MESSAGE *message = NULL;
  AGENT_LOCATION *locations;
  char *filename = my_strcpy ("/tmp/stateXXXXXX");

    /* assertions on the parameters */

  assert (interp != NULL);

    /* AGENT_LOCATION */

  locations = AGENT_LOCATION::get_location (interp);
  assert (locations != NULL);

    /* make sure that we can jump */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return NULL;
  } else if (interp -> interactive) {
    Tcl_AppendResult (interp, "can not JUMP in interactive mode", (char *) NULL);
    return NULL;
  } 

    /* create a temporary file */

  if ((fd = mkstemp (filename)) < 0) {
    Tcl_AppendResult (interp, "unable to create temporary file", (char *) NULL);
    return NULL;
  }

	/* change the mode of the file to everybody readable */

  fchmod (fd, 0644);

    /* capture the state image */

  if ((message = agentCapture (interp, locations)) == NULL) {
    code = TCL_ERROR;
  } else if (message_send (fd, *message) < 0) {
    Tcl_AppendResult (interp, "unable to write to temporary file \"", filename, "\"", (char *) NULL);
    code = TCL_ERROR;
  } else {
    code = TCL_OK;
  }

    /* cleanup and return */

  close (fd); 
  delete_check (message); 

  if (code != TCL_OK) {
    unlink (filename);
    delete (filename);
    filename = NULL;
  }
 
  return (filename);
}


/* Agent_Fork

   Purpose: Create a copy of an agent on a remote machine

     Input: interp     = the current interpreter
            seconds    = timeout interval 
            machine_id = id of the remote machine

    Output: The procedure returns TCL_ERROR and sets the interpreter result to
            an appropriate error message if it is unable to fork the child 
            agent.  Otherwise the procedure returns TCL_OK and sets the 
            interpreter result to the identification of the new child agent.
            The interpreter result in the child agent is set to "CHILD".
*/

AGENT_ID *agentFork (Tcl_Interp *interp, MACHINE_ID *machine_id, AGENT_LOCATION *locations, clock_t stop)
{
  int sockfd = -1;
  Tcl_DString error;
  AGENT_ID *id = NULL;
  Tcl_DString *state = NULL;
  MESSAGE *mesg_from_server = NULL;

    /* get the state image */

  Tcl_SetResult (interp, "CHILD", TCL_STATIC);   
  Tcl_DStringInit (&error);

  if ((state = Tcl_SaveState (interp, &error)) == NULL) { 
    Tcl_DStringFree (&error);
    Tcl_DStringResult (interp, &error);
    return NULL;
  }

  Tcl_DStringFree (&error);
  Tcl_ResetResult (interp);

    /* send the message */

  MESSAGE mesg_to_server (up_socket.messages[REQ_STATE]);
  mesg_to_server.elements[0].string = locations -> root -> server;
  mesg_to_server.elements[1].number = locations -> root -> ip;
  mesg_to_server.elements[2].string = locations -> root -> name;
  mesg_to_server.elements[3].number = locations -> root -> id;
  mesg_to_server.elements[4].string = locations -> local -> server;
  mesg_to_server.elements[5].number = locations -> local -> ip;
  mesg_to_server.elements[6].string = locations -> local -> name;
  mesg_to_server.elements[7].number = locations -> local -> id;
  mesg_to_server.elements[8].string = locations -> language;
  mesg_to_server.elements[9].string = Tcl_DStringValue (state);

  if ((sockfd = message_conn_and_send (machine_id -> server, machine_id -> server_ip, mesg_to_server, stop, FALSE)) < 0) { 
    Tcl_AppendResult (interp, "unable to send to \"", machine_id -> server, "\"", (char *) NULL);
  } else if ((mesg_from_server = message_receive (sockfd, down_socket)) == NULL) { 
    Tcl_AppendResult (interp, "\"", machine_id -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if (mesg_from_server -> flag != RESP_ID) {
    Tcl_AppendResult (interp, "\"", machine_id -> server, "\" unable to comply (server error)", (char *) NULL);
  } else {
    char   *server     = mesg_from_server -> elements[0].string;
    UINT_32 ip         = mesg_from_server -> elements[1].number;
    UINT_32 numeric_id = mesg_from_server -> elements[3].number;
    id                 = new AGENT_ID (server, ip,  NULL, numeric_id);
  }
 
    /* clean up */

  delete_check (mesg_from_server);
  tclFreeDString (state);
  close_check (sockfd);
  return id;
}

AGENT_ID *Agent_Fork (Tcl_Interp *interp, double seconds, MACHINE_ID *machine_id)
{
  AGENT_ID *id;
  clock_t stop;
  RESTRICT *restrict;
  AGENT_LOCATION *locations;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (machine_id != NULL);

    /* RESTRICT and AGENT_LOCATION */

  restrict = RESTRICT::get_restrict (interp);
  locations = AGENT_LOCATION::get_location (interp);
  assert (restrict != NULL);
  assert (locations != NULL);

    /* check whether the agent is registered */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return NULL;
  } else if (interp -> interactive) {
    Tcl_AppendResult (interp, "can not FORK in interactive mode", (char *) NULL);
    return NULL;
  } 

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);

    /* fork */

  id = agentFork (interp, machine_id, locations, stop);

    /* remove the restriction */

  restrict -> remove ();
  return (id);
}

/* Agent_GetEvent

      Purpose: Check to see if any events are available for the agent

        Input: interp    = Tcl interpreter
               seconds   = number of seconds
                     
               "seconds" is the time out interval.  The procedure waits 
               until a message arrive if seconds is less than 0.  The
               procedure waits until a message has arrived or until 
               "seconds" seconds have elapsed if seconds is 0 or greater.
                 
       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an error message if an error occurs.  Otherwise the 
               procedure returns TCL_OK and either (1) sets *event to NULL
               if no event arrived before the timeout expired or (2) sets 
               *event to a dynamically allocated EVENT_EXT structure that 
               contains the message.
*/

int Agent_GetEvent (Tcl_Interp *interp, double seconds, EVENT_EXT **event)
{
  int maskCode; 
  int returnCode;
  INCOMING_EVENT *inc_event;
  MASK_SET *masks = MASK_SET::get_mask_set (interp);
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);

    /* check that the agent is registered */
 
  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return TCL_ERROR;
  } 

    /* add a time restriction if nessary */

  if (seconds >= 0.0) {
    PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
    Restrict_AddPermit (interp, &permit);
  }
 
    /* check the internal queues for a message */

  *event = NULL;

  if ((maskCode = masks -> check_queues (EVENT_ITEM, (INCOMING **) &inc_event)) == INVALID_MASK) {
    Tcl_AppendResult (interp, "no event mask in the \"mask\" array", (char *) NULL);
    returnCode = TCL_ERROR;
  } else if (maskCode == INVALID_CONNECTION) {
    Tcl_AppendResult (interp, "broken connection to server", (char *) NULL);
    returnCode = TCL_ERROR;
  } else if (inc_event != NULL) {
    *event = new EVENT_EXT (inc_event);
    delete inc_event;
    returnCode = TCL_OK;
  } else {
    returnCode = TCL_OK;
  }

    /* remove the time restriction */

  if (seconds >= 0.0) {
    Restrict_RemovePermit (interp); 
  }

  return (returnCode);
}

/* Agent_Receive

      Purpose: Check to see if any messages are available for the agent

        Input: interp    = Tcl interpreter
               seconds   = number of seconds
                     
               "seconds" is the time out interval.  The procedure waits 
               until a message arrive if seconds is less than 0.  The
               procedure waits until a message has arrived or until 
               "seconds" seconds have elapsed if seconds is 0 or greater.
                 
       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an error message if an error occurs.  Otherwise the 
               procedure returns TCL_OK and either (1) sets *message to NULL
               if no message arrived before the timeout expired or (2) sets 
               *message to a dynamically allocated MESSAGE_EXT structure that 
               contains the message.
*/

int Agent_Receive (Tcl_Interp *interp, double seconds, MESSAGE_EXT **message)
{
  int maskCode;
  int returnCode;
  INCOMING_MESSAGE *inc_message; 
  MASK_SET *masks = MASK_SET::get_mask_set (interp);
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);

    /* check that the agent is registered */
 
  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return TCL_ERROR;
  } 

    /* add a time restriction if necessary */

  if (seconds >= 0.0) {
    PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
    Restrict_AddPermit (interp, &permit);
  }
     
    /* check the internal queues for a message */

  *message = NULL;

  if ((maskCode = masks -> check_queues (MESSAGE_ITEM, (INCOMING **) &inc_message)) == INVALID_MASK) {
    Tcl_AppendResult (interp, "no message mask in the \"mask\" array", (char *) NULL);
    returnCode = TCL_ERROR;
  } else if (maskCode == INVALID_CONNECTION) {
    Tcl_AppendResult (interp, "broken connection to server", (char *) NULL);
    returnCode = TCL_ERROR;
  } else if (inc_message != NULL) {
    *message = new MESSAGE_EXT (inc_message);
    delete inc_message;
    returnCode = TCL_OK;
  } else {
    returnCode = TCL_OK;
  }

    /* remove the time restriction */

  if (seconds >= 0.0) {
    Restrict_RemovePermit (interp);
  }
 
  return (returnCode);
}

/* Agent_Event

      Purpose: Send an event to an agent

        Input: interp  = the Tcl interpreter
               seconds = timeout interval
               id      = the destination agent
               tag     = the event tag
               string  = the vent string

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and sends the event to the 
               destination agent.
*/

int Agent_Event (Tcl_Interp *interp, double seconds, AGENT_ID *id, char *tag, char *string)
{
  clock_t stop;
  int sockfd = -1;
  RESTRICT *restrict;
  AGENT_LOCATION *locations;
  int returnCode = TCL_ERROR;
  MESSAGE *mesg_from_server = NULL;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (id != NULL);
  assert (tag != NULL);
  assert (string != NULL);

    /* RESTRICT and AGENT_LOCATION */

  restrict = RESTRICT::get_restrict (interp);
  locations = AGENT_LOCATION::get_location (interp);
  assert (restrict != NULL);
  assert (locations != NULL);
 
    /* check whether the agent is registered */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has NOT been registered", (char *) NULL);
    return TCL_ERROR; 
  }

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);

    /* send the message */

  MESSAGE mesg_to_server (up_socket.messages[REQ_EVENT]);
  mesg_to_server.elements[0].string = locations -> local -> server; 
  mesg_to_server.elements[1].number = locations -> local -> ip;
  mesg_to_server.elements[2].string = locations -> local -> name;
  mesg_to_server.elements[3].number = locations -> local -> id;
  mesg_to_server.elements[4].string = id -> name;
  mesg_to_server.elements[5].number = id -> id;
  mesg_to_server.elements[6].string = tag;
  mesg_to_server.elements[7].string = string;

  if ((sockfd = message_conn_and_send (id -> server, id -> ip, mesg_to_server, stop, FALSE)) < 0) {
    Tcl_AppendResult (interp, "unable to send to \"", id -> server, "\"", (char *) NULL);
  } else if ((mesg_from_server = message_receive (sockfd, down_socket)) == NULL) {
    Tcl_AppendResult (interp, "\"", id -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if (mesg_from_server -> flag != RESP_OK) {
    Tcl_AppendResult (interp, "\"", id -> server, "\" unable to comply (server error)", (char *) NULL);
  } else {
    returnCode = TCL_OK;
  }
 
    /* remove restriction */

  restrict -> remove ();

    /* clean up */

  close_check (sockfd);
  delete_check (mesg_from_server);
  return returnCode;
}
 
/* Agent_SendResult

   Purpose: Send the result of script execution to the root agent

     Input: interp   = the current interpreter
            maxTries = maximum number of transmission attempts
            maxSleep = maximum number of seconds to wait between attempts
            code     = Tcl result code

    Output: The procedure returns TCL_ERROR if it is unable to send the result
            to the root agent and TCL_OK otherwise.  The procedure uses an
            exponential backoff algorithm and keeps trying to send the result
            until it either succeeds or makes the specified number of attempts.
*/

int Agent_SendResult (Tcl_Interp *interp, int maxTries, int maxSleep, UINT_32 code)
{
  int i = 0;
  char *argv[3];
  int sockfd = -1;
  int sleepTime = 1;
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);

    /* get the root and local identifications */

  AGENT_ID *local = locations -> local;
  AGENT_ID *root  = locations -> root;
      
    /* do not send to ourselves */

  if ((local -> ip == root -> ip) && (local -> id == root -> id)) {
    return TCL_OK;
  }

     /* set up the message */

  MESSAGE mesg_to_root (up_socket.messages[REQ_MESSAGE]);
  mesg_to_root.elements[0].string = local -> server;
  mesg_to_root.elements[1].number = local -> ip;
  mesg_to_root.elements[2].string = local -> name;
  mesg_to_root.elements[3].number = local -> id;
  mesg_to_root.elements[4].string = root -> name;
  mesg_to_root.elements[5].number = root -> id;
  mesg_to_root.elements[6].number = code;

  if (code != TCL_OK) {
    if ((argv[0] = interp -> result) == NULL) {
      argv[0] = "";
    }

    if ((argv[1] = Tcl_GetVar2 (interp, "errorCode", NULL, TCL_GLOBAL_ONLY)) == NULL) {
      argv[1] = "";
    }
 
    if ((argv[2] = Tcl_GetVar2 (interp, "errorInfo", NULL, TCL_GLOBAL_ONLY)) == NULL) {
      argv[2] = "";
    }

    mesg_to_root.elements[7].string  = Tcl_Merge (3, argv);
    mesg_to_root.elements[7].dealloc = TRUE; 
  } else {
    mesg_to_root.elements[7].string  = interp -> result;
  }

    /* send the result */

  while (1) {

    if ((sockfd = message_conn_and_send (NULL, root -> ip, mesg_to_root, -1, FALSE)) >= 0) {
      close (sockfd);
      return TCL_OK;
    }

    if (++i >= maxTries) {
      return TCL_ERROR;
    }

    sleep (sleepTime);

    if (sleepTime * 2 < maxSleep) {
      sleepTime *= 2;
    } else {
      sleepTime = maxSleep;
    }
  }
} 
            
/* Agent_Send

      Purpose: Send a message to an agent

        Input: interp  = the Tcl interpreter
               seconds - timeout interval
               id      = id of the destination agent
               code    = message code
               string  = message string
 
       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an error message if an error occurs.  Otherwise
               the procedure returns TCL_OK.
*/

int Agent_Send (Tcl_Interp *interp, double seconds, AGENT_ID *id, int code, char *string)
{
  clock_t stop;
  int sockfd = -1;              
  RESTRICT *restrict;
  AGENT_LOCATION *locations;
  int returnCode = TCL_ERROR;
  MESSAGE *mesg_from_server = NULL;     

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (id != NULL);
  assert (string != NULL);

    /* RESTRICT and AGENT_LOCATION */

  restrict = RESTRICT::get_restrict (interp);
  locations = AGENT_LOCATION::get_location (interp);
  assert (restrict != NULL);
  assert (locations != NULL);

    /* check whether the agent is registered */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has NOT been registered", (char *) NULL);
    return TCL_ERROR; 
  } else if (code < 0) {
    Tcl_AppendResult (interp, "message code must be 0 or greater", (char *) NULL);
    return TCL_ERROR;
  }

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);

    /* send the message */

  MESSAGE mesg_to_server (up_socket.messages[REQ_MESSAGE]);
  mesg_to_server.elements[0].string = locations -> local -> server; 
  mesg_to_server.elements[1].number = locations -> local -> ip;
  mesg_to_server.elements[2].string = locations -> local -> name;
  mesg_to_server.elements[3].number = locations -> local -> id;
  mesg_to_server.elements[4].string = id -> name;
  mesg_to_server.elements[5].number = id -> id;
  mesg_to_server.elements[6].number = code;
  mesg_to_server.elements[7].string = string;

  if ((sockfd = message_conn_and_send (id -> server, id -> ip, mesg_to_server, stop, FALSE)) < 0) {
    Tcl_AppendResult (interp, "unable to send to \"", id -> server, "\"", (char *) NULL);
  } else if ((mesg_from_server = message_receive (sockfd, down_socket)) == NULL) {
    Tcl_AppendResult (interp, "\"", id -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if (mesg_from_server -> flag != RESP_OK) {
    Tcl_AppendResult (interp, "\"", id -> server, "\" unable to comply (server error)", (char *) NULL);
  } else {
    returnCode = TCL_OK;
  }

    /* remove the restriction */

  restrict -> remove ();
 
    /* clean up */

  close_check (sockfd);
  delete_check (mesg_from_server);
  return returnCode;
}

/* Agent_Submit

      Purpose: Submit an agent to the server

        Input: interp        = the Tcl interpreter
               seconds       = timeout interval
               machine_id    = the destination machine
               language      = language name 
               numProcedures = number of procedures      
               procedures    = procedure names
               numVariables  = number of variables
               variables     = variable names
               script        = script 

       Output: The procedure returns NULL and sets the interpreter
               result to an error message if an error occurs.  Otherwise
               the procedure returns an AGENT_ID structure that contains
               the id of the submitted agent.
*/

AGENT_ID *Agent_Submit 
	(Tcl_Interp *interp, double seconds,
	MACHINE_ID *machine_id, char *language,
	int numProcedures, char **procedures, 
        int numVariables, char **variables,
	char *script)
{
  clock_t stop; 
  int sockfd = -1;
  Tcl_DString state;
  AGENT_ID *id = NULL; 
  RESTRICT *restrict;
  AGENT_LOCATION *locations;
  MESSAGE *mesg_from_server = NULL;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (machine_id != NULL);
  assert (((numProcedures == 0) && (procedures == NULL)) || ((numProcedures > 0) && (procedures != NULL)));
  assert (((numVariables  == 0) && (variables  == NULL)) || ((numVariables  > 0) && (variables  != NULL)));

    /* RESTRICT and AGENT_LOCATION structures */

  restrict  = RESTRICT::get_restrict (interp);
  locations = AGENT_LOCATION::get_location (interp);
  assert (locations != NULL);
  assert (restrict != NULL);

    /* check whether the agent is registered */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return NULL;
  } else if (script == NULL) {
    Tcl_AppendResult (interp, "must specify a script", (char *) NULL);
    return NULL;
  }
 
    /* use the native language if a language was not specified */

  if (language == NULL) {
    language = locations -> language;
  }

    /* set up the base message */

  MESSAGE mesg_to_server (up_socket.messages[REQ_SCRIPT]);
  mesg_to_server.elements[0].string = locations -> root -> server;
  mesg_to_server.elements[1].number = locations -> root -> ip;
  mesg_to_server.elements[2].string = locations -> root -> name;
  mesg_to_server.elements[3].number = locations -> root -> id;
  mesg_to_server.elements[4].string = locations -> local -> server;
  mesg_to_server.elements[5].number = locations -> local -> ip;
  mesg_to_server.elements[6].string = locations -> local -> name;
  mesg_to_server.elements[7].number = locations -> local -> id;
  mesg_to_server.elements[8].string = language;

    /* get the variables */

  Tcl_DStringInit (&state);

  if (numVariables != 0) {
    if (Tcl_SaveVariableList (interp, numVariables, variables, &state) != TCL_OK) {
      Tcl_AppendResult (interp, ": unable to save variables", (char *) NULL);
      Tcl_DStringFree (&state);
      return NULL;
    }
  }

    /* get the procedures */

  if (numProcedures != 0) {
    if (Tcl_SaveProcedureList (interp, numProcedures, procedures, &state) != TCL_OK) { 
      Tcl_AppendResult (interp, ": unable to save procedures", (char *) NULL);
      Tcl_DStringFree (&state);
      return NULL;
    }
  }

    /* add the script */

  Tcl_DStringAppend (&state, script, strlen(script));
  mesg_to_server.elements[9].string = Tcl_DStringValue (&state);

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);
    
    /* send the message */

  if ((sockfd = message_conn_and_send (machine_id -> server, machine_id -> server_ip, mesg_to_server, stop, FALSE)) < 0) {
    Tcl_AppendResult (interp, "unable to send to \"", machine_id -> server, "\"", (char *) NULL);
  } else if ((mesg_from_server = message_receive (sockfd, down_socket)) == NULL) {
    Tcl_AppendResult (interp, "\"", machine_id -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if (mesg_from_server -> flag != RESP_ID) {
    Tcl_AppendResult (interp, "\"", machine_id -> server, "\" unable to comply (server error)", (char *) NULL);
  } else {
    char   *server     = mesg_from_server -> elements[0].string;
    UINT_32 ip         = mesg_from_server -> elements[1].number;
    UINT_32 numeric_id = mesg_from_server -> elements[3].number;
    id                 = new AGENT_ID (server, ip, NULL, numeric_id);
  }

    /* remove the restriction */

  restrict -> remove ();

    /* cleanup */

  delete_check (mesg_from_server);
  Tcl_DStringFree (&state);
  close_check (sockfd);
  return id;
}

/* Agent_SplitId

   Purpose: Split apart a string that contains an agent identification

     Input: interp = Tcl interpreter
            string = the list

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns a pointer to a dynamically allocated AGENT_ID structure
            that contains the agent identification.
*/

AGENT_ID *Agent_SplitId (Tcl_Interp *interp, char *string)
{
  UINT_32 ip;               /* IP number                         */
  AGENT_ID *id = NULL;      /* agent identification              */
  int listArgc;             /* number of components in the id    */
  char **listArgv;          /* components of the id              */
  int number;               /* scratch area for numer conversion */

  if (Tcl_SplitList (interp, string, &listArgc, &listArgv) == TCL_OK)
  {
    if (listArgc == 2)
    {
      if (Tcpip_StringToIp (interp, listArgv[0], &ip) < 0)
        ip = UNKNOWN_IP;

      if (Tcl_GetInt (interp, listArgv[1], &number) != TCL_OK)
        id = new AGENT_ID (listArgv[0], ip, listArgv[1], 0); 
      else if (number >= 0)
        id = new AGENT_ID (listArgv[0], ip, NULL, number);

      Tcl_ResetResult (interp);
    }
    else if (listArgc == 4)
    {
      if (Tcpip_StringToIp (interp, listArgv[1], &ip) == 0)
      {
        if ((Tcl_GetInt (interp, listArgv[3], &number) == TCL_OK) && (number >= 0))
          id = new AGENT_ID (listArgv[0], ip, listArgv[2], number);
      }
    }
  }

  if (id == NULL)
    Tcl_AppendResult
      (interp, ": agent identification must be {server name}, {server id} or {server IP name id}", (char *) NULL);

  delete listArgv;
  return id;  
}

/* Agent_MachineToString

   Purpose: Convert a machine identification into a string

     Input: interp = Tcl interpreter
            id     = machine identification

    Output: The procedure returns NULL on error or a pointer to a dynamically
            allocated Tcl_DString.
*/

Tcl_DString *Agent_MachineToString (Tcl_Interp *interp, MACHINE_ID *id)
{
  Tcl_DString *string;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (id != NULL);

    /* initialize the Tcl_DString */

  string = new Tcl_DString;
  Tcl_DStringInit (string);

    /* add the machine namd and IP address */

  Tcl_DStringAppendElement (string, id -> server);

  if (id -> server_ip == UNKNOWN_IP) {
    Tcl_DStringAppendElement (string, "");
  } else {
    char *ip_string = Tcpip_IpToString (interp, id -> server_ip);
    Tcl_DStringAppendElement (string, ip_string);
    delete ip_string;  
  }

  return (string);
}

/* Agent_IdToString

   Purpose: Convert an agent identification into a string

     Input: interp = Tcl interpreter
            id     = agent identification

    Output: The procedure returns NULL on error or a pointer to a dynamically
            allocated Tcl_DString.
*/

Tcl_DString *Agent_IdToString (Tcl_Interp *interp, AGENT_ID *id)
{
  char temp[16];
  Tcl_DString *string = new Tcl_DString;
  Tcl_DStringInit (string);

  Tcl_DStringAppendElement (string, id -> server);

  if (id -> ip == UNKNOWN_IP) {
    Tcl_DStringAppendElement (string ,"");      
  } else {  
    char *ip_string = Tcpip_IpToString (interp, id -> ip);
    Tcl_DStringAppendElement (string, ip_string);
    delete ip_string;  
  }

  Tcl_DStringAppendElement (string, id -> name);
  sprintf (temp, "%d", id -> id);
  Tcl_DStringAppendElement (string, temp);

  return string;  
}

/* Agent_MeetingToString 

   Purpose: Convert a meeting status flag into a string

     Input: interp         = Tcl interpreter
            meeting_status = meeting status flag

    Output: The procedure returns NULL on error or a pointer to a dyanamically
            allocated Tcl_DString.
*/

Tcl_DString *Agent_MeetingToString (Tcl_Interp *interp, UINT_8 meeting_status)
{
  Tcl_DString *string = new Tcl_DString;

  Tcl_DStringInit (string);

  switch (meeting_status)
  {
    case MEET_REQUEST: Tcl_DStringAppendElement (string, "REQUEST");
                       break;

    case MEET_ACCEPT : Tcl_DStringAppendElement (string, "ACCEPT");
                       break;
 
    case MEET_CONNECT: Tcl_DStringAppendElement (string, "CONNECT");
                       break;

    case MEET_REFUSE : Tcl_DStringAppendElement (string, "REFUSED");
                       break;

    default          : tclFreeDString (string);
                       return NULL;
  }

  return string;
}
