/* Agent Tcl
   Bob Gray
   Sumit Chawla
   Saurab Nog 
   25 January 1995
  
   tclBasicCmd.cc

   This file implements the Tcl commands that handle migration, message
   passing and meetings.

   Copyright (c) 1995, Robert S. Gray, Sumit Chawla, Saurab Nog, 
   Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "interrupt.h"
#include <unistd.h>
#include "cmd_utilities.h"
#include "message.h"
#include "my_strings.h"
#include "tcl.h"
#include "tcpip.h"
#include "tclAgent.h"
#include "tclAgentInt.h"
#include "tclLocation.h"
#include "tclMask.h"
#include "tclTcpip.h"
#include "tclVersion.h"
#include "tcl_utilities.h"
#include "truefalse.h"

struct COMMAND_INFO
{
  char *name;
  Tcl_CmdProc *proc;
  int interruptable;    /* 1 if the command is interruptable   */
  int overwrite;        /* 1 if the command can be overwritten */
};

static COMMAND_INFO agent_commands[] = 
{
     /* commands that can be replaced */

  {"agent_begin"	, Agent_BeginCmd	, 0, 1},
  {"agent_name"		, Agent_NameCmd		, 0, 1},
  {"agent_end"		, Agent_EndCmd		, 0, 1},
  {"agent_send"		, Agent_SendCmd		, 0, 1},
  {"agent_receive"	, Agent_ReceiveCmd	, 0, 1},
  {"agent_disk"		, Agent_DiskCmd		, 1, 1}, 
  {"agent_jump"		, Agent_JumpCmd		, 1, 1},
  {"agent_submit"	, Agent_SubmitCmd 	, 0, 1},
  {"agent_fork"		, Agent_ForkCmd		, 1, 1},
  {"agent_req"          , Agent_ReqCmd          , 0, 1},
  {"agent_getreq"       , Agent_GetReqCmd       , 0, 1},
  {"agent_event"	, Agent_EventCmd	, 0, 1},
  {"agent_getevent"     , Agent_GetEventCmd     , 0, 1},
  {"agent_root"         , Agent_RootCmd         , 0, 1},
  {"agent_transfer"	, Agent_TransferCmd	, 0, 1},
#ifdef AGENT_TOOLS
  {"agent_force"	, Agent_ForceCmd	, 0, 1},
#endif

     /* equivalent commands that can not be replaced */

  {"_agent_begin"	, Agent_BeginCmd	, 0, 0},
  {"_agent_name"	, Agent_NameCmd		, 0, 0},
  {"_agent_end"		, Agent_EndCmd		, 0, 0},
  {"_agent_send"	, Agent_SendCmd		, 0, 0},
  {"_agent_receive"	, Agent_ReceiveCmd	, 0, 0},
  {"_agent_disk"	, Agent_DiskCmd		, 1, 0},
  {"_agent_jump"	, Agent_JumpCmd		, 1, 0},
  {"_agent_submit"	, Agent_SubmitCmd 	, 0, 0},
  {"_agent_fork"	, Agent_ForkCmd		, 1, 0},
  {"_agent_req"         , Agent_ReqCmd          , 0, 0},
  {"_agent_getreq"      , Agent_GetReqCmd       , 0, 0},
  {"_agent_event"	, Agent_EventCmd	, 0, 0},
  {"_agent_getevent"    , Agent_GetEventCmd     , 0, 0},
  {"_agent_root"        , Agent_RootCmd         , 0, 0},
  {"_agent_transfer"	, Agent_TransferCmd	, 0, 0},
#ifdef AGENT_TOOLS
  {"_agent_force"	, Agent_ForceCmd	, 0, 0},
#endif

     /* commands that can not be replaced */

  {"agent_select"       , Agent_SelectCmd       , 0, 0},
  {"agent_version"      , Agent_VersionCmd      , 0, 0},
  {"agent_sleep"	, Agent_SleepCmd	, 0, 0},
  {"agent_elapsed"	, Agent_ElapsedCmd      , 0, 0},
  {"agent_info"		, Agent_InfoCmd		, 0, 0},
  {"crypt"		, Agent_CryptCmd	, 0, 0},
  {"mask"		, Agent_MaskCmd		, 0, 0},
  {"exit"		, Agent_ExitCmd		, 0, 0},
  { NULL 		, (Tcl_CmdProc *) NULL	, 0, 0}
};

/* Agent_TransferCmd

   Tcl syntax: agent_tranfser <server> <filename> [ -time <seconds> ]

      Purpose: Transfer a server message from a disk file to a server

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and sets the interpreter result to the
               empty string.
*/

int Agent_TransferCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int i;
  double seconds;

    /* parse the arguments */

  if (argc < 3) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " <server> <filename> [ -time <seconds> ]\"", (char *) NULL);
    return TCL_ERROR;
  } else if (argc == 3) { 
    seconds = DEFAULT_CONNECT_SECONDS;
  } else if ((i = parseShortTimeout (interp, seconds, argc, argv, 4)) < 0) {
    return TCL_ERROR;
  } else if (++i < argc) {
    Tcl_AppendResult (interp, "extraneous arguments", (char *) NULL);
    return TCL_ERROR;
  }

     /* transfer the message */

  MACHINE_ID machine (argv[1], UNKNOWN_IP);
  return (Agent_Transfer (interp, seconds, &machine, argv[2]));
}

/* Agent_RootCmd

   Tcl syntax: agent_root

      Purpose: Turn the current agent into a root agent

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK.
*/

int Agent_RootCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
    /* check the number of command arguments */

  if (argc != 1)
  {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* turn the current agent into a root agent */

  return (Agent_Root (interp));
}

/* Agent_VersionCmd

   Tcl synatx: agent_version

      Purpose: Get the version information

        Input: None

       Output: The procedure returns TCL_OK and sets the interpreter result
               to the version information.
*/

int Agent_VersionCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
    /* check the number of arguments */

  if (argc != 1)
  {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

  Tcl_SetResult (interp, AGENT_VERSION, TCL_STATIC); 
  return TCL_OK;
}

/* Agent_SelectCmd

      Authors: Sumit Chawla, Saurab Nog, 5 November 1995
      Revised: Bob Gray, 12 November 1994
	
   Tcl syntax: agent_select <list of file descriptors>
	       [-nonblocking | -time <seconds> | -blocking]

      Purpose: Wait for one or more file descriptors to become ready for
               reading.  There are three special file descriptors --
               "message", "meeting" and "event".  These file descriptors
               refer to incoming messages, meeting requests and events from
               other agents.

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
               to a list of the file descriptors that are ready for reading.
*/

int Agent_SelectCmd (ClientData clientData,Tcl_Interp *interp,int argc,char ** argv )
{
  double seconds;                 /* timeout interval                   */
  int listArgc = 0;               /* number of descriptors in the list  */
  int error = TCL_ERROR;          /* return value                       */
  char **listArgv = NULL;         /* the list of descriptors            */
  char temp[16];
  int i;
  int * result_array = NULL;      /* whether each descriptor is ready   */
  int * descriptor_array = NULL;  /* descriptors of interest            */

    /* check the number of arguments */

  if (argc < 2) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " <list of file descriptors> [-nonblocking | -time seconds | -blocking]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* parse the timeout */

  if (argc == 2) {
    seconds = -1.0;
  } else if ((i = parseTimeout (interp, seconds, argc, argv, 2)) < 0) {
    return TCL_ERROR;
  } else if (i < argc) {
    Tcl_AppendResult (interp, "extraneous arguments", (char *) NULL);
    return TCL_ERROR;
  }

    /* split apart the list of file descriptors */

  if ((Tcl_SplitList (interp,argv[1], &listArgc, &listArgv) != TCL_OK)) {
    Tcl_AppendResult (interp, "must be a valid Tcl list", (char *) NULL);
    return TCL_ERROR;
  }

    /* call Agent_Select */
 
  descriptor_array = new int [listArgc];
  result_array     = new int [listArgc];

  for(i = 0; i < listArgc; i++) {
    if (strcmp(listArgv[i],"message") == 0) {
      descriptor_array[i] = MESSAGE_FD;
    } else if (strcmp(listArgv[i],"meeting") == 0) {
      descriptor_array[i] = MEETING_FD;
    } else if (strcmp(listArgv[i],"event") == 0) {
      descriptor_array[i] = EVENT_FD;
    } else if (Tcl_GetInt(interp, listArgv[i], &descriptor_array[i]) != TCL_OK) {
      Tcl_AppendResult (interp, ": file descriptor must be \"meeting\", \"message\", \"event\" or integer", TCL_STATIC);
      goto cleanup;
    } else if (descriptor_array[i] < 0) {
      Tcl_AppendResult (interp, "negative file descriptor \"", listArgv[i], "\"", (char *) NULL);
      goto cleanup;
    }
  }
		
  error = Agent_Select(interp,seconds,listArgc,descriptor_array,result_array);

    /* assemble the list of ready descriptors */
	
  if (error == TCL_OK) {

    Tcl_ResetResult (interp);	

    for(i = 0; i < listArgc; i++) {
      if (result_array[i] != 0) {
        if (descriptor_array[i] == MESSAGE_FD) {
          Tcl_AppendElement (interp, "message");
        } else if (descriptor_array[i] == MEETING_FD) {
          Tcl_AppendElement (interp, "meeting");
        } else if (descriptor_array[i] == EVENT_FD) {
          Tcl_AppendElement (interp, "event");
        } else {
          sprintf (temp, "%d", descriptor_array[i]);
          Tcl_AppendElement (interp, temp);
        }
      }  
    }
  }

cleanup:

  delete_check (descriptor_array);
  delete_check (result_array);
  delete_check (listArgv);
  return (error);
}

/* Agent_BeginCmd

   Tcl syntax: agent_begin ?machine? ?-time seconds?

      Purpose: Notify the machine that we wish to become an agent
               under its control

        Input: clientData = client data (unused)
               interp     = the current interpreter
               argc       = number of command arguments
               argv       = the command arguments

       Output: The procedure returns TCL_ERROR on error.  Otherwise the
               procedure returns TCL_OK and sets the agent(home),
               agent(home-id), agent(local) and agent(local-id) variables
               to the appropriate value. 
*/

int Agent_BeginCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{ 
  AGENT_ID *id;
  Tcl_DString *string;
  MACHINE_ID *machine_id;
  double seconds = DEFAULT_CONNECT_SECONDS;

    /* get the timeout if specified */

  if (argc == 1) {
    machine_id = new MACHINE_ID (NULL, UNKNOWN_IP);
  } else if (argc == 2) {
    machine_id = new MACHINE_ID (argv[1], UNKNOWN_IP); 
  } else if (argc == 3) {
    if (parseShortTimeout (interp, seconds, argc, argv, 1) < 0) {
      return TCL_ERROR;
    } else {
      machine_id = new MACHINE_ID (NULL, UNKNOWN_IP);
    }
  } else if (argc == 4) {
    if (parseShortTimeout (interp, seconds, argc, argv, 2) < 0) {
      return TCL_ERROR;
    } else {
      machine_id = new MACHINE_ID (argv[1], UNKNOWN_IP);
    }
  } else {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " ?machine? ?-time seconds?\"", (char *) NULL);
    return TCL_ERROR;
  }
 
    /* begin the agent */

  if ((id = Agent_Begin (interp, seconds, machine_id)) == NULL) {
    delete machine_id;
    return TCL_ERROR;
  }

    /* convert the id to a string */

  if ((string = Agent_IdToString (interp, id)) == NULL) {
    delete machine_id;
    delete id;
    return TCL_ERROR;
  }
 
    /* put the string into the result */

  Tcl_DStringResult (interp, string);
  delete string;
  delete machine_id;
  delete id;
  return TCL_OK;    
} 

/* Agent_NameCmd 

   Tcl syntax: agent_name string [-time seconds]

      Purpose: Assign a descriptive name to the agent

        Input: clientData = client data (unused)
               interp     = the current interpreter
               argc       = number of command arguments
               argv       = the command arguments
        
       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and sets the intepreter result to the
               new 4-element identification of the agent. 
*/

int Agent_NameCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int returnCode;
  AGENT_ID *id = NULL;
  Tcl_DString *string = NULL;
  double seconds = DEFAULT_CONNECT_SECONDS;

    /* check the number of arguments */
 
  if (argc < 2) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"agent_name string ?-time seconds?\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the timeout if it is specified */

  if (argc > 2) {
    if (parseShortTimeout (interp, seconds, argc, argv, 2) < 0) {
      return TCL_ERROR;
    }
  }

    /* assign the name */

  if ((id = Agent_Name (interp, seconds, argv[1])) == NULL) {
    returnCode = TCL_ERROR;
  } else if ((string = Agent_IdToString (interp, id)) == NULL) {
    returnCode = TCL_ERROR;
  } else  {
    Tcl_DStringResult (interp, string);
    returnCode =  TCL_OK;
  }

    /* cleanup */

  delete_check (id);
  tclFreeDString (string);
  return (returnCode); 
}

/* Agent_EndCmd

   Tcl syntax: agent_end ?-time seconds? 

      Purpose: Notify the controlling machine that we are no longer an agent

        Input: clientData = client data (unused)
               interp     = the current interpreter
               argc       = number of command arguments
               argv       = the command arguments

       Output: 
*/

int Agent_EndCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  double seconds = DEFAULT_CONNECT_SECONDS;

    /* check the number of arguments */

  if (argc == 3) {
    if (parseShortTimeout (interp, seconds, argc, argv, 1) < 0) {
      return TCL_ERROR;
    }
  } else if (argc != 1) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"agent_end ?-time seconds?\"", (char *) NULL);
    return TCL_ERROR;
  }
 
    /* end the agent */
  
  return (Agent_End (interp, seconds));
}

/* Agent_EventCmd
 
   Tcl syntax: agent_event {destination} tag string [-time seconds]

      Purpose: Send an event to an agent

        Input: clientData = client data (unused)
               interp     = the current interpreter
               argc       = number of command arguments
               argv       = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the  
               procedure returns TCL_OK and sets the interpreter result to the
               empty string.
*/

int Agent_EventCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  AGENT_ID *id = NULL;
  int returnCode = TCL_ERROR;
  double seconds = DEFAULT_CONNECT_SECONDS;

     /* check the number of arguments */

  if (argc < 4) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " {destination} tag string ?-time seconds?", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the agent identification */

  if ((id = Agent_SplitId (interp, argv[1])) == NULL) {
    return TCL_ERROR;
  }

    /* get the timout if one is specified */

  if (argc > 4) {
    if (parseShortTimeout (interp, seconds, argc, argv, 4) < 0) {
      goto cleanup;
    } else if (argc > 6) {
      Tcl_AppendResult (interp, "extraneous arguments", (char *) NULL);
      goto cleanup;
    } 
  }

    /* send the event */

  returnCode = Agent_Event (interp, seconds, id, argv[2], argv[3]);

cleanup:

  delete_check (id);
  return (returnCode);
}

/* Agent_SendCmd

   Tcl syntax: agent_send {destination} [integer] string [-time seconds]  

      Purpose: Send a result or a message to an agent

        Input: clientData = client data (unused)
               interp     = the current interpreter
               argc       = number of command arguments
               argv       = the command arguments

       Output: The procedure returns TCL_ERROR on error.  Otherwise the
               procedure sends the string to the specified agent and 
               returns TCL_OK.
*/

int Agent_SendCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int code = 0;                
  char *string = NULL;
  AGENT_ID *id = NULL;
  int returnCode = TCL_ERROR;
  double seconds = DEFAULT_CONNECT_SECONDS;

    /* check the number of arguments */

  if (argc < 3) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " {destination} [integer] string [-time seconds]\"", (char *) NULL);
    goto cleanup;
  }

    /* split out the agent identification  */

  argc -= 1;
  argv += 1; 

  if ((id = Agent_SplitId (interp, argv[0])) == NULL) {
    goto cleanup;
  }

    /* get the message code if it is there */

  argc -= 1;
  argv += 1; 

  if ((argc == 2) || (argc == 4)) {
    if (Tcl_GetInt (interp, argv[0], &code) != TCL_OK) {
      Tcl_AppendResult (interp, "message code must be an integer", (char *) NULL);
      goto cleanup;
    } else {
      argc -= 1;
      argv += 1;
    }
  }

    /* make sure that we have a message string */

  if (argc == 0) {
    Tcl_AppendResult (interp, "must specify a message string", (char *) NULL);
    goto cleanup;
  } else {
    string = argv[0];
    argc -= 1;
    argv += 1;
  }

    /* get the timeout interval */

  if (argc == 0) {
    seconds = DEFAULT_CONNECT_SECONDS;
  } else if (parseShortTimeout (interp, seconds, argc, argv, 0) < 0) {
    goto cleanup;
  } else {
    argc -= 2;
    argv += 2;
  }

    /* check for extraneous arguments */

  if (argc != 0) {
    Tcl_AppendResult (interp, "extraneous arguments", (char *) NULL);
    goto cleanup;
  } 
 
    /* send the message */

  returnCode = Agent_Send (interp, seconds, id, code, string);

cleanup:

  delete_check (id);
  return returnCode;
}

/* Agent_DiskCmd

   Tcl syntax: agent_disk

      Purpose: Save the current state of the agent (as if the agent were
               about to jump) to a disk file.  The disk file can be 
               transferred to a server later with the "agent_transfer"
               command.

        Input: clientData = client data (unused)
               interp     = the current interpreter
               argc       = the number of command arguments
               argv       = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns (1) the name of the file if the agent
               has just been saved to disk or (2) "JUMPED" if the disk
               file has been transferred to a server and the agent has been
               restored from disk.
*/

int Agent_DiskCmd (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
  char *filename;

    /* check the number of command arguments */

  if (argc != 1) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* save the state to disk */

  if ((filename = Disk_Jump (interp)) == NULL) {
    return TCL_ERROR;
  }

  Tcl_SetResult (interp, filename, TCL_DYNAMIC);
  return TCL_OK;
}
        
/* Agent_JumpCmd

   TCL Syntax: agent_jump machine [-time seconds]

      Purpose: Jump to another machine

        Input: clientData = client data (unused)
               interp     = the current interpreter
               argc       = number of command arguments
               argv       = the command arguments 

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message if it is unable to 
               jump to the remote machine.  The procedure returns TCL_OK
               and sets the interpreter result to "SAME" if the remote
               machine is the same as the current machine.  Otheriwse the  
               procedure calls exit() to terminate the program since the 
               agent has successfully transported to a different machine.
               The agent will resume execution on this machine with the
               interpreter result set to "JUMPED".
*/

int Agent_JumpCmd (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
  double seconds;

    /* check the number of arguments */

  if (argc == 2) {
    seconds = DEFAULT_CONNECT_SECONDS;
  } else if (argc != 4) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " machine [-time seconds]\"", (char *) NULL);
    return TCL_ERROR;
  } else if (parseShortTimeout (interp, seconds, argc, argv, 2) < 0) {
    return TCL_ERROR;
  } 

    /* jump */

  MACHINE_ID machine_id (argv[1], UNKNOWN_IP);
  return (Agent_Jump (interp, seconds, &machine_id));
}

/* Agent_ForkCmd

   TCL syntax: agent_fork machine [-time seconds]

      Purpose: Fork a copy of the current agent

        Input: clientData = client data (unused)
               interp     = Tcl interpreter
               argc       = number of arguments
               argv       = the arguments   
 
       Output: The procedure returns TCL_ERROR if it is unable to fork
               the agent.  Otherwise the procedure returns TCL_OK and 
               sets interp -> result as follows: result for the parent agent
               will be set to the 4-element identification vector of the child
               agent; result for the child agent will be set to "CHILD".
*/

int Agent_ForkCmd (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int code;
  double seconds;
  AGENT_ID *id = NULL;
  Tcl_DString *string = NULL;

    /* check the arguments */

  if (argc == 2) {
    seconds = DEFAULT_CONNECT_SECONDS;
  } else if (argc != 4) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " machine [-time seconds]\"", (char *) NULL);
    return TCL_ERROR;
  } else if (parseShortTimeout (interp, seconds, argc, argv, 2) < 0) {
    return TCL_ERROR;
  }

    /* fork the agent */

  MACHINE_ID machine_id (argv[1], UNKNOWN_IP);
  
  if ((id = Agent_Fork (interp, seconds, &machine_id)) == NULL) {
    code = TCL_ERROR;
  } else if ((string = Agent_IdToString (interp, id)) == NULL) {
    code = TCL_ERROR;
  } else {
    Tcl_DStringResult (interp, string);
    code = TCL_OK;
  }

    /* clean up */

  delete_check (id);
  tclFreeDString (string);
  return (code);
}

/* Agent_SubmitCmd

   Tcl syntax: agent_submit machine 
			[-time seconds]
			[-language language]
			[-vars varname ...]
			[-procs procname ...]
			-script script
  
      Purpose: Submit an agent to the server

        Input: clientData = client data (unused)
               interp     = Tcl interpreter
               argc       = number of arguments
               argv       = arguments

       Output: The procedure returns TCL_ERROR on error.  Otherwise the 
               procedure returns TCL_OK.
*/

int find_list_end (int i, int argc, char **argv)
{
  int count = 0;

  for (i++; i < argc; i++) {
    if (strcmp(argv[i], "-end") && strcmp(argv[i], "-vars") && strcmp(argv[i], "-procs") && strcmp(argv[i], "-script") && strcmp(argv[i],"-language")) {
      count++;
    } else {
      break; 
    }
  }

  return count;
}

int Agent_SubmitCmd (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{ 
  int i;
  AGENT_ID *id;
  Tcl_DString *string;
  MACHINE_ID *machine_id;
  double seconds = DEFAULT_CONNECT_SECONDS;

    /* defaults */

  char **variables   = NULL;
  char **procedures  = NULL;
  char *script       = NULL;
  char *language     = NULL;
  int numVariables   = 0;
  int numProcedures  = 0;

    /* check the number of arguments */

  if (argc <= 3) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " machine [-time seconds] [-language language] [-vars varname ...] [-procs procname ...] -script script\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the arguments */

  for (i = 2; i < argc; i++) {

    if (!strcmp(argv[i], "-time")) {

      if ((i = parseShortTimeout (interp, seconds, argc, argv, i)) < 0) {
        return TCL_ERROR;
      }

    } else if (!strcmp(argv[i], "-vars")) {

      if (variables) {
        Tcl_AppendResult (interp, "can not have more than one variable list", (char *) NULL);
        return TCL_ERROR;
      } else if ((numVariables = find_list_end (i, argc, argv)) == 0) {
        Tcl_AppendResult (interp, "\"-vars\" must be followed by at least one variable name", (char *) NULL);
        return TCL_ERROR;
      }

      variables = &argv[i] + 1; 
      i        += numVariables;

    } else if (!strcmp(argv[i], "-procs")) {

      if (procedures) {
        Tcl_AppendResult (interp, "can not have more than one procedure list", (char *) NULL);
        return TCL_ERROR;
      } else if ((numProcedures  = find_list_end (i, argc, argv)) == 0) {
        Tcl_AppendResult (interp, "\"-procs\" must be followed by at least one procedure name", (char *) NULL);
        return TCL_ERROR;
      }

      procedures = &argv[i] + 1;
      i         += numProcedures;
 
    } else if (!strcmp(argv[i],"-language")) {

      if (language) {
        Tcl_AppendResult (interp, "can not specify more than one language", (char *) NULL);
        return TCL_ERROR;
      } else if (++i == argc) {
        Tcl_AppendResult (interp, "\"-language\" must be followed by a language name", (char *) NULL);
        return TCL_ERROR;
      }

      language = argv[i];

    } else if (!strcmp(argv[i], "-script")) {

      if (script) {
        Tcl_AppendResult (interp, "can not have more than one script", (char *) NULL);
        return TCL_ERROR;
      } else if (++i >= argc) {
        Tcl_AppendResult (interp, ": need a script after \"-script\"", (char *) NULL);
        return TCL_ERROR; 
      }

      script = argv[i];

    } else {

      Tcl_AppendResult (interp, "unknown option \"", argv[i], "\"", (char *) NULL);
      return TCL_ERROR;
    }
  }

    /* submit the agent */

  machine_id = new MACHINE_ID (argv[1], UNKNOWN_IP);

  if ((id = Agent_Submit (interp, seconds, machine_id, language, numProcedures, procedures, numVariables, variables, script)) == NULL) {
    return TCL_ERROR;
  }

    /* return the id of the new agent */

  if ((string = Agent_IdToString (interp, id)) == NULL) {
    delete id;
    return TCL_ERROR;
  }

  Tcl_DStringResult (interp, string);
  delete string;
  delete id;
  return TCL_OK; 
}         

/* Agent_GetEventCmd

   Tcl syntax: agent_getevent tag string <-blocking | -time seconds | -nonblocking>

      Purpose: Check to see if any events are available for the agent

        Input: clientData = client data (unused)
               interp     = Tcl interpreter
               argc       = number of arguments
               argv       = the arguments

       Output: The procedure returns TCL_ERROR on error.  The procedure 
               returns TCL_OK and sets the interpreter result to -1 if no 
	       event arrives during the timeout interval.  Otherwise the 
               procedure returns TCL_OK, sets the interpreter result to 
               the sender identification, sets the "tag" variable to the
               event tag and sets the "string" variable to the event string.                
*/

int Agent_GetEventCmd (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int i;
  double seconds;                 /* number of seconds for time out */
  int returnCode;                 /* return code from the procedure */
  EVENT_EXT *event;               /* an event from an agent         */
  Tcl_DString *id_string = NULL;  /* agent identification string    */

    /* check the number of arguments and agent registration */

  if ((argc != 4) && (argc != 5)) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " tag string <-blocking | -time seconds | -nonblocking>\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* determine if the call is BLOCKING, NONBLOCKING or TIMED */

  if ((i = parseTimeout (interp, seconds, argc, argv, 3)) < 0) {
    return TCL_ERROR; 
  } else if (i < argc) {
    Tcl_AppendResult (interp, "extraneous argument after ", argv[3], (char *) NULL);
    return TCL_ERROR;
  }
  
    /* receive the message */
 
  if (Agent_GetEvent (interp, seconds, &event) == TCL_ERROR) {
    return TCL_ERROR; 
  } else if (event == NULL) {
    Tcl_SetResult (interp, "-1", TCL_STATIC);
    return TCL_OK;
  }

    /* set the interpreter result and the variables */

  if (Tcl_SetVar (interp, argv[1], event -> tag, 0) == NULL) {
    Tcl_AppendResult (interp, ": unable to set tag variable", (char *) NULL);
    returnCode = TCL_ERROR;
  } else if (Tcl_SetVar (interp, argv[2], event -> string, 0) == NULL) {
    Tcl_AppendResult (interp, ": unable to set string variable", (char *) NULL); 
    returnCode = TCL_ERROR;
  } else if ((id_string = Agent_IdToString (interp, event -> id)) == NULL) {
    returnCode = TCL_ERROR;
  } else {
    Tcl_SetResult (interp, Tcl_DStringValue (id_string), TCL_VOLATILE); 
    returnCode = TCL_OK;
  }

    /* clean up */

  delete event;
  tclFreeDString (id_string);
  return returnCode;
}

/* Agent_ReceiveCmd

   Tcl syntax: agent_receive code value <-blocking | -time seconds | -nonblocking>

      Purpose: Check to see if any results are available for the agent

        Input: clientData = client data (unused)
               interp     = Tcl interpreter
               argc       = number of arguments
               argv       = the arguments

       Output: The procedure returns TCL_ERROR on error.  The procedure
               sets interp -> result to -1 if no result is available.
               otherwise the procedure sets interp -> result to the Tcl
               result code and sets the variable indicated by varname to
               the Tcl result string.
*/

int Agent_ReceiveCmd (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int i;
  char temp[16];                    /* scratch area for message code  */
  double seconds;                   /* number of seconds for time out */
  int return_code;                  /* return code from the procedure */
  Tcl_DString *id_string = NULL;    /* agent identification string    */
  MESSAGE_EXT *message;             /* a message from an agent        */

    /* check the number of arguments and agent registration */

  if ((argc != 4) && (argc != 5)) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " code value <-blocking | -time seconds | -nonblocking>\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* determine if the call is BLOCKING, NONBLOCKING or TIMED */

  if ((i = parseTimeout (interp, seconds, argc, argv, 3)) < 0) {
    return TCL_ERROR; 
  } else if (i < argc) {
    Tcl_AppendResult (interp, "extraneous argument after ", argv[3], (char *) NULL);
    return TCL_ERROR;
  }
  
    /* receive the message */
 
  if (Agent_Receive (interp, seconds, &message) != TCL_OK) {
    return TCL_ERROR; 
  } else if (message == NULL) {
    Tcl_SetResult (interp, "-1", TCL_STATIC);
    return TCL_OK;
  }

    /* set the interpreter result and the variables */

  sprintf (temp, "%d", message -> code);

  if (Tcl_SetVar (interp, argv[1], temp, 0) == NULL) {
    Tcl_AppendResult (interp, ": unable to set code variable", (char *) NULL);
    return_code = TCL_ERROR;
  } else if (Tcl_SetVar (interp, argv[2], message -> string, 0) == NULL) {
    Tcl_AppendResult (interp, ": unable to set string variable", (char *) NULL); 
    return_code = TCL_ERROR; 
  } else if ((id_string = Agent_IdToString (interp, message -> id)) == NULL) {
    return_code = TCL_ERROR;
  } else {
    Tcl_SetResult (interp, Tcl_DStringValue (id_string), TCL_VOLATILE); 
    return_code = TCL_OK;
  }

    /* clean up */

  tclFreeDString (id_string);
  delete message;
  return return_code;
}

/* Agent_ReqCmd

   Tcl syntax: agent_req {destination} [-time seconds] <-request | -accept | -reject | -connect port>

      Purpose: Request a meeting with another agent

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = number of command arguments
               argv   = the command arguments
 
       Output: The procedure returns TCL_ERROR on error.  Otherwise the
               procedure sends the meeting request and returns TCL_OK.
*/

int Agent_ReqCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int port = 0;            
  AGENT_ID *id = NULL;      
  UINT_8 meeting_status;   
  int returnCode = TCL_ERROR; 
  double seconds = DEFAULT_CONNECT_SECONDS;

    /* check the number of arguments */

  if (argc < 3) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], "\" {destination} [-time seconds] <-accept | -request | -reject | -connect port>\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* split out the agent identification */

  if ((id = Agent_SplitId (interp, argv[1])) == NULL) {
    return TCL_ERROR;
  }

    /* get the timeout if one is specified */

  argc -= 2;
  argv += 2;

  if (!strcmp(argv[0], "-time")) {
    if (parseShortTimeout (interp, seconds, argc, argv, 0) < 0) {
      goto cleanup;
    } else {
      argc -= 2;
      argv += 2;
    }
  }

    /* break out the meeting flag */

  if (argc == 0) {
    Tcl_AppendResult (interp, "must specify the meeting status", (char *) NULL);
    goto cleanup;
  } else if (!strcmp(argv[0], "-connect")) {
    meeting_status = MEET_CONNECT;
  } else if (!strcmp(argv[0], "-accept")) {
    meeting_status = MEET_ACCEPT;
  } else if (!strcmp(argv[0], "-refuse")) {
    meeting_status = MEET_REFUSE;
  } else if (!strcmp(argv[0], "-request")) {
    meeting_status = MEET_REQUEST;
  } else {
    Tcl_AppendResult (interp, "unknown option \"", argv[2], "\": should be -connect, -accept, -refuse or -request", (char *) NULL);
    goto cleanup;
  }  

    /* get the port number if necessary */

  argc -= 1;
  argv += 1;

  if (meeting_status == MEET_CONNECT) {
    if (argc == 0) {
      Tcl_AppendResult (interp, "-connect must be followed by a port number", (char *) NULL);
      goto cleanup; 
    } else if ((Tcl_GetInt (interp, argv[0], &port) != TCL_OK) || (port < 0)) {
      Tcl_SetResult (interp, "port number must be 0 or greater", TCL_STATIC);
      goto cleanup; 
    } else {
      argc -= 1;
      argv += 1;
    }
  } 

    /* make sure that we do not have extraneous arguments */

  if (argc != 0) {
    Tcl_AppendResult (interp, "extraneous arguments", (char *) NULL);
    goto cleanup; 
  }

    /* send the meeting request */

  returnCode = Agent_Req (interp, seconds, id, meeting_status, port);

cleanup:

  delete_check (id);
  return (returnCode);
}

/* Agent_GetReqCmd 

   Tcl syntax: agent_getreq status port <-blocking | -time seconds | -nonblocking>

      Purpose: Get a meeting request

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR on error.  The procedure
               returns TCL_OK and sets the interpreter result to -1 if there
               are no pending meeting requests for the agent.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
               to the three-element list {server name id} which specifies
               the identity of the agent that is requesting the meeting.  In
               addition the procedure sets the status and port variables.
*/

int Agent_GetReqCmd  (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int i;
  double seconds;                     /* number of seconds for timeout    */
  Tcl_DString *status_string = NULL;  /* string to contain meeting status */
  Tcl_DString *port_string = NULL;    /* string to contain meeting port   */
  Tcl_DString *id_string = NULL;      /* string to contain agent id       */
  MEETING_EXT *meeting;               /* meeting information              */
  int returnCode;                     /* return code from the procedure   */ 

    /* check the number of arguments */

  if ((argc != 4) && (argc != 5)) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " status port <-blocking | -nonblocking> [-time seconds]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* determine if the call is BLOCKING, NONBLOCKING or TIMED */

  if ((i = parseTimeout (interp, seconds, argc, argv, 3)) < 0) {
    return TCL_ERROR;
  } else if (i < argc) {
    Tcl_AppendResult (interp, "extraneous argument after ", argv[3], (char *) NULL);
    return TCL_ERROR;
  }  
  
    /* receive the meeting request */
    
  if (Agent_GetReq (interp, seconds, &meeting) != TCL_OK) {
    return TCL_ERROR;
  } else if (meeting == NULL) {
    Tcl_SetResult (interp, "-1", TCL_STATIC);
    return TCL_OK; 
  }
 
    /* convert the agent identification and the meeting status to a string */
  
  if ((id_string = Agent_IdToString (interp, meeting -> id)) == NULL) {
    returnCode = TCL_ERROR;
  } else if ((status_string = Agent_MeetingToString (interp, meeting -> status)) == NULL) {
    returnCode = TCL_ERROR;
  } else if ((port_string = Tcpip_PortToString (interp, meeting -> port)) == NULL) {
    returnCode = TCL_ERROR;
  } else if (Tcl_SetVar (interp, argv[2], Tcl_DStringValue (port_string), 0) == NULL) {
    Tcl_AppendResult (interp, ": unable to set port variable", (char *) NULL);
    returnCode = TCL_ERROR;
  } else if (Tcl_SetVar (interp, argv[1], Tcl_DStringValue (status_string), 0) == NULL) {
    Tcl_AppendResult (interp, ": unable to set status variable", (char *) NULL);
    returnCode = TCL_ERROR;
  } else {
    Tcl_SetResult (interp, Tcl_DStringValue(id_string), TCL_VOLATILE);
    returnCode = TCL_OK;
  }

    /* clean up */

  tclFreeDString (id_string);
  tclFreeDString (status_string);
  tclFreeDString (port_string);
  return returnCode;
}

/* Agent_ExitCmd

   Tcl syntax: exit [code] 

      Purpose: Redefines the "exit" command to call agentForceEnd if necessary

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = number of command arguments
               argv   = the command argumnets

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and calls agentForceEnd if necessary.
*/

int Agent_ExitCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int exitCode;
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);

    /* check the number of arguments */

  if ((argc != 1) && (argc != 2)) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " [code]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* exit code is 0 if not specified */
 
  if (argc == 1) {
    exitCode = 0;
  } else if (Tcl_GetInt(interp, argv[1], &exitCode) != TCL_OK) {
    return TCL_ERROR;
  }
 
    /* send a TRAN_END if necessary */

  if (locations -> registered == AGENT_REGISTERED) {
    agentForceEnd (interp, DEFAULT_CONNECT_SECONDS, locations -> local, NULL, END_NORMAL);
  }

    /* exit */

  exit(exitCode);
  return TCL_OK;   /* Bettter not ever reach this! */
}

/* ArrivalTrap

   Purpose: This procedure is called whenever there is an incoming item AND
            there is an interrupt handler for that item in the mask AND the
            Tcl interpreter is in a safe state.  The procedure executes the
            interrupt handler.

     Input: clientData = MASK_SET structure

    Output: The procedure executes the interrupt handler and then returns
            the Tcl result code.  
*/

static int ArrivalTrap (Tcl_Interp *interp, MASK_SET *maskSet)
{
  int code;
  INCOMING *item;
  char scratch[32];
  Tcl_DString script;
  Tcl_DString arguments;
  MaskArrivalData *data = NULL;

    /* get the first event in the queue */

  if ((data = maskSet -> get_event (data)) == NULL) {
    return TCL_OK;
  } 

    /* construct the argument list */

  item = data -> item;
  Tcl_DStringInit (&arguments);

    /* first the identifiction of the sender */

  Tcl_DString *idString = Agent_IdToString (interp, item -> id);
  Tcl_DStringAppendElement (&arguments, Tcl_DStringValue(idString));
  tclFreeDString (idString);

    /* now different things depending on the item type */ 
 
  if (item -> type == MESSAGE_ITEM) {
    INCOMING_MESSAGE *message = (INCOMING_MESSAGE *) item;
    sprintf (scratch, "%d", message -> code);
    Tcl_DStringAppendElement (&arguments, scratch);
    Tcl_DStringAppendElement (&arguments, message -> string);
  } else if (item -> type == EVENT_ITEM) { 
    INCOMING_EVENT *event = (INCOMING_EVENT *) item;
    Tcl_DStringAppendElement (&arguments, event -> tag);
    Tcl_DStringAppendElement (&arguments, event -> string);
  } else {
    INCOMING_MEETING *meeting = (INCOMING_MEETING *) item;
    Tcl_DString *locString = Tcpip_PortToString (interp, meeting -> port);
    Tcl_DString *staString = Agent_MeetingToString (interp, meeting -> status);
    Tcl_DStringAppendElement (&arguments, Tcl_DStringValue (locString));
    Tcl_DStringAppendElement (&arguments, Tcl_DStringValue (staString));
    tclFreeDString (locString);
    tclFreeDString (staString);
  }

    /* save result, errorCode and errorInfo */

  char *resultSave = my_strcpy (interp -> result);
  char *codeSave   = my_strcpy (Tcl_GetVar2 (interp, "errorCode", NULL, TCL_GLOBAL_ONLY));
  char *infoSave   = my_strcpy (Tcl_GetVar2 (interp, "errorInfo", NULL, TCL_GLOBAL_ONLY));

    /* fire off the interrupt handlers */

  do
  {
      /* first construct the appropriate script */

    Tcl_DStringInit (&script);
    Tcl_DStringAppend (&script, data -> handler, -1);
    Tcl_DStringAppend (&script, " ", -1);
    Tcl_DStringAppend (&script, Tcl_DStringValue(&arguments), -1);

      /* execute the script */

    code = Tcl_GlobalEval (interp, Tcl_DStringValue(&script));
    Tcl_DStringFree (&script);

      /* break on error */

    if (code != TCL_OK) {
      Tcl_AddErrorInfo (interp, "\n    (handler for an incoming item)");
      Tcl_Eval (interp, "tkerror \"interrupt handler failed\"");
      break;
    }

  } while ((data = maskSet -> get_event (data)) != NULL);

    /* clean up (including unprocessed handlers for the item) */

  while (data != NULL) {
    data = maskSet -> get_event (data);
  }

    /* restore result, errorCode and errorInfo */

  Tcl_SetResult (interp, resultSave, TCL_VOLATILE);
  delete (resultSave);
 
  if (codeSave) {
    Tcl_SetVar2 (interp, "errorCode", NULL, codeSave, TCL_GLOBAL_ONLY);
    delete (codeSave);
  } else {
    Tcl_UnsetVar2 (interp, "errorCode", NULL, TCL_GLOBAL_ONLY);
  }

  if (infoSave) {
    Tcl_SetVar2 (interp, "errorInfo", NULL, infoSave, TCL_GLOBAL_ONLY);
    delete (infoSave);
  } else {
    Tcl_UnsetVar2 (interp, "errorInfo", NULL, TCL_GLOBAL_ONLY);
  }
 
  Tcl_DStringFree (&arguments);
  return (code);
}

/* Agent_Init

   Purpose: Initialize the Agent TCL package

     Input: interp   = the current interpreter
            language = symbolic name for the language
            server   = TRUE if the agent arrived via the server
 
    Output: The procedure returns TCL_ERROR on initialization failure.
            The procedure returns TCL_OK    on initialization success.
*/
   
static void agentCleanup (ClientData clientData, Tcl_Interp *interp)
{
  MASK_SET *mask_set = MASK_SET::get_mask_set (interp);
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);
  delete_check (mask_set);
  delete_check (locations);
}

int Agent_Init (Tcl_Interp *interp, char *language, int server)
{
  char *hostname;
  UINT_32 hostip;
  COMMAND_INFO *ptr;
  MASK_SET *mask_set;
  struct Tcl_CmdInfo info;
  AGENT_LOCATION *agent_location;

    /* command that will source the agent.tcl script */

  static char initCmd[] =
    "if [file exists [info library]/agent.tcl] {\n\
       source [info library]/agent.tcl\n\
     } else {\n\
       set msg \"can't find [info library]/agent.tcl; perhaps you \"\n\
       append msg \"need to\\ninstall Tcl or set your TCL_LIBRARY \"\n\
       append msg \"environment variable?\"\n\
       error $msg\n\
    }";

    /* set up the cleanup routine that is called on interpreter deletion */

  Tcl_CallWhenDeleted (interp, agentCleanup, (ClientData) NULL);
 
    /* get the host name and IP address */

  if ((hostname = Tcpip_Hostname (interp)) == NULL) {
    return TCL_ERROR;
  } else if (Tcpip_Getip (interp, hostname, &hostip) < 0) {
    return TCL_ERROR;
  }

    /* set up the mask set and turn on the interrupt handler -- this */
    /* will be replaced with an event handler if Tk is then loaded   */
    /* into the interpreter                                          */

  mask_set = new MASK_SET (interp, hostip);
  mask_set -> setEvents (ArrivalTrap, NULL);
 
    /* set up the agent identification */

  agent_location = new AGENT_LOCATION (interp, hostname, hostip, language, server);
 
    /* install the agent commands */

  for (ptr = agent_commands; ptr -> name != NULL; ptr++) {
    if (!(Tcl_GetCommandInfo (interp, ptr -> name, &info) && ptr -> overwrite)) {
      if (ptr -> interruptable) {
        Tcl_CreateIntCommand 
          (interp, ptr -> name, ptr -> proc, 
    	    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
      } else {
        Tcl_CreateCommand 
          (interp, ptr -> name, ptr -> proc, 
  	    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
      }
    }
  }

    /* source the agent.tcl script */

  Tcl_GlobalEval (interp, initCmd);
  return (interp -> resultCode);
}
