/* Agent Tcl
   Bob Gray
   18 February 1995
  
   tclLocation.cc

   This file implements the routines that keep track of agent locations.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "my_strings.h"
#include "tcl.h"
#include "tclAgent.h"
#include "tclAgentInt.h"
#include "tclLocation.h"
#include "tclTcpip.h"
#include "tcl_utilities.h"
#include "timers.h"
#include "truefalse.h"

static char *agentIdTrace (ClientData clientData, Tcl_Interp *interp, char *name1, char *name2, int flags);

  /* initialize the POINTER_MAP structure in AGENT_LOCATION */

POINTER_MAP *AGENT_LOCATION::interp_map = NULL;

  /* get the AGENT_LOCATION structure associated with a given interpreter */

AGENT_LOCATION *AGENT_LOCATION::get_location (Tcl_Interp *interp)
{
  return ((AGENT_LOCATION *) interp_map -> lookup ((void *) interp));
}

  /* constructor for class AGENT_LOCATION */

AGENT_LOCATION::AGENT_LOCATION (Tcl_Interp *interp, char *host_name, UINT_32 host_ip, char *language, int server)
{
  int trace_flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;
 
    /* default location */

  AGENT_LOCATION::interp        = interp;
  AGENT_LOCATION::root          = new AGENT_ID (NULL, 0, NULL, 0);
  AGENT_LOCATION::local         = new AGENT_ID (NULL, 0, NULL, 0);
  AGENT_LOCATION::actual        = new MACHINE_ID (host_name, host_ip);
  AGENT_LOCATION::modified      = FALSE; 
  AGENT_LOCATION::registered    = AGENT_NOT_REGISTERED;
  AGENT_LOCATION::toplevel      = AGENT_NOT_ROOT;
  AGENT_LOCATION::startTime     = TIMERS::getWall ();
  AGENT_LOCATION::language      = my_strcpy (language);
  AGENT_LOCATION::server        = server;

    /* add the interpreter to the map -- first creating the map if necessary */

  if (interp_map == NULL) {
    interp_map = new POINTER_MAP ();
  }

  interp_map -> add ((void *) interp, (void *) this);

    /* load the Tcl array and turn on the array trace */
 
  refresh ();
  Tcl_TraceVar2 (interp, "agent", NULL, trace_flags, agentIdTrace, (ClientData) this);  
  Tcl_GlueVar2 (interp, "agent", NULL, TCL_GLOBAL_ONLY);
}

  /* destructor for class AGENT_LOCATION */

AGENT_LOCATION::~AGENT_LOCATION ()
{
  int trace_flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

    /* turn off the variable trace */

  Tcl_UntraceVar2 (interp, "agent", NULL, trace_flags, agentIdTrace, (ClientData) this);

    /* free all the storage */

  delete actual;
  delete local;
  delete root;
 
    /* remove the interpreter from the map */

  interp_map -> remove ((void *) interp);
}    
 
/* AGENT_LOCATION::refresh

   Purpose: Refresh the Tcl array that holds location information

     Input: None
 
    Output: The procedure loads the current location information into the Tcl
            array.
*/

void AGENT_LOCATION::refresh_id_var (char *name, char *string)
{
  if (string == NULL) {
    Tcl_SetVar2 (interp, "agent", name, "", TCL_GLOBAL_ONLY);
  } else {
    Tcl_SetVar2 (interp, "agent", name, string, TCL_GLOBAL_ONLY);
  }
}

void AGENT_LOCATION::refresh (void)
{
  char temp[16];
  char *string;
  Tcl_DString *dstring;

  modified = TRUE;      /* allow variable modification   */

  if (!registered) {
    Tcl_SetVar2 (interp, "agent", "root-server" , "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2 (interp, "agent", "root-ip"     , "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2 (interp, "agent", "root-name"   , "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2 (interp, "agent", "root-id"     , "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2 (interp, "agent", "local-server", "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2 (interp, "agent", "local-ip"    , "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2 (interp, "agent", "local-name"  , "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2 (interp, "agent", "local-id"    , "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2 (interp, "agent", "local"       , "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2 (interp, "agent", "root"        , "", TCL_GLOBAL_ONLY);
  } else {
    refresh_id_var ("root-server" , root  -> server);
    refresh_id_var ("root-name"   , root  -> name  );
    refresh_id_var ("local-server", local -> server);
    refresh_id_var ("local-name"  , local -> name  );
    
    sprintf (temp, "%u", root -> id); 
    Tcl_SetVar2 (interp, "agent", "root-id", temp, TCL_GLOBAL_ONLY);

    sprintf (temp, "%u", local -> id); 
    Tcl_SetVar2 (interp, "agent", "local-id", temp, TCL_GLOBAL_ONLY);
 
    string = Tcpip_IpToString (interp, root -> ip);
    Tcl_SetVar2 (interp, "agent", "root-ip", string, TCL_GLOBAL_ONLY);
    delete string;

    string = Tcpip_IpToString (interp, local -> ip);
    Tcl_SetVar2 (interp, "agent", "local-ip", string, TCL_GLOBAL_ONLY);
    delete string;

    dstring = Agent_IdToString (interp, root);
    Tcl_SetVar2 (interp, "agent", "root", Tcl_DStringValue (dstring), TCL_GLOBAL_ONLY);
    tclFreeDString (dstring);

    dstring = Agent_IdToString (interp, local);
    Tcl_SetVar2 (interp, "agent", "local", Tcl_DStringValue (dstring), TCL_GLOBAL_ONLY);
    tclFreeDString (dstring);
  }

  refresh_id_var ("actual-server", actual -> server);

  string = Tcpip_IpToString (interp, actual -> server_ip);
  Tcl_SetVar2 (interp, "agent", "actual-ip", string, TCL_GLOBAL_ONLY);
  delete string;

  sprintf (temp, "%d", registered);
  Tcl_SetVar2 (interp, "agent", "registered", temp, TCL_GLOBAL_ONLY);

  sprintf (temp, "%d", toplevel);
  Tcl_SetVar2 (interp, "agent", "toplevel", temp, TCL_GLOBAL_ONLY);

  sprintf (temp, "%d", server);
  Tcl_SetVar2 (interp, "agent", "server", temp, TCL_GLOBAL_ONLY);

  Tcl_SetVar2 (interp, "agent", "language", language, TCL_GLOBAL_ONLY);
 
  modified = FALSE;    /* disallow variable modification */
}

/* AGENT_LOCATION::reload

   Purpose: Reload the location information

     Input: root       = root id
            local      = local id
            registered = AGENT_REGISTERED or AGENT_NOT_REGISTERED
            toplevel   = AGENT_ROOT or AGENT_NOT_ROOT
 
    Output: The procedure updates the agent location and refreshes the Tcl
            array.
*/

void AGENT_LOCATION::reload (AGENT_ID *new_root, AGENT_ID *new_local, int new_registered, int new_toplevel)
{
  registered = new_registered;
  toplevel   = new_toplevel;

  if (new_root != root)
  {
    delete root;

    if (new_root == NULL)
      root = new AGENT_ID (NULL, 0, NULL, 0);
    else
      root = new AGENT_ID (*new_root);
  }

  if (new_local != local)
  {
    delete local;

    if (new_local == NULL)
      local = new AGENT_ID (NULL, 0, NULL, 0);
    else
      local = new AGENT_ID (*new_local);
  }

  refresh();
}

static char *agentIdTrace (ClientData clientData, Tcl_Interp *interp, char *name1, char *name2, int flags)
{
  AGENT_LOCATION *agent_location = (AGENT_LOCATION *) clientData;
  int             trace_flags    = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

    /* allow the operation if modified flag is SET */

  if (agent_location -> modified) {
    return NULL;  
  }

    /* restore original values on a write */

  if (flags & TCL_TRACE_WRITES) {
    agent_location -> refresh ();
    return "can not overwrite agent identification";
  }

    /* otherwise we are trying to unset                 */

  if (!(flags & TCL_INTERP_DESTROYED)) {

    agent_location -> refresh ();
    Tcl_GlueVar2 (interp, "agent", NULL, TCL_GLOBAL_ONLY);

    if (flags & TCL_TRACE_DESTROYED) {
      Tcl_TraceVar2 (interp, "agent", NULL, trace_flags, agentIdTrace, (ClientData) agent_location);  
    }
  }

  return NULL;
}
