/* Agent Tcl
 * Bob Gray
 * 10 Spetember 1995
 *
 * tclMain.cc --
 *
 *	Main program for the Tcl shell
 *
 * Copyright (c) 1995, Robert S. Gray Dartmouth College
 * Copyright (c) 1988-1994 The Regents of the University of California.
 * Copyright (c) 1994 Sun Microsystems, Inc.
 *
 * See the file "agent.terms" for information on usage and redistribution
 * of this file and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <stdio.h>
#include <stdlib.h>
#include <tcl.h>
#include <errno.h>
#ifdef SYSV
#include <string.h>
#endif 
#include <unistd.h>
#include "fd_utilities.h"
#include "tclMask.h"
#include "message.h"
#include "my_sizes.h"
#include "my_strings.h"
#include "redirect.h"
#include "tclLocation.h"
#include "tcpip.h"
#include "transmit.h"
#include "truefalse.h"

/* 
 * Maximum number of seconds to wait between transmission attempts 
 * Maximum number of transmission attempts
 */

const MAXIMUM_SLEEP = 512;
const MAXIMUM_TRIES = 10;

/*
 * Declarations for various library procedures and variables (don't want
 * to include tclPort.h here, because people might copy this file out of
 * the Tcl source directory to make their own modified versions).
 * Note:  "exit" should really be declared here, but there's no way to
 * declare it without causing conflicts with other definitions elsewher
 * on some systems, so it's better just to leave it out.
 */

extern int		errno;
extern int		isatty _ANSI_ARGS_((int fd));
extern char *		strcpy _ANSI_ARGS_((char *dst, CONST char *src));

static Tcl_Interp *interp;	/* Interpreter for application. */
static Tcl_DString command;	/* Used to buffer incomplete commands being
				 * read from stdin. */

/* Agent_ServerMain

   Purpose: Execute agents that arrive through the server

     Input: socketName  = name of the Unix domain socket
            appInitProc = Tcl initialization procedure
 
    Output: The procedure excutes the incoming agent.
*/

void Agent_ServerMain (char *socketName, Agent_AppInitProc *appInitProc)
{
  int sockfd;           /* socket descriptor                    */
  MESSAGE *message;     /* message carrying the script or state */
  int return_code;      /* return_code from Tcl_Eval            */
  Tcl_DString error;    /* error message from Tcl_LoadState     */

    /* connect to the socket */

  if ((sockfd = tcpip_unix_socket ()) < 0) {
    exit (1);
  }

  if (tcpip_unix_connect (sockfd, socketName) < 0) {
    exit (1);
  }

    /* receive the message */

  if ((message = message_receive (sockfd, to_interpreter)) == NULL) {
    exit (1);
  }

  close (sockfd);

    /* redirect stdin, stdout and stderr */

  if (fileRedirect() < 0) {
    exit (1);
  }

    /* break out the message components */

  char *root_server  = message -> elements[0].string;
  UINT_32 root_ip    = message -> elements[1].number;
  char *root_name    = message -> elements[2].string;
  UINT_32 root_id    = message -> elements[3].number;
  char *local_server = message -> elements[4].string;
  UINT_32 local_ip   = message -> elements[5].number;
  UINT_32 local_id   = message -> elements[6].number;
  char *script       = message -> elements[7].string;

    /* root and local identification */

  AGENT_ID *root  = new AGENT_ID (root_server, root_ip, root_name, root_id);
  AGENT_ID *local = new AGENT_ID (local_server, local_ip, NULL, local_id);

    /* initialize the interpreter */

  Tcl_DStringInit (&error);

  if (message -> flag != INTERP_STATE) {
    interp = Tcl_CreateInterp ();
  } else {
    Tcl_DStringInit (&error);
    if ((interp = Tcl_CreateAndLoadInterp (script, &error)) == NULL) {
      exit (1); 
    }
    Tcl_DStringFree (&error);
  }

  interp -> interactive = 0;


  char *temp = my_strcpy (interp -> result);

  if ((*appInitProc)(interp, TRUE) != TCL_OK) {
    exit (1);
  }

  Tcl_SetResult (interp, temp, TCL_DYNAMIC);

    /* load the locations */

  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);

  if (root -> server == NULL) {
    locations -> reload (local, local, AGENT_REGISTERED, AGENT_ROOT);
  } else {
    locations -> reload (root, local, AGENT_REGISTERED, AGENT_NOT_ROOT);
  }

    /* set up the background handler for messages, events and meetings */

  MASK_SET *mask_set = MASK_SET::get_mask_set (interp);
 
  if (mask_set -> sigio_on (local) < 0) {
    exit (1);
  }

    /* execute the script */

  if (message -> flag == INTERP_STATE) {
    return_code = Tcl_Eval (interp, NULL);
  } else {
    return_code = Tcl_Eval (interp, script);
  }
 
  Agent_SendResult (interp, MAXIMUM_TRIES, MAXIMUM_SLEEP, return_code);
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_Main --
 *
 *	Main program for tclsh and most other Tcl-based applications.
 *
 * Results:
 *	None. This procedure never returns (it exits the process when
 *	it's done.
 *
 * Side effects:
 *	This procedure initializes the Tcl world and then starts
 *	interpreting commands;  almost anything could happen, depending
 *	on the script being interpreted.
 *
 *----------------------------------------------------------------------
 */

void Agent_Main (int argc, char **argv, Agent_AppInitProc *appInitProc)
{
    char buffer[1000], *cmd, *args, *fileName;
    int code, gotPartial, tty;
    int exitCode = 0;

    /*
     * Handle agents that arrive via the server
     */

    if ((argc > 2) && (!strcmp(argv[1], "-agent"))) {
	Agent_ServerMain (argv[2], appInitProc);
        goto done;
    }

    /*  
     * Handle agents that do NOT arrive through the server
     *
     * First make command-line arguments available in the Tcl variables "argc"
     * and "argv".  If the first argument does not * start with a "-" then it 
     * is the name of a script file.
     */

    interp = Tcl_CreateInterp();

    fileName   = NULL;

    if ((argc > 1) && (argv[1][0] != '-')) {
	fileName = argv[1];
	argc--;
	argv++;
    }

    args = Tcl_Merge(argc-1, argv+1);
    Tcl_SetVar(interp, "argv", args, TCL_GLOBAL_ONLY);
    ckfree(args);
    sprintf(buffer, "%d", argc-1);
    Tcl_SetVar(interp, "argc", buffer, TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "argv0", (fileName != NULL) ? fileName : argv[0],
	    TCL_GLOBAL_ONLY);

    /*
     * Set the "tcl_interactive" variable.
     */

    tty = isatty(0);
    interp -> interactive = (fileName == NULL) && tty;
    Tcl_SetVar(interp, "tcl_interactive",
		interp -> interactive ? "1" : "0", TCL_GLOBAL_ONLY);
    Tcl_GlueVar2 (interp, "tcl_interactive", NULL, TCL_GLOBAL_ONLY);

    /*
     * Invoke application-specific initialization.
     */

    if ((*appInitProc)(interp, FALSE) != TCL_OK) {
	fprintf(stderr, "application-specific initilization: %s\n", 
		interp -> result);
    }

    /*
     * If a script file was specified then just source that file and quit.
     */

    if (fileName != NULL) {
	code = Tcl_EvalFile(interp, fileName);
	if (code != TCL_OK) {
	    fprintf(stderr, "%s\n", interp -> result);
	    exitCode = 1;
	}
	goto done;
    }

    /*
     * We're running interactively.  Source a user-specific startup
     * file if Tcl_AppInit specified one and if the file exists.
     */

    if (tcl_RcFileName != NULL) {
	Tcl_DString buffer;
	char *fullName;
	FILE *f;

	fullName = Tcl_TildeSubst(interp, tcl_RcFileName, &buffer);
	if (fullName == NULL) {
	    fprintf(stderr, "%s\n", interp->result);
	} else {
	    f = fopen(fullName, "r");
	    if (f != NULL) {
		code = Tcl_EvalFile(interp, fullName);
		if (code != TCL_OK) {
		    fprintf(stderr, "%s\n", interp->result);
		}
		fclose(f);
	    }
	}
	Tcl_DStringFree(&buffer);
    }

    /*
     * Process commands from stdin until there's an end-of-file.
     */

  gotPartial = 0;
  Tcl_DStringInit(&command);

  while (1) {

    clearerr(stdin);

      /* display the prompt if we are on a tty */

    if (tty) {

      char *promptCmd;
      promptCmd = Tcl_GetVar(interp, gotPartial ? "tcl_prompt2" : "tcl_prompt1", TCL_GLOBAL_ONLY);

      if (promptCmd == NULL) {

defaultPrompt:

        if (!gotPartial) {
          fputs("agent-tcl> ", stdout);
        }

      } else {

        if ((code = Tcl_Eval(interp, promptCmd))  != TCL_OK) {
          fprintf(stderr, "%s\n", interp->result);
          Tcl_AddErrorInfo(interp, "\n    (script that generates prompt)");
          goto defaultPrompt;
        }
      }

      fflush(stdout);
    }

      /* loop until we get something from stdin */

    while (fgets(buffer, 1000, stdin) == NULL) {
      if (ferror(stdin)) {
        if (errno != EINTR) {
          goto done;
        } 
        if (tcl_AsyncReady) {
          (void) Tcl_AsyncInvoke((Tcl_Interp *) NULL, 0);
        } 
        clearerr (stdin);
      }

      buffer[0] = 0;
    }	

	cmd = Tcl_DStringAppend(&command, buffer, -1);
	if ((buffer[0] != 0) && !Tcl_CommandComplete(cmd)) {
	    gotPartial = 1;
	    continue;
	}

	gotPartial = 0;
	code = Tcl_RecordAndEval(interp, cmd, 0);
	Tcl_DStringFree(&command);
	if (code != TCL_OK) {
	    fprintf(stderr, "%s\n", interp->result);
	} else if (tty && (*interp->result != 0)) {
	    printf("%s\n", interp->result);
	}
    }

    /*
     * Rather than calling exit, invoke the "exit" command so that
     * users can replace "exit" with some other command to do additional
     * cleanup on exit.  The Tcl_Eval call should never return.
     */

    done:
    sprintf(buffer, "exit %d", exitCode);
    Tcl_Eval(interp, buffer);
}
