/* Agent Tcl
   Bob Gray
   30 August 1995

   tclMask.cc

   This file implements the library routines that handle the message, meeting 
   and event masks.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#ifdef AIX
#include <sys/select.h>
#endif 
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <assert.h>
#ifdef SYSV
#ifndef SOLARIS
#include <bstring.h>
#endif
#endif
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "agentId.h"
#include "assert.h"
#include "interrupt.h"
#include "message.h"
#include "my_alloc.h"
#include "my_sizes.h"
#include "my_strings.h"
#include "sigio.h"
#include "tcl.h"
#include "tcpip.h"
#include "tclAgent.h"
#include "tclRestrictInt.h"
#include "tclMask.h"
#include "transmit.h"
#include "truefalse.h"
#include "transmit.h"

static char *agentMaskTrace 
	(ClientData clientData, Tcl_Interp *interp, char *name1, char *name2, int flags);

static MASK_INT *agentMaskLookup 
	(Tcl_Interp *interp, int handle, char *routine);

static char *maskName[NUM_ITEMS] = 
	{"meeting", "message", "event"};

  /* initialize the static data structures in MASK_SET */

POINTER_MAP *MASK_SET::interp_map = NULL;

  /* get the MASK_SET structure associated with a given interpreter */

MASK_SET *MASK_SET::get_mask_set (Tcl_Interp *interp) {
  return ((MASK_SET *) interp_map -> lookup ((void *) interp));
}

/* MaskArrivalData::MaskArrivalData and MaskArrivalData::~MaskArrivalData

   Purpose: These procedures are the constructor and destructor for class
            MaskArrivalData.
*/

MaskArrivalData::MaskArrivalData (INCOMING *item, char *handler)
{
  MaskArrivalData::item    = item;
  MaskArrivalData::handler = my_strcpy (handler);
  item -> count           += 1;
}

MaskArrivalData::~MaskArrivalData ()
{
  if (item -> count <= 1) {
    delete_check (item);
  } else {
    item -> count -= 1;
  }

  delete_check (handler);
}

/* MASK_ENTRY::MASK_ENTRY

   Purpose: These procedures are the constructor and destructor for class
	    MASK_ENTRY.
*/

MASK_ENTRY::MASK_ENTRY (void)
{
  id      = new AGENT_ID ();
  tag     = (char *) NULL; 
  handler = (char *) NULL;
}

MASK_ENTRY::MASK_ENTRY (MASK_ENTRY *mask)
{
  id      = new AGENT_ID (mask -> id);
  tag     = my_strcpy (mask -> tag);
  handler = my_strcpy (mask -> handler);
}

MASK_ENTRY::MASK_ENTRY (const MASK_ENTRY &mask) 
{
  id      = new AGENT_ID (mask.id);
  tag     = my_strcpy (mask.tag);
  handler = my_strcpy (mask.handler);
}

MASK_ENTRY::~MASK_ENTRY ()
{
  delete_check (id);
  delete_check (tag);
  delete_check (handler);
}

/* MASK_SET::MASK_SET and MASK_SET::~MASK_SET

   Purpose: These procedures are the constructor and destructor for class
            MASK_SET.

     Input: interp    = the current interpreter
*/

MASK_SET::MASK_SET (Tcl_Interp *interp, UINT_32 ip)
{
  int trace_flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

    /* add the interpreter to the mask map -- first creating the map if nec. */

  if (interp_map == NULL) {
    interp_map = new POINTER_MAP ();
  }
 
  interp_map -> add ((void *) interp, (void *) this);

    /* initialize the hash table of masks */

  Tcl_InitHashTable (&mask_table, TCL_ONE_WORD_KEYS);

    /* initialize the mask handle and count */

  mask_interp = interp;
  mask_handle = 0;
  mask_count  = 0;
  mod_allowed = FALSE;
  handler     = NULL;
  sockfd      = -1;
  arrivalProc = (MaskEventProc *) NULL;
  brokenProc  = (MaskEventProc *) NULL;
  eventCount  = 0;
  arrived     = FALSE;
  actualhost  = ip;

    /* initialize the meeting, message and event masks */

  masks[MESSAGE_ITEM]          = new_mask (MASK_ACCEPT_ALL);
  masks[MEETING_ITEM]          = new_mask (MASK_ACCEPT_ALL);
  masks[EVENT_ITEM]            = new_mask (MASK_ACCEPT_ALL);
  masks[MESSAGE_ITEM] -> flags = MESSAGE_BIT;
  masks[MEETING_ITEM] -> flags = MEETING_BIT;
  masks[EVENT_ITEM]   -> flags = EVENT_BIT;

    /* create the Tcl interrupt handlers */

  arrivalAsync = Tcl_AsyncCreate (maskAsyncArrival, (ClientData) interp);
  eventAsync   = Tcl_AsyncCreate (maskAsyncEvent, (ClientData) interp);
  brokenAsync  = Tcl_AsyncCreate (maskAsyncBroken, (ClientData) interp);
 
    /* load the Tcl array and turn on the trace */

  refresh ();
  Tcl_TraceVar2 (interp, "mask", NULL, trace_flags, agentMaskTrace, (ClientData) this);
  Tcl_GlueVar2 (interp, "mask", NULL, TCL_GLOBAL_ONLY);
} 

MASK_SET::~MASK_SET ()
{
  MASK_INT *mask;
  Tcl_HashEntry *entry;
  Tcl_HashSearch search;
  int trace_flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

    /* turn off the SIGIO handler */

  if (handler != NULL) {
    sigio_off ();
  }

    /* delete the Tcl interrupt handlers */

  Tcl_AsyncDelete (arrivalAsync);
  Tcl_AsyncDelete (brokenAsync);
  Tcl_AsyncDelete (eventAsync);

    /* turn off the trace */

  Tcl_UntraceVar2 (mask_interp, "mask", NULL, trace_flags, agentMaskTrace, (ClientData) this);

    /* delete all the storage */

  for (entry = Tcl_FirstHashEntry (&mask_table, &search); entry != NULL; entry = Tcl_NextHashEntry (&search)) {
    mask = (MASK_INT *) Tcl_GetHashValue (entry);
    delete (mask);
  } 

  Tcl_DeleteHashTable (&mask_table);

    /* remove the interpreter from the map */

  interp_map -> remove ((void *) mask_interp);
}

/* MASK_INT::MASK_INT and MASK_INT::~MASK_INT

   Purpose: These procedures are the constructor and destructor for class
            MASK_INT.

     Input: type   = MASK_ACCEPT_NONE or MASK_ACCEPT_ALL
            handle = mask handle
            set    = MASK_SET that contains this mask 
*/

MASK_INT::MASK_INT (int type, int handle, MASK_SET *set)
{
    /* assertions on the parameters */

  assert ((type == MASK_ACCEPT_NONE) || (type == MASK_ACCEPT_ALL));
  assert (set != NULL);

    /* fill in the variables */

  MASK_INT::type    = type;
  MASK_INT::handler = NULL;
  MASK_INT::flags   = 0;
  MASK_INT::handle  = handle;
  MASK_INT::set     = set;
}
  
MASK_INT::~MASK_INT ()
{
  int i;
  MASK_ENTRY *entry;
  int refresh = FALSE;

    /* update MASK_SET if this is a distinguished mask */

  for (i = 0; i < NUM_ITEMS; i++) {
    if (set -> masks[i] == this) {
      refresh = TRUE;
      set -> masks[i] = NULL;
    }
  }

  if (refresh) {
    set -> refresh ();
  }

     /* remove all the mask entries */

  while ((entry = (MASK_ENTRY *) list.dequeue ()) != NULL) {
    delete (entry);
  } 

     /* remove the default handler */

  delete_check (handler);
}

/* MASK::MASK and MASK::~MASK

   Purpose: These procedures are the constructor and destructor for
            class MASK. 
*/

MASK::MASK (int type, int handle, int count, char *handler)
{
    /* assertions on the parameters */

  assert (count >= 0);

    /* set up the variables */

  MASK::type    = type; 
  MASK::count   = count;
  MASK::handle  = handle;
  MASK::handler = my_strcpy (handler);
 
  if (count == 0) {
    masks = NULL;
  } else {
    masks = new MASK_ENTRY * [count];
  }

  for (int i = 0; i < count; i++) {
    masks[i] = NULL;
  }
}

MASK::MASK (MASK_INT *mask)
{
  register int i;

  type    = mask -> type;
  handle  = mask -> handle; 
  count   = mask -> list.get_count ();
  handler = my_strcpy (mask -> handler);

  if (count == 0) {
    masks = NULL;
  } else {
    masks = new MASK_ENTRY * [count];
  }

  for (i = 0; i < count; i++) {
    masks[i] = new MASK_ENTRY ((MASK_ENTRY *) mask -> list.peek(i));
  }
}

MASK::~MASK ()
{
  delete_check (handler);

  for (int i = 0; i < count; i++) {
    delete_check (masks[i]);
  }
}

/* MASK_SET::refresh

   Purpose: Refresh the Tcl array "mask" 

     Input: None

    Output: The procedure loads the current mask information into the Tcl
             array "mask".
*/

void MASK_SET::refresh (void)
{
  char temp[16];

  mod_allowed = TRUE;

  for (int i = 0; i < NUM_ITEMS; i++) {
    if (masks[i] == NULL) {
      Tcl_SetVar2 (mask_interp, "mask", maskName[i], "NONE", TCL_GLOBAL_ONLY);
    } else {
      sprintf (temp, "%d", masks[i] -> handle);
      Tcl_SetVar2 (mask_interp, "mask", maskName[i], temp, TCL_GLOBAL_ONLY);
    }
  }

  mod_allowed = FALSE;
}

/* MASK_INT::match_exact

   Purpose: Perform an exact match between mask entries.  This procedure is
            used when removing a mask entry.

     Input: mask1 = first mask entry
            mask2 = second mask entry

    Output: The procedure returns TRUE if the two masks match exactly and
            FALSE otherwise.
*/
 
int MASK_INT::match_exact (MASK_ENTRY *mask1, MASK_ENTRY *mask2)
{
  if (my_strcmp(mask1 -> id -> server, mask2 -> id -> server)) {
    return FALSE;
  }

  if (mask1 -> id -> ip != mask2 -> id -> ip) {
    return FALSE;
  }

  if (my_strcmp(mask1 -> id -> name, mask2 -> id -> name)) {
    return FALSE;
  }

  if (mask1 -> id -> id != mask2 -> id -> id) {
    return FALSE;
  }

  if (my_strcmp(mask1 -> tag, mask2 -> tag)) {
    return FALSE;
  }

  if (my_strcmp(mask1 -> handler, mask2 -> handler)) {
    return FALSE;
  }

  return TRUE;
}

/* MASK_INT::match_inexact

   Purpose: Perform an inexact match between a mask entry and an incoming
            item.  This procedure is used when determining if the mask entry
            lets the item through to the agent.

     Input: maskExt = the mask entry
            item    = the incoming item

    Output: The procedure returns TRUE if the mask entry and incoming item
            match and FALSE otherwise.
*/

static int check_nulls (char *src, char *dest)
{
  if ((src == NULL) && (dest == NULL)) {
    return TRUE;
  } else if ((src == NULL) && (*dest == '\0')) {
    return TRUE;
  } else if ((dest == NULL) && (*src == '\0')) {
    return TRUE;
  } else {
    return FALSE;
  }
}

int MASK_INT::match_inexact (MASK_ENTRY *maskExt, INCOMING *item)
{
  static UINT_32 localhost = inet_addr ("127.0.0.1");
 
    /* compare the mask from the list against the given mask */

  if (maskExt -> id -> ip != ANY_IP) {
    if (maskExt -> id -> ip != item -> id -> ip) {
      if ((maskExt -> id -> ip != localhost) || (item -> id -> ip != set -> actualhost)) {
        return FALSE; 
      } 
    }
  } 

  if (maskExt -> id -> id != ANY_ID) {
    if (maskExt -> id -> id != item -> id -> id) {
      return FALSE; 
    } 
  } 

  if (maskExt -> id -> name != ANY_NAME) {
    if (!check_nulls (maskExt -> id -> name, item -> id -> name)) {
      if (my_strcmp (maskExt -> id -> name, item -> id -> name)) {
        return FALSE; 
      }
    }
  }

  if (item -> type == EVENT_ITEM) {
    if (maskExt -> tag != ANY_TAG) {
      if (!check_nulls (maskExt -> tag, ((INCOMING_EVENT *) item) -> tag)) {
        if (my_strcmp (maskExt -> tag, ((INCOMING_EVENT *) item) -> tag)) {
          return FALSE; 
        }
      }
    }
  }

  return TRUE;
}

/* MASK_INT::remove

   Purpose: Remove one or more masks from a list

     Input: count = number of masks
            masks = the masks

    Output: The procedure removes the masks.

      Note: It is NOT an error if a mask is not in the list.
*/

void MASK_INT::remove (int count, MASK_ENTRY **masks)
{
  register int i;
  register int j;
  sigset_t oldMask;
  sigset_t newMask;
  register MASK_ENTRY *mask;
  register int stop = list.get_count();

    /* assertions on the parameters */

  assert (count > 0);
  assert (masks != NULL);

    /* block SIGIO if this mask is the current message, meeting or event  */
    /* mask -- i.e. block SIGIO if it is possible for the SIGIO interrupt */
    /* handler to indirectly access this mask                             */

  if (flags != 0) {
    sigemptyset (&oldMask);
    sigemptyset (&newMask);
    sigaddset (&newMask, SIGIO);
    sigprocmask (SIG_BLOCK, &newMask, &oldMask);
  }

    /* remove the masks */

  for (i = 0; i < stop; i++) {

    mask = (MASK_ENTRY *) list.dequeue ();

    for (j = 0; j < count; j++) {
      if (match_exact (mask, masks[j])) {
	break;
      }
    }

    if (j < count) {
      continue;
    }

    list.enqueue ((void *) mask);
  }

    /* is the mask now empty */

  if (list.get_count() == 0) {
    type = MASK_ACCEPT_NONE;
  }

    /* unblock SIGIO if we blocked it */
 
  if (flags != 0) {
    sigprocmask (SIG_SETMASK, &oldMask, NULL);
  }
}

/* MASK_SET::queue_events

   Purpose: Queue all event handlers that are defined for an incoming item

     Input: which = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM
            item  = the incoming item
 
    Output: The procedure returns TRUE if one or more event handlers are queued.
            Otherwise the procedure returns FALSE.

      Note: This procedure assumes that SIGIO is blocked.
*/

int MASK_SET::queue_events (int which, INCOMING *item)
{
  register int i;
  register int stop;
  MaskArrivalData *data;
  int eventQueued = FALSE;
  register MASK_ENTRY *entry;

    /* assertions on the parameters */

  assert ((which == MEETING_ITEM) || (which == MESSAGE_ITEM) || (which == EVENT_ITEM));
  assert (item != NULL);

    /* check if we have a handler proc and a mask with entries */

  if ((arrivalProc == NULL) || (masks[which] == NULL) || (masks[which] -> type == MASK_ACCEPT_NONE)) {
    return FALSE;
  }

    /* check each mask in the list */

  stop = masks[which] -> list.get_count ();

  for (i = 0; i < stop; i++) {

    entry = (MASK_ENTRY *) masks[which] -> list.peek (i);

    if (entry -> handler != NULL) { 
      if (masks[which] -> match_inexact (entry, item)) {
        data = new MaskArrivalData (item, entry -> handler);
        eventQueue.enqueue ((void *) data);
        eventQueued = TRUE;
      }
    }
  }

    /* check for a default event handler if we have not found any others */

  if (!eventQueued) {
    if (masks[which] -> handler != NULL) {
      data = new MaskArrivalData (item, masks[which] -> handler);
      eventQueue.enqueue ((void *) data);
      eventQueued = TRUE;
    }
  }

    /* fire the asynchronous Tcl procedure */

  if (eventQueued) {
    Tcl_AsyncMark (eventAsync);
    eventCount += 1;
  }

    /* return TRUE if event was queued and FALSE otherwise */

  return (eventQueued);
}


/* MASK_INT::filter

   Purpose: Filter an incoming item through the mask

     Input: id = agent identification

    Output: The procedure returns TRUE if there is an entry in the mask
            that matches the incoming item.  Otherwise the procedure returns
            FALSE.

      Note: The procedure assumes that SIGIO is blocked.
*/

int MASK_INT::filter (INCOMING *item)
{
  register int i;
  sigset_t oldMask;
  register MASK_ENTRY *entry;
  register int total = list.get_count ();

    /* assertions on the parameters */

  assert (item != NULL);
  assert ((sigprocmask (0, NULL, &oldMask), sigismember (&oldMask, SIGIO)));

    /* check if the the list contains ALL masks or NO masks */

  if (type == MASK_ACCEPT_ALL) {
    return TRUE;
  } else if (type == MASK_ACCEPT_NONE) {
    return FALSE;
  }

    /* check each mask entry */

  for (i = 0; i < total; i++) {

    entry = (MASK_ENTRY *) list.peek (i);

    if (match_inexact (entry, item)) {
      return TRUE;
    }
  }

    /* no mask entry matches the identification */

  return FALSE;
}

/* MASK_INT::add

   Purpose: Add one or more masks to the list

     Input: count = number of masks
            masks = the masks

    Output: The procedure adds the masks to the list.
*/

void MASK_INT::add (int count, MASK_ENTRY **entries)
{
  register int i;
  sigset_t newMask;
  sigset_t oldMask;
  register MASK_ENTRY *maskCopy;

    /* assertions on the parameters */

  assert (count > 0);
  assert (entries != NULL);

    /* block SIGIO if this mask is the current message, meeting or event  */
    /* mask -- i.e. block SIGIO if it is possible for the SIGIO interrupt */
    /* handler to indirectly access this mask                             */

  if (flags != 0) {
    sigemptyset (&oldMask);
    sigemptyset (&newMask);
    sigaddset (&newMask, SIGIO);
    sigprocmask (SIG_BLOCK, &newMask, &oldMask);
  } 

  for (i = 0; i < count; i++) {
    maskCopy = new MASK_ENTRY (entries[i]);
    list.enqueue ((void *) maskCopy);
  } 
  
  type = MASK_ACCEPT_SOME;

  if (flags != 0) {

      /* unblock SIGIO if necessary */

    sigprocmask (SIG_SETMASK, &oldMask, NULL);

      /* queue any event handlers that can now fire */

    for (i = 0; i < NUM_ITEMS; i++) {
      if (set -> masks[i] == this) {
        set -> queue_all_events (i);
      }
    }
  }
}

/* MASK_INT::empty_and_set

   Purpose: Remove all of the mask entries and set the type

     Input: type = MASK_ACCEPT_ALL or MASK_ACCEPT_NONE

    Output: The procedure removes all of the mask entries and sets the type.
*/

void MASK_INT::empty_and_set (int type)
{ 
  register int i; 
  sigset_t oldMask;
  sigset_t newMask;
  MASK_ENTRY *entry;

    /* block SIGIO if this mask is the current message, meeting or event  */
    /* mask -- i.e. block SIGIO if it is possible for the SIGIO interrupt */
    /* handler to indirectly access this mask                             */

  if (flags != 0) {
    sigemptyset (&oldMask);
    sigemptyset (&newMask);
    sigaddset (&newMask, SIGIO);
    sigprocmask (SIG_BLOCK, &newMask, &oldMask);
  }

    /* empty out the mask */

  while ((entry = (MASK_ENTRY *) list.dequeue()) != NULL) {
    delete (entry);
  }
    
  MASK_INT::type = type;

    /* unblock SIGIO if we blocked it */

  if (flags != 0) {

      /* unblock SIGIO if we blocked it */

    sigprocmask (SIG_SETMASK, &oldMask, NULL);

      /* queue any event handlers that can now fire */

    if (type == MASK_ACCEPT_ALL) {
      for (i = 0; i < NUM_ITEMS; i++) {
        if (set -> masks[i] == this) {
          set -> queue_all_events (i);
        }
      }
    }
  }
}

/* MASK_SET::lookup_mask

   Purpose: Find a mask with a given handle

     Input: handle = the mask handle

    Output: The procedure returns NULL if there is no mask with the
            given handle.  Otherwise the procedure returns a pointer to
            a MASK structure.
*/

MASK_INT *MASK_SET::lookup_mask (int handle)
{
  Tcl_HashEntry *entry;

  if ((entry = Tcl_FindHashEntry (&mask_table, (char *) handle)) == NULL) {
    return NULL;
  }

  return ((MASK_INT *) Tcl_GetHashValue (entry));
}

/* MASK_SET::remove_mask

   Purpose: Delete a mask

     Input: handle = ALL_MASKS or mask handle

    Output: If "handle" is ALL_MASKS the procedure deletes all the masks
            in the set and returns 0.  If "handle" is a mask handle, the
            procedure returns -1 if there is no mask with the given handle.
            Otherwise the procedure deletes the mask and returns 0.
*/

int MASK_SET::remove_mask (int handle)
{
  MASK_INT *mask;
  Tcl_HashEntry *entry;
  Tcl_HashSearch search;

  if (handle == ALL_MASKS) {

       /* delete ALL the masks */
    
    for (entry = Tcl_FirstHashEntry (&mask_table, &search); entry != NULL; entry = Tcl_NextHashEntry (&search)) {
      mask = (MASK_INT *) Tcl_GetHashValue (entry);
      delete mask;
    }

    mask_count = 0;
    Tcl_DeleteHashTable (&mask_table);
    Tcl_InitHashTable (&mask_table, TCL_ONE_WORD_KEYS); 

  } else { 

      /* delete the mask with the given handle */

    if ((entry = Tcl_FindHashEntry (&mask_table, (char *) handle)) == NULL) {
      return -1;
    }

    mask = (MASK_INT *) Tcl_GetHashValue (entry);
    Tcl_DeleteHashEntry (entry); 
    delete mask;

    mask_count -= 1;    
  }

  return 0;
}  

/* MASK_SET::update_mask

   Purpose: Update one of the special masks 

     Input: which  = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM
            handle = mask handle 

    Output: The procedure returns -1 if there is no mask with the given handle.
            Otherwise the procedure returns 0 and updates the special mask.
*/

int MASK_SET::update_mask (int which, int handle)
{
  MASK_INT *mask;
  sigset_t oldMask;
  sigset_t newMask;

    /* assertions on the parameters */

  assert ((which == MESSAGE_ITEM) || (which == MEETING_ITEM) || (which == EVENT_ITEM));
  assert (handle >= 0);

    /* find the mask with the given handle */

  if ((mask = lookup_mask (handle)) == NULL) {
    return -1;
  }

  mask -> flags += 1;

    /* block SIGIO, swap in the new mask and unblock SIGIO */

  sigemptyset (&oldMask);
  sigemptyset (&newMask);
  sigaddset (&newMask, SIGIO);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

  if (masks[which] != NULL) {
    masks[which] -> flags -= 1;
  }

  masks[which] = mask;

  sigprocmask (SIG_SETMASK, &oldMask, NULL);

    /* queue any pending events */

  queue_all_events (which);
  return 0;
}
 
/* MASK_SET::new_mask

   Purpose: Create a new mask

     Input: type = MASK_ACCEPT_ALL or MASK_ACCEPT_NONE

    Output: The procedure returns NULL if it is unable to create the mask
            because the maximum number of masks has been exceeded.
            Otherwise the procedure returns a pointer to a dynamically
            allocated MASK_INT structure and sets the "handle" argument 
	    to the mask handle.
*/

MASK_INT *MASK_SET::new_mask (int type)
{
  int new_entry = 0;
  MASK_INT *mask = NULL;
  Tcl_HashEntry *hashEntry;

    /* check the number of masks */

  if (mask_count >= MAX_32) {
    return NULL;
  }

    /* find an unused mask handle */

  while (!new_entry) {

    hashEntry = Tcl_CreateHashEntry (&mask_table, (char *) mask_handle, &new_entry);

    if (new_entry) {
      mask_count += 1;
      mask = new MASK_INT (type, mask_handle, this);
      Tcl_SetHashValue (hashEntry, (ClientData) mask);
    } 
  
    mask_handle = (mask_handle + 1) % MAX_32;
  } 

    /* pass back the mask */

  return (mask);
}

/* agentMaskLookup

   Purpose: Find the mask that has the given handle

     Input: interp  = the current interpreter
            handle  = the mask handle
            routine = name of the routine that called agentMaskLookup

    Output: The procedure returns NULL if there is no mask with the given
            handle.  Otherwise the procedure returns a pointer to the
	    MASK structure that contains the mask.  
*/
 
MASK_INT *agentMaskLookup (Tcl_Interp *interp, int handle, char *routine)
{
  MASK_INT *mask;
  MASK_SET *mask_set;

    /* check the handle */

  if (handle < 0) {
    Tcl_AppendResult (interp, "invalid handle passed to ", routine, (char *) NULL);
    return NULL;
  }

    /* get the set of masks for this interpreter and lookup the mask handle */

  mask_set = MASK_SET::get_mask_set (interp);

  if ((mask = mask_set -> lookup_mask (handle)) == NULL) {
    Tcl_AppendResult (interp, "no mask with that handle", (char *) NULL);
  }

  return (mask);
}

/* Agent_MaskEmpty

   Purpose: Make the mask REJECT any incoming item

     Input: interp = the current interpreter
            handle = the mask handle

    Output: The procedure returns TCL_ERROR and sets the interpreter result to
            an appropriate error message on error.  Otherwise the procedure
            returns TCL_OK and empties the mask.
*/

int Agent_MaskEmpty (Tcl_Interp *interp, int handle)
{
  MASK_INT *mask;

    /* get the mask with the specified handle */

  if ((mask = agentMaskLookup (interp, handle, "Agent_MaskEmpty")) == NULL) {
    return TCL_ERROR;
  }

    /* fill the mask */

  mask -> empty ();
  return TCL_OK;
}

/* Agent_MaskContents

   Purpose: Return the contents of a mask

     Input: interp = the current interpreter
	    handle = mask handle

    Output: The procedure returns NULL and sets the interpreter result to an
            appropriate error message if there is no mask with the given
            handle.  Otherwise the procedure returns a pointer to a
            DYNAMICALLY allocated MASK_ENTRY structure that contains a COPY
            of the mask contents.
*/

MASK *Agent_MaskContents (Tcl_Interp *interp, int handle)
{
  MASK_INT *mask;
  MASK *mask_ext;

    /* get the mask with the specified handle */

  if ((mask = agentMaskLookup (interp, handle, "Agent_MaskContents")) == NULL) {
    return NULL;
  }

    /* create the external mask representation */

  mask_ext = new MASK (mask);
  return (mask_ext);
}

/* Agent_MaskRemove

   Purpose: Remove entries from a mask
  
     Input: interp  = the current interpreter
            handle  = the mask handle
            count   = the number of entries
            entries = the entries

    Output: The procedure returns TCL_ERROR and sets the interpreter result to
            an appropriate error message on error.  Otherwise the procedure
            returns TCL_OK and removes the entries from the mask.
*/

int Agent_MaskRemove (Tcl_Interp *interp, int handle, int count, MASK_ENTRY **entries)
{
  MASK_INT *mask;

    /* lookup up the mask */

  if ((mask = agentMaskLookup (interp, handle, "Agent_MaskRemove")) == NULL) {
    return TCL_ERROR;
  }

    /* remove the entries from the mask */

  mask -> remove (count, entries);
  return TCL_OK;
}
 
/* Agent_MaskAdd

   Purpose: Add entries to a mask
  
     Input: interp  = the current interpreter
            handle  = the mask handle
            count   = the number of entries
            entries = the entries

    Output: The procedure returns TCL_ERROR and sets the interpreter result to
            an appropriate error message on error.  Otherwise the procedure
            returns TCL_OK and adds the entries to the mask.
*/

int Agent_MaskAdd (Tcl_Interp *interp, int handle, int count, MASK_ENTRY **entries)
{
  MASK_INT *mask;

    /* lookup up the mask */

  if ((mask = agentMaskLookup (interp, handle, "Agent_MaskAdd")) == NULL) {
    return TCL_ERROR;
  }

    /* add the entries to the mask */

  mask -> add (count, entries);
  return TCL_OK;
}
 
/* Agent_MaskFill

   Purpose: Make the mask accept ANY incoming item

     Input: interp = the current interpreter
            handle = the mask handle

    Output: The procedure returns TCL_ERROR and sets the interpreter result to
            an appropriate error message on error.  Otherwise the procedure
            returns TCL_OK and fills the mask.
*/

int Agent_MaskFill (Tcl_Interp *interp, int handle)
{
  MASK_INT *mask;

    /* get the mask with the specified handle */

  if ((mask = agentMaskLookup (interp, handle, "Agent_MaskFill")) == NULL) {
    return TCL_ERROR;
  }

    /* fill the mask */

  mask -> fill ();
  return TCL_OK;
}

/* Agent_MaskSetDefault

   Purpose: Associate a default event handler with a mask

     Input: interp      = the current interpreter
            handle      = the mask handle
            handlerName = name of the default event handler

    Output: The procedure returns TCL_error and sets the interpreter result
            to an appropriate message if there is no mask with the specified
            handle.  Otherwise the procedure returns TCL_OK and records the
            default handler.
*/
            
int Agent_MaskSetDefault (Tcl_Interp *interp, int handle, char *handlerName)
{
  MASK_INT *mask;

    /* get the mask with the specified handle */

  if ((mask = agentMaskLookup (interp, handle, "Agent_SetDeault")) == NULL) {
    return TCL_ERROR;
  }

    /* set the default handler */

  mask -> set_handler (handlerName);
  return TCL_OK;
}

/* Agent_MaskGetDefault

   Purpose: Get the default event handler associated with a mask

     Input: interp = the current interpreter
            handle = the mask handle

    Output: The procedure returns TCL_ERROR and sets the interpeter result to
            an appropriate error message if there is no mask with the 
            specified handle.  Otherwis the procedure returns TCL_OK and sets
            *handlerName to the name of the default event handler.  The
            procedure sets *handlerName to NULL if there is no default event
            handler.
*/

int Agent_MaskGetDefault (Tcl_Interp *interp, int handle, char **handlerName)
{
  MASK_INT *mask;

    /* get the mask with the specified handle */

  if ((mask = agentMaskLookup (interp, handle, "Agent_MaskGetDefault")) == NULL) {
    return TCL_ERROR;
  }

    /* get the default event handler */

  *handlerName = mask -> get_handler ();
  return TCL_OK;
}

/* Agent_MaskNew

   Purpose: Create a new mask

     Input: interp = the current interpreter

    Output: The procedure returns TCL_ERROR and sets the interpreter result
            to an appropriate error message on error.  Otherwise the procedure
            returns the handle for the new mask.
*/

int Agent_MaskNew (Tcl_Interp *interp)
{
  MASK_INT *mask;

    /* get the mask set for the given interpreter */

  MASK_SET *mask_set = MASK_SET::get_mask_set (interp);

    /* create the new mask */

  if ((mask = mask_set -> new_mask (MASK_ACCEPT_NONE)) == NULL) {
    Tcl_AppendResult (interp, "maximum number of masks exceeded", (char *) NULL);
    return -1;
  }

  return (mask -> get_handle());
}

/* Agent_MaskDelete

   Purpose: Delete a mask

     Input: interp = the current interpreter
            handle = ALL_MASKS or mask handle

    Output: The procedure returns TCL_ERROR and sets the interpreter result
            to an appropriate error message on error.  Otherwise the procedure
            returns TCL_OK and sets the interpreter result to the empty string.
*/

int Agent_MaskDelete (Tcl_Interp *interp, int handle)
{
  MASK_SET *mask_set;

    /* assert a valid handle */

  if ((handle != ALL_MASKS) && (handle < 0)) {
    Tcl_AppendResult (interp, "invalid handle passed to Agent_MaskDelete", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the mask set for the given interpreter */

  mask_set = MASK_SET::get_mask_set (interp);

    /* remove the mask */

  if (mask_set -> remove_mask (handle) < 0) {
    Tcl_AppendResult (interp, "no mask with that handle", (char *) NULL);
    return TCL_ERROR;
  }

    /* one less mask */

  return TCL_OK;
}

/* Agent_MaskToString

   Purpose: Create a string representation for a mask entry

     Input: interp = Tcl interpreter
            mask   = the mask

    Output: The procedure returns a pointer to a DYNAMICALLY allocated
            Tcl_DString that contains the string representation of the
            mask entry.
*/

Tcl_DString *Agent_MaskToString (Tcl_Interp *interp, MASK_ENTRY *mask)
{
  register AGENT_ID *id = mask -> id; 
  register Tcl_DString *string = new Tcl_DString;
  Tcl_DStringInit (string);

    /* server name and IP address */

  if (id -> server == ANY_SERVER) {
    Tcl_DStringAppendElement (string, "ANY");
    Tcl_DStringAppendElement (string, "ANY");
  } else {
    char *ip_string = Tcpip_IpToString (interp, id -> ip);
    Tcl_DStringAppendElement (string, id -> server);
    Tcl_DStringAppendElement (string, ip_string);
    delete (ip_string);
  }

    /* agent name */

  if (id -> name == ANY_NAME) {
    Tcl_DStringAppendElement (string, "ANY");
  } else {
    Tcl_DStringAppendElement (string, id -> name);
  }

    /* agent id */

  if (id -> id == ANY_ID) {
    Tcl_DStringAppendElement (string, "ANY");
  } else {
    char temp[16];
    sprintf (temp, "%d", id -> id);
    Tcl_DStringAppendElement (string, temp);
  }

    /* event tag and handler name */

  if (mask -> tag != NULL) {
    Tcl_DStringAppendElement (string, "-tag");
    Tcl_DStringAppendElement (string, mask -> tag);
  } 

  if (mask -> handler != NULL) {
    Tcl_DStringAppendElement (string, "-handler");
    Tcl_DStringAppendElement (string, mask -> handler);
  } 

  return (string);
}

/* Agent_SplitMask

   Purpose: Split apart a string that contains a mask entry

     Input: interp = Tcl interpreter
            string = the string

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns a pointer to a dynamically allocated AGENT_ID structure
            that contains the mask entry.
*/

static int parseNumeric (Tcl_Interp *interp, MASK_ENTRY *mask, char *string)
{
  int number;   /* scratch area for number conversion */

  if (strcmp(string, "ANY")) {
    if ((Tcl_GetInt(interp, string, &number) != TCL_OK) || (number < 0)) {
      Tcl_AppendResult (interp, "invalid numeric id \"", string, "\"", (char *) NULL);
      return TCL_ERROR; 
    }

    mask -> id -> id = (UINT_32) number;
  }

  return TCL_OK;
}

static int findServer (Tcl_Interp *interp, MASK_ENTRY *mask, char *string)
{
  UINT_32 ip;

  if (strcmp(string, "ANY")) {
    if (Tcpip_Getip (interp, string, &ip) != TCL_OK) {
      return TCL_ERROR; 
    }

    mask -> id -> server = my_strcpy (string);
    mask -> id -> ip     = ip;
  }

  return TCL_OK;
}

MASK_ENTRY *Agent_SplitMask (Tcl_Interp *interp, char *string)
{
  int listArgc = 0;          /* number of components in the id        */
  MASK_ENTRY *mask = NULL;     /* the mask                              */
  char **listArgv = NULL;    /* components of the id                  */
  int tagFound = FALSE;      /* TRUE if -tag parameter is present     */
  int handlerFound = FALSE;  /* TRUE if -handler parameter is present */
  register int i, j;         /* indices into listArgv                 */

    /* split the string into its components */

  if ((Tcl_SplitList (interp, string, &listArgc, &listArgv) != TCL_OK)) {
    Tcl_AppendResult (interp, "must be a valid Tcl list", (char *) NULL);
    return NULL; 
  }

  mask = new MASK_ENTRY ();

    /* check for -tag and -handler options */

  for (i = listArgc - 2, j = listArgc - 1; i >= 0; i -= 2, j -= 2) {
    if (!strcmp(listArgv[i], "-tag")) {
      if (!strcmp(listArgv[j], "-tag") || !strcmp(listArgv[j], "-handler")) {
        Tcl_AppendResult (interp, "misplaced \"", listArgv[j], (char *) NULL);
        goto error;
      } else if (tagFound) {
        Tcl_AppendResult (interp, "can not have multiple \"-tag\" parameters", (char *) NULL);
        goto error;
      } else {
        mask -> tag = my_strcpy (listArgv[j]);
        tagFound = TRUE;
      }
    } else if (!strcmp(listArgv[i], "-handler")) {
      if (!strcmp(listArgv[j], "-tag") || !strcmp(listArgv[j], "-handler")) {
        Tcl_AppendResult (interp, "misplaced \"", listArgv[j], (char *) NULL);
        goto error;
      } else if (handlerFound) {
        Tcl_AppendResult (interp, "can not have multiple \"-handler\" parameters", (char *) NULL);
        goto error;
      } else {
        mask -> handler = my_strcpy (listArgv[j]);
        handlerFound = TRUE;
      }
    } else {
      break;
    }
  }
   
  listArgc = j + 1;

    /* the id can be underspecified */

  if (listArgc == 4) {

      /* four arguments means we have a server name and an IP ... */

    if (strcmp(listArgv[0], "ANY") && strcmp(listArgv[1], "ANY")) {
      mask -> id -> server = my_strcpy (listArgv[0]);

      if (Tcpip_StringToIp (interp, listArgv[1], &mask -> id -> ip) < 0) {
        goto error;
      }
    }

      /* and a symbolic name ... */

    if (strcmp(listArgv[2], "ANY")) {
      mask -> id -> name = my_strcpy (listArgv[2]);
    }

      /* and a numeric id */

    if (parseNumeric (interp, mask, listArgv[3]) != TCL_OK) {
      goto error; 
    }

  } else if (listArgc == 3) {

      /* three arguments means a server name ... */

    if (findServer (interp, mask, listArgv[0]) != TCL_OK) {
      goto error;
    } 

      /* and a symbolic name ... */

    if (strcmp(listArgv[1], "ANY")) {
      mask -> id -> name = my_strcpy (listArgv[1]);
    }

      /* and a numeric id */

    if (parseNumeric (interp, mask, listArgv[2]) != TCL_OK) {
      goto error;
    }

  } else if (listArgc == 2) {

      /* two arguments means a server name ... */

    if (findServer (interp, mask, listArgv[0]) != TCL_OK) {
      goto error;
    }

      /* and either a name or numeric id */

    if (parseNumeric (interp, mask, listArgv[1]) != TCL_OK) {
      mask -> id -> name = my_strcpy (listArgv[1]);
      Tcl_ResetResult (interp);
    }

  } else if (listArgc == 1) {
  
      /* one argument means that we better have "ANY" */

    if (strcmp(listArgv[0], "ANY")) {
      Tcl_AppendResult (interp, "underspecified source", (char *) NULL);
      goto error;
    }

  } else {

      /* otherwise the format is wrong */

    Tcl_AppendResult (interp, "must be \"SERVER IP NAME ID ?-tag TAG? ?-handler HANDLER\"?", (char *) NULL);
    goto error;
  } 

    /* success */

  delete listArgv;
  return mask;

 error:

    /* error */

  delete listArgv;
  delete mask;
  return NULL;
}

static char *agentMaskTrace (ClientData clientData, Tcl_Interp *interp, char *name1, char *name2, int flags)
{
  int i;
  int handle;
  char *value;
  MASK_SET *mask_set = (MASK_SET *) clientData;

  if (mask_set -> is_mod_allowed ()) {
    return NULL;
  }

    /* check mask handle on a WRITE */

  if (flags & TCL_TRACE_WRITES) {

    value = Tcl_GetVar2 (interp, name1, name2, TCL_GLOBAL_ONLY);

    if ((Tcl_GetInt (interp, value, &handle) != TCL_OK) || (handle < 0)) {
      Tcl_ResetResult (interp);
      mask_set -> refresh ();
      return "invalid handle";
    }

    for (i = 0; i < NUM_ITEMS; i++) {
      if (!strcmp(name2, maskName[i])) {
        break;
      }
    }

    if (i == NUM_ITEMS) {
      Tcl_UnsetVar2 (interp, name1, name2, TCL_GLOBAL_ONLY);
      return "invalid array element";
    }

    if (mask_set -> update_mask (i, handle) < 0) {
      mask_set -> refresh ();
      return "no mask with that handle";
    }

    return NULL;
  }

    /* restore the array on an UNSET */

  if (!(flags & TCL_INTERP_DESTROYED)) {

    mask_set -> refresh ();
    Tcl_GlueVar2 (interp, "mask", NULL, TCL_GLOBAL_ONLY);

    if (flags & TCL_TRACE_DESTROYED) {
      int trace_flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;
      Tcl_TraceVar2 (interp, "mask", NULL, trace_flags, agentMaskTrace, (ClientData) mask_set);
    }
  }

  return NULL;
}

/* MASK_SET::setEvents

   Purpose: Specify the procedures that will be called if (1) an item arrives
            and there is an event handler specified in the mask or (2) the
            connection to the server breaks

     Input: arrivalProc = procedure to be called on arrival
            brokenProc  = procedure to be called on broken connection

    Output: The procedure records the procedures.
*/

void MASK_SET::setEvents (MaskEventProc *arrivalProc, MaskEventProc *brokenProc)
{
  MASK_SET::arrivalProc = arrivalProc;
  MASK_SET::brokenProc  = brokenProc;
}


/* MASK_SET::queue_all_events

   Purpose: Queue all event handlers that are defined for a queue of incoming
            items

     Input: which = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM

    Output: The procedure queues the event handlers.
*/

void MASK_SET::queue_all_events (int which)
{
  register int i;
  sigset_t oldMask;
  sigset_t newMask;
  register int stop;
  register INCOMING *item;

    /* assertions on the parameters */

  assert ((which == MESSAGE_ITEM) || (which == MEETING_ITEM) || (which == EVENT_ITEM));

    /* block SIGIO */

  sigemptyset (&oldMask);
  sigemptyset (&newMask);
  sigaddset (&newMask, SIGIO);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

    /* check that we have a handler proc and a mask with entries */

  if ((arrivalProc != NULL) && (masks[which] != NULL) && (masks[which] -> type != MASK_ACCEPT_NONE)) { 

      /* look for event handlers */

    stop = queues[which].get_count ();

    for (i = 0; i < stop; i++) {

      item = (INCOMING *) queues[which].dequeue();

      if (!queue_events (which, item)) {
        queues[which].enqueue ((void *) item);
      } 
    }
  }

    /* unblock SIGIO */

  sigprocmask (SIG_SETMASK, &oldMask, NULL);
}

/* MASK_INT::set_handler

   Purpose: Set the default event handler

     Input: handlerName = name of the default event handler

    Output: The procedure sets the default event handler.
*/

void MASK_INT::set_handler (char *handlerName)
{
  register int i;
  sigset_t newMask;
  sigset_t oldMask;

    /* block SIGIO if this mask is the current message, meeting or event  */
    /* mask -- i.e. block SIGIO if it is possible for the SIGIO interrupt */
    /* handler to indirectly access this mask                             */

  if (flags != 0) {
    sigemptyset (&oldMask);
    sigemptyset (&newMask);
    sigaddset (&newMask, SIGIO);
    sigprocmask (SIG_BLOCK, &newMask, &oldMask);
  }

    /* change the default handler */

  delete_check (handler);
  handler = my_strcpy (handlerName);

  if (flags != 0) {

      /* unblock SIGIO if necessary */

    sigprocmask (SIG_SETMASK, &oldMask, NULL);

      /* queue any event handlers that can now fire */

    for (i = 0; i < NUM_ITEMS; i++) {
      if (set -> masks[i] == this) {
        set -> queue_all_events (i);
      }
    }
  }
}
 
/* MASK_INT::get_handler

   Purpose: Get the default event handler

     Input: None

    Output: The procedure returns a pointer to a dynamically allocated string
            the contains the name of the default event handler.  The procedure
            returns NULL if there is no default event handler.
*/

char *MASK_INT::get_handler (void)
{
  return (my_strcpy (handler));
}

/* MASK_SET::get_event

   Purpose: Get the next event from the event queue

     Input: previous = previous MaskArrivalData

    Output: If data is NULL, the procedure returns a pointer to the next
            MaskArrivalData structure in the event queue and removes that 
            structure from the queue.  If data is not NULL, the procedure
            returns and removes the next MaskArrivalData structure ONLY IF
            it refers to the same incoming item as data.  The procedure
            returns NULL otherwise. 
*/

MaskArrivalData *MASK_SET::get_event (MaskArrivalData *previous)
{
  sigset_t oldMask;
  sigset_t newMask;
  MaskArrivalData *next;

    /* block SIGIO */

  sigemptyset (&oldMask);
  sigemptyset (&newMask);
  sigaddset (&newMask, SIGIO);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

    /* no previous MaskArrivalData */

  if (previous == NULL) {

    next = (MaskArrivalData *) eventQueue.dequeue ();

  } else if ((next = (MaskArrivalData *) eventQueue.peek(0)) != NULL) {

    if (previous -> item == next -> item) {
      next = (MaskArrivalData *) eventQueue.dequeue ();
    } else {
      next = NULL;
    } 
  }

    /* unblock SIGIO and return */

  sigprocmask (SIG_SETMASK, &oldMask, NULL);
  return (next);
}

/* MASK_SET::sigio_on

   Purpose: This procedure sets up the SIGIO handler which automatically
            retrieves incoming items from the agent server and adds this
            item to the appropriate queue in MASK_SET.

     Input: None

    Output: The procedure returns -1 if it is unable to contact the agent
            server and set up the SIGIO handler.  Otherwise the procedure
            returns 0.
*/

int MASK_SET::sigio_on (AGENT_ID *id) 
{
  sigset_t oldMask;     /* old interrupt mask */
  sigset_t newMask;     /* new interrupt mask */

    /* procedure should only be called if there is no SIGIO handler */
  
  assert (handler == NULL);

    /* block SIGIO */

  sigemptyset (&oldMask);
  sigemptyset (&newMask);
  sigaddset (&newMask, SIGIO);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

    /* create the instace of the SIGIO_HANDLER */

  handler = new SIGIO_HANDLER (maskSigioTrap, (SigioClientData) this);

    /* contact the agent server and set up the socket down which all */
    /* incoming items will be sent (the TRUE argument                */
    /* to message_conn_and_send turns on the SIGIO signal for the    */
    /* socket so that we receive aysnchronous notification of        */
    /* incoming items).                                              */

  MESSAGE mesg_to_server (up_socket.messages[REQ_GET]);
  mesg_to_server.elements[0].number = id -> id;

  if ((sockfd = message_conn_and_send (NULL, id -> ip, mesg_to_server, -1, TRUE)) < 0) {
    return -1;
  }

    /* unblock SIGIO */

  sigprocmask (SIG_SETMASK, &oldMask, NULL);
  return 0;
}

/* MASK_SET::sigio_off

   Purpose: Turn off the SIGIO handler

     Input: None

    Output: The procedure turns off the SIGIO handler.
*/

void MASK_SET::sigio_off (void)
{
  char buf = 0;
  register int i;
  sigset_t newMask;
  sigset_t oldMask;     
  register INCOMING *item;

    /* handler might have already been deleted */

  if (handler == NULL) {
    return;
  }

    /* block SIGIO */

  sigemptyset (&oldMask);
  sigemptyset (&newMask);
  sigaddset (&newMask, SIGIO);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

    /* write the done flag to the socket, delete the handler, close the socket */

  tcpip_writen (sockfd, &buf, 1);
  delete (handler);
  handler = NULL;
  close (sockfd);
  sockfd = -1;

    /* remove all entries from the queues */

  eventCount = 0;

  while ((item = (INCOMING *) eventQueue.dequeue ()) != NULL) {
    /* empty */
  }

  for (i = 0; i < NUM_ITEMS; i++) {
    while ((item = (INCOMING *) queues[i].dequeue ()) != NULL) {
      /* empty */
    }
  }    

    /* unblock SIGIO */

  sigprocmask (SIG_SETMASK, &oldMask, NULL);
}

/* MASK_SET::addItemToQueue

   Purpose: Add an incoming item to one of the internal queues

     Input: which = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM
            item  = the incoming item

    Output: The procedure adds the incoming item to the specified queue.

      Note: This procedure assumes that SIGIO is blocked.
*/

void MASK_SET::addItemToQueue (INCOMING *item)
{
  int which; 

    /* assertions on the input parameters */

  assert (item != NULL);

    /* fire an event handler if there is a handler defined in the mask */

  which = item -> type;

  if (!queue_events (which, item)) {
    queues[which].enqueue ((void *) item);
  }
} 
    
/* MASK_SET::check_queues

   Purpose: Check the internal queues for items that match one of the masks

     Input: which = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM
 
    Output: The procedure returns INVALID_MASK if the specified mask does not
            exist and INVALID_CONNECTION if the connection to the server has
            been broken.  Otherwise the procedure returns 0 and sets *item to 
            NULL if there if no item that matches the mask OR a pointer to a 
	    dynamically allocated INCOMING structure if there is an item that 
	    matches the mask.
*/

int MASK_SET::check_queues (int which, INCOMING **item)
{
  register int i;
  sigset_t oldMask;
  sigset_t newMask;
  register int total;
  register int first = TRUE;
  register INCOMING *incoming;
  register RESTRICT *restrict = RESTRICT::get_restrict (mask_interp);

    /* assertions on the input parameters */

  assert ((which == MESSAGE_ITEM) || (which == MEETING_ITEM) || (which == EVENT_ITEM));

    /* make sure that the specified mask exists */

  if (masks[which] == NULL) {
    return INVALID_MASK;
  }

    /* assume that no item makes it through the mask */

  *item = NULL;

    /* block SIGIO and SIGALRM */

  sigemptyset (&oldMask);
  sigemptyset (&newMask);
  sigaddset   (&newMask, SIGIO);
  sigaddset   (&newMask, SIGPROF);
  sigaddset   (&newMask, SIGALRM);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

    /* loop since we may be blocking */

  while (1) {   

      /* receive any incoming items; bail om broken server connection */

    if (receiveIncomingItems () < 0) {
      return INVALID_CONNECTION;
    }

      /* check every item in the queue */

    if (hasArrived() || first) {
     
      total = queues[which].get_count ();

         /* check each item */

      for (i = 0; i < total; i++) {

        incoming = (INCOMING *) queues[which].dequeue ();

        if (masks[which] -> filter(incoming)) {
          *item = incoming;
          break;
        }

        queues[which].enqueue ((void *) incoming);
      }

        /* move unchecked items to end of queue */

      for (i = i + 1; i < total; i++) {
        queues[which].enqueue (queues[which].dequeue ());
      }

        /* done if we have an item */

      if (*item != NULL) {
        break;
      }
    }

       /* check for permit violation */
  
    if (restrict -> hasFired() || first) { 
      if (restrict -> permitViolation ()) {
        break;
      } else {
        first = FALSE;
      }
    }

      /* wait for a SIGIO or SIGALRM */

    sigsuspend (&oldMask);
  }

     /* unblock SIGIO and SIGALRM and return */

  sigprocmask (SIG_SETMASK, &oldMask, NULL);
  return 0;
}

/* maskAsyncBroken

   Purpose: This procedure is called when the connection to the server breaks
            AS SOON AS the Tcl interpreter is in a safe state.  The procedure
            then just calls the specified user procedure.

     Input: clientData = the MASK_SET structure
            interp     = the Tcl interpreter
            code       = the Tcl result code

    Output: The procedure calls the specified user procedure.
*/

int maskAsyncBroken (ClientData clientData, Tcl_Interp *interp, int code)
{
  Tcl_Interp *maskInterp = (Tcl_Interp *) clientData;

    /* MASK_SET */

  MASK_SET *maskSet = MASK_SET::get_mask_set (maskInterp);

    /* assert that brokenProc is defined */

  assert (maskSet -> brokenProc != NULL);

    /* call brokenProc */

  (void) (maskSet -> brokenProc)(maskInterp, maskSet);
  return (code);
}

/* maskAsycEvent

   Purpose: This procedure is called when an item arrives AS SOON AS the
            interpreter is in a safe state.  The procedure then calls the
            specified user procedure for any item that has an event handler
            defined in the mask.

     Input: clientData = the MASK_SET structure
            interp     = the Tcl interpreter
            code       = the Tcl result code

    Output: The procedure calls the specified user procedure for any item
            that has an event handler defined in the mask.
*/

int maskAsyncEvent (ClientData clientData, Tcl_Interp *interp, int code)
{
  MASK_SET *maskSet;
  Tcl_Interp *maskInterp;

    /* MASK_SET */

  maskInterp = (Tcl_Interp *) clientData;
  maskSet = MASK_SET::get_mask_set (maskInterp);

    /* assert that arrivalProc is defined */

  assert (maskSet != NULL);
  assert (maskSet -> arrivalProc != NULL);

    /* call arrivalProc for each event */

  while (maskSet -> eventCount > 0) {

      /* we now have one less event */

    maskSet -> eventCount -= 1;

      /* call the handler */

    (void) (maskSet -> arrivalProc)(maskInterp, maskSet);
  }

  return (code);
}

/* maskArrivalAsync

   Purpose: This procedure is called when there is one or more items
	    available on the server connection (or when the server connection
	    is broken).

     Input: clientData = the MASK_SET structure
			 (class MASK_SET *)

            interp     = the Tcl interpreter
			 (struct Tcl_Interp *)

            code       = the Tcl result code
			 (int)

    Output: The procedure receives any available items.
*/

int maskAsyncArrival (ClientData clientData, Tcl_Interp *interp, int code)
{
  MASK_SET *maskSet;
  Tcl_Interp *maskInterp = (Tcl_Interp *) clientData; 

    /* MASK_SET */

  maskSet = MASK_SET::get_mask_set (maskInterp);
  assert (maskSet != NULL);

    /* receive any incoming events */

  (void) maskSet -> receiveIncomingItems();
  return (code);
} 

/* maskSigioTrap

   Purpose: This procedure is called for each MASK_SET instance whenever
            SIGIO fires.

     Input: data = pointer to a MASK_SET instace

    Output: The procedure checks the socket for an incoming item and adds the
            item to the appropriate queue.

      NOTE: This procedure assumes that SIGIO is blocked.
*/

void maskSigioTrap (SigioClientData data)
{
  MASK_SET *maskSet;

    /* MASK_SET */

  maskSet = (MASK_SET *) data;
  assert (maskSet != NULL);

    /* mask the Tcl arrival handler */

  Tcl_AsyncMark (maskSet -> arrivalAsync);
}

/* MASK_SET::receiveIncomingItems

   Purpose: Receive any incoming items off the server connection

     Input: None

    Output: The procedure receives all incoming items off the server
	    connection and adds these items to the appropriate internal
	    queue.  The procedure returns 0 on success and -1 if any
	    connection error occurs.
*/
 
int MASK_SET::receiveIncomingItems (void)
{
  int rc = 0;
  int selectCode;
  fd_set readSet;
  MESSAGE *message;
  struct timeval waitTime;

    /* return immediately on broken connection */

  if (sockfd < 0) {
    return (-1);
  }

    /* file descriptor of interest */

  FD_ZERO (&readSet);
  FD_SET (sockfd, &readSet);

    /* poll rather than block */

  waitTime.tv_sec  = 0;
  waitTime.tv_usec = 0;

    /* handle all incoming messages */

  while ((selectCode = select (sockfd + 1, &readSet, NULL, NULL, &waitTime)) > 0) {

    if ((message = message_receive (sockfd, down_socket)) == NULL) {

      if (brokenProc) {
        Tcl_AsyncMark (brokenAsync);
      } 

      sigio_off ();
      rc = -1;
      break;

    } else {

      receiveIncomingItem (message);
      delete message;
    }           
  }

  if (selectCode < 0) {
  
      /* THERE WILL EVENTUALLY BE ERROR HANDLING CODE HERE */

    rc = -1;
  }

   return (rc);
}

/* MASK_SET::receiveIncomingItem

   Purpose: Receive a single incoming item

     Input: message = MESSAGE structure that contains the incoming item
		      (class MESSAGE *)

    Output: The procedure receives the incoming item and puts it into the
	    appropriate queue.
*/

void MASK_SET::receiveIncomingItem (MESSAGE *message)
{
  if (message -> flag == RESP_MESSAGE) {

    AGENT_ID *id;                  
    INCOMING_MESSAGE *inc_message;
  
      /* break apart an incoming message */
     
    char *src_server = message -> elements[0].string; 
    UINT_32 src_ip   = message -> elements[1].number;
    char *src_name   = message -> elements[2].string;
    UINT_32 src_id   = message -> elements[3].number; 
    UINT_32 code     = message -> elements[4].number;
    char *string     = message -> elements[5].string;

      /* construct the INCOMING_MESSAGE structure */

    id = new AGENT_ID (src_server, src_ip, src_name, src_id);
    inc_message = new INCOMING_MESSAGE (id, code, string); 

      /* add the message to the appropriate queue */
 
    addItemToQueue (inc_message);
    arrived = TRUE;

  } else if (message -> flag == RESP_MEETING) {

    AGENT_ID *id;             
    REMOTE_PORT *port;              
    INCOMING_MEETING *inc_meeting; 

      /* break out the meeting request */

    char *  src_server = message -> elements[0].string;
    UINT_32 src_ip     = message -> elements[1].number;        
    char *  src_name   = message -> elements[2].string;
    UINT_32 src_id     = message -> elements[3].number;
    UINT_8  status     = message -> elements[4].number;
    char *  act_server = message -> elements[5].string;
    UINT_32 act_ip     = message -> elements[6].number;
    UINT_32 act_port   = message -> elements[7].number;

      /* construct the INCOMING_MEETING structure */

    port = new REMOTE_PORT (act_server, act_ip, act_port);
    id = new AGENT_ID (src_server, src_ip, src_name, src_id);
    inc_meeting = new INCOMING_MEETING (id, status, port);

      /* add the meeting to the appropriate queue */

    addItemToQueue (inc_meeting);
    arrived = TRUE;

  } else if (message -> flag == RESP_EVENT) {

    AGENT_ID *id;    
    INCOMING_EVENT *inc_event; 

      /* break out the meeting request */

    char *  src_server = message -> elements[0].string;
    UINT_32 src_ip     = message -> elements[1].number;        
    char *  src_name   = message -> elements[2].string;
    UINT_32 src_id     = message -> elements[3].number;
    char *tag          = message -> elements[4].string;
    char *string       = message -> elements[5].string;

      /* construct the INCOMING_EVENT structure */

    id = new AGENT_ID (src_server, src_ip, src_name, src_id);
    inc_event = new INCOMING_EVENT (id, tag, string);

      /* add the meeting to the appropriate queue */

    addItemToQueue (inc_event);
    arrived = TRUE;

  } else {

    /* THERE WILL EVENTUALLY BE ERROR HANDLING CODE HERE */
  }
}

/* MASK_SET::hasArrived

   Purpose: See if an item has arrived since the last call to hasArrived()

     Input: None

    Output: The procedure returns TRUE if an item has arrived since the last
            call to hasArrived().  Otherwise the procedure returns FALSE.

      NOTE: SIGIO should be blocked when this procedure is called.
*/

int MASK_SET::hasArrived (void)
{
  int flag;
  sigset_t mask;

    /* assertions on signal mask */

  assert ((sigprocmask (0, NULL, &mask), sigismember(&mask, SIGIO)));

    /* get the arrived flag */

  flag = arrived;
  arrived = FALSE;
  return (flag);
}
