/* Agent Tcl
   Bob Gray
   30 August 1995

   tclMask.h

   This file defines the internal routines that handle the message, meeting 
   and event masks.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _TCL_MASK_H
#define _TCL_MASK_H

#include <sys/types.h>
#include "map.h"
#include "message.h"
#include "queue.h"
#include "sigio.h"
#include "tcl.h"
#include "tclAgent.h"
#include "tclRestrictInt.h"
#include "truefalse.h"

class MASK_SET;

  /* flag bits */

const MEETING_BIT	= 0x1;
const MESSAGE_BIT	= 0x2;
const EVENT_BIT		= 0x4;

  /* error codes */

const INVALID_MASK	 = -1;
const INVALID_CONNECTION = -2;

  /* structure that holds the data associated with an event */

struct MaskArrivalData
{
  INCOMING *item;      /* the incoming item         */
  char *handler;       /* name of the event handler */

     /* constructor and destructor */

  MaskArrivalData (INCOMING *item, char *handler);
 ~MaskArrivalData ();
};
 
  /* procedure to call when an event needs to be fired */

typedef int (MaskEventProc) 
	(Tcl_Interp *interp, MASK_SET *maskSet);
 
  /* SIGIO handlers */

void maskSigioTrap 
	(SigioClientData data);

int maskAsyncArrival
	(ClientData clientData, Tcl_Interp *interp, int code);

int maskAsyncBroken
	(ClientData clientData, Tcl_Interp *interp, int code);

int maskAsyncEvent
	(ClientData clientData, Tcl_Interp *interp, int code);

  /* MASK_INT is an internal representation of a list of masks */

class MASK_INT
{
   friend class MASK;
   friend class MASK_SET;

   int type;       /* MASK_ACCEPT_NONE, MASK_ACCEPT_SOME or MASK_ACCEPT_ALL       */
   int flags;      /* OR'ed combination of MEETING_BIT, MESSAGE_BIT and EVENT_BIT */
   int handle;     /* mask handle                                                 */
   MASK_SET *set;  /* MASK_SET structure that contains this mask                  */
   char *handler;  /* default event handler                                       */
   QUEUE list;     /* the masks                                                   */

     /* match two mask entries when removing */

   int match_exact (MASK_ENTRY *id, MASK_ENTRY *maskId);

     /* empty the mask and set the flag */

   void empty_and_set (int type);

 public:

     /* match a mask entry and an incoming item */

   int match_inexact (MASK_ENTRY *maskExt, INCOMING *item);

     /* get the mask handle */

   inline int get_handle (void) { return handle; };

     /* add and remove mask entries */

   void add (int count, MASK_ENTRY **mask);
   void remove (int count, MASK_ENTRY **masks);

     /* get and set the default handler */

   void set_handler (char *handlerName);
   char *get_handler (void);

     /* fill and empty the mask */

   inline void fill  (void) { empty_and_set (MASK_ACCEPT_ALL ); };
   inline void empty (void) { empty_and_set (MASK_ACCEPT_NONE); };

     /* filter an incoming item through the mask */

   int filter (INCOMING *item);

     /* constructor */

   MASK_INT (int type, int handle, MASK_SET *set);
  ~MASK_INT ();
};

  /* MASK_SET is the set of masks defined in a given interpreter */

class MASK_SET
{
    friend class MASK_INT;

    friend void maskSigioTrap 
	(SigioClientData data);

    friend int Agent_Select 
	(Tcl_Interp *interp, double seconds, int n, int *fd, int *ready);

    friend int maskAsyncArrival
	(ClientData clientData, Tcl_Interp *interp, int code);

    friend int maskAsyncBroken
	(ClientData clientData, Tcl_Interp *interp, int code);

    friend int maskAsyncEvent
	(ClientData clientData, Tcl_Interp *interp, int code);

      /* map interpreter addresses to corresponding MASK_SET */

    static POINTER_MAP *interp_map;

      /* message, meeting and event masks and the corresponding queues */

    MASK_INT *masks[NUM_ITEMS];
    QUEUE queues [NUM_ITEMS];

      /* the event queue and the number of items that have been added to */
      /* the event queue since the last call to arrivalHand              */

    QUEUE eventQueue;
    int eventCount;

      /* TRUE if a new item has arrived since the last called to hasArrived */

    int arrived;

      /* asynchronous handlers for arrival, event handling and broken */
      /* connection						      */

    Tcl_AsyncHandler arrivalAsync;   /* Tcl async handler for arrivals          */
    Tcl_AsyncHandler brokenAsync;    /* Tcl async handler for broken connection */
    Tcl_AsyncHandler eventAsync;     /* Tcl async handler for event handling    */

      /* user-specified procedures to be called for events and broken connections */

    MaskEventProc *arrivalProc;   /* procedure to be called on arrival           */
    MaskEventProc *brokenProc;    /* procedure to be called on broken connection */

      /* miscellaneous */

    int sockfd;		          /* socket descriptor for incoming messages     */ 
    int mod_allowed;              /* TRUE if we are reloading the mask array     */
    Tcl_Interp *mask_interp;      /* interpreter associated with this instance   */
    Tcl_HashTable mask_table;     /* hash table that contains the masks          */
    int mask_handle;              /* the next untried mask handle                */
    int mask_count;               /* the number of masks                         */
    SIGIO_HANDLER *handler;       /* SIGIO handler                               */
    UINT_32 actualhost;           /* IP address of the current host              */

      /* queue all event handlers that are defined for an incoming item */

    int queue_events (int which, INCOMING *item);

      /* queue all event handlers that are defined for a queue of incoming items */

    void queue_all_events (int which);

      /* put an incoming item into the appropriate queue */

    void addItemToQueue (INCOMING *item);

      /* receive any incoming items */

    int receiveIncomingItems (void);

      /* receive a single incoming item */

    void receiveIncomingItem (MESSAGE *message);

  public:

      /* get the MASK_SET structure associated with a given interpreter */
 
    static MASK_SET *get_mask_set (Tcl_Interp *interp);

      /* get the interpreter associated with this MASK_SET structure */

    inline Tcl_Interp *get_interp (void) { return mask_interp; };

      /* returns TRUE of a new item has arrived since the last call */

    int hasArrived (void);

      /* reload the mask array */

    void refresh (void);

      /* return TRUE if we are reloading the mask array */

    inline int is_mod_allowed (void) { return mod_allowed; };

      /* set the Arrival and Broken handlers */

    void setEvents (MaskEventProc *arrivalProc, MaskEventProc *brokenProc);
 
      /* create a new mask, find a mask and remove a mask */

    MASK_INT *new_mask (int flag);
    MASK_INT *lookup_mask (int handle);
    int remove_mask (int handle);
 
      /* update a special mask, check the queues for items that match */
      /* one of the special masks                                     */

    int update_mask (int which, int handle);
    int check_queues (int which, INCOMING **incoming);

      /* get the next event from the event queue */

    MaskArrivalData *get_event (MaskArrivalData *data);

      /* turn SIGIO interrupt on and off */
 
    int sigio_on (AGENT_ID *id);
    void sigio_off (void);

      /* constructor and destructor */

    MASK_SET (Tcl_Interp *interp, UINT_32 actualhost);
   ~MASK_SET ();
};

#endif 
