/* Agent Tcl
   Bob Gray
   30 August 1995

   tclMask.cc

   This file implements the Tcl commands that handle the message, meeting 
   and event masks.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tclAgent.h"
#include "tclAgentInt.h"
#include "tclMask.h"

/* maskRemoveCmd

   Tcl syntax: mask remove <handle> [ ALL | <agent_id> ...]

      Purpose: Remove an entry from a mask

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
               to the empty string.
*/

int maskRemoveCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int i;
  int count;                  /* number of masks to be removed */
  int handle;                 /* mask handle                   */
  MASK_ENTRY **masks;           /* masks to be removed           */
  int returnCode = TCL_OK;    /* return value                  */

    /* check the number of arguments */

  if (argc < 4) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " add <handle> [ ALL | <agent_id> ... ]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the handle */

  if ((Tcl_GetInt (interp, argv[2], &handle) != TCL_OK) || (handle < 0)) {
    Tcl_SetResult (interp, "invalid mask handle: must be an integer 0 or greater", TCL_STATIC);
    return TCL_ERROR;
  }

    /* ALL means remove every entry from the mask */

  if (!strcmp(argv[3], "ALL")) {

      /* ALL should not take any arguments */

    if (argc != 4) { 
      Tcl_AppendResult (interp, "\"ALL\" must appear alone", (char *) NULL);
      return TCL_ERROR;
    }

      /* remove every mask */

    return (Agent_MaskEmpty (interp, handle));
  }

    /* remove the masks listed in the command */

  argc -= 3;
  argv += 3;
  count = 0;
  masks = new MASK_ENTRY * [argc];

  for (i = 0; i < argc; i++) {

      /* get each mask */

    if ((masks[i] = Agent_SplitMask (interp, argv[i])) == NULL) {
      Tcl_AppendResult (interp, ": invalid mask entry \"", argv[i], "\"", (char *) NULL);
      returnCode = TCL_ERROR;
      break;
    }

    count += 1;
  }

    /* remove the masks if each mask was specified correctly */

  if (returnCode == TCL_OK) {
    returnCode = Agent_MaskRemove (interp, handle, count, masks);
  }

    /* cleanup and return */

  for (i = 0; i < count; i++) {
    delete (masks[i]);
  }

  return (returnCode);
}

/* maskAddCmd

   Tcl syntax: mask add <handle> [ ALL | <agent_id> ...]

      Purpose: Add an entry to a mask

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
	       to the empty string.
*/

int maskAddCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int i;
  int count;                 /* number of masks to be added */
  int handle;                /* mask handle                 */
  MASK_ENTRY **masks;          /* masks to be added           */
  int returnCode = TCL_OK;   /* return value                */

    /* check the number of arguments */

  if (argc < 4) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " add <handle> [ ALL | <agent_id> ... ]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the handle */

  if ((Tcl_GetInt (interp, argv[2], &handle) != TCL_OK) || (handle < 0)) {
    Tcl_SetResult (interp, "invalid mask handle: must be an integer 0 or greater", TCL_STATIC);
    return TCL_ERROR;
  }

    /* ALL means add every possible mask */

  if (!strcmp(argv[3], "ALL")) {

      /* ALL means that there should be no other arguments */

    if (argc != 4) { 
      Tcl_AppendResult (interp, "\"ALL\" must appear alone", (char *) NULL);
      return TCL_ERROR;
    }

      /* add every possible mask */

    return (Agent_MaskFill (interp, handle));
  }

    /* add the masks listed in the command */

  argc -= 3;
  argv += 3;
  count = 0;
  masks = new MASK_ENTRY * [argc];

  for (i = 0; i < argc; i++) {

      /* get each mask */

    if ((masks[i] = Agent_SplitMask (interp, argv[i])) == NULL) {
      Tcl_AppendResult (interp, ": invalid mask entry \"", argv[i], "\"", (char *) NULL);
      returnCode = TCL_ERROR;
      break;
    }

    count += 1;
  }

    /* add the masks if each mask was obtained correctly */

  if (returnCode == TCL_OK) {
    returnCode = Agent_MaskAdd (interp, handle, count, masks);
  }

    /* cleanup and return */

  for (i = 0; i < count; i++) {
    delete (masks[i]);
  }

  return (returnCode);
}

/* maskDisplayCmd

   Tcl syntax: mask display <handle>

      Purpose: Display the contents of a mask

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
	       to a list of the mask contents.
*/

int maskDisplayCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  MASK *mask;
  int handle;
  Tcl_DString *string;

    /* check the number of arguments */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " display <handle>\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the mask handle */

  if ((Tcl_GetInt (interp, argv[2], &handle) != TCL_OK) || (handle < 0)) {
    Tcl_SetResult (interp, "invalid mask handle: must be an integer 0 or greater", TCL_STATIC);
    return TCL_ERROR;
  }

    /* get the contents of the mask */

  if ((mask = Agent_MaskContents (interp, handle)) == NULL) {
    return TCL_ERROR;
  }

    /* turn the contents into a human readable list */

  if (mask -> type == MASK_ACCEPT_ALL) {
    Tcl_SetResult (interp, "ALL", TCL_STATIC);
  } else if (mask -> type == MASK_ACCEPT_NONE) {
    Tcl_SetResult (interp, "NONE", TCL_STATIC);
  } else {
    for (int i = 0; i < mask -> count; i++) {
      string = Agent_MaskToString (interp, mask -> masks[i]);
      Tcl_AppendElement (interp, Tcl_DStringValue(string));
      Tcl_DStringFree (string);
      delete string;
    } 
  }

  delete (mask);
  return TCL_OK;
}

/* maskDeleteCmd 

   Tcl syntax: mask delete [ ALL | <handle> ]

      Purpose: Delete a mask

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter 
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
               to the empty string.
*/

int maskDeleteCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int handle;

    /* check the number of arguments */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " delete [ ALL | <handle> ]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the mask handle */

  if (!strcmp(argv[2], "ALL")) {
    handle = ALL_MASKS;
  } else if ((Tcl_GetInt(interp, argv[2], &handle) != TCL_OK) || (handle < 0)) {
    Tcl_SetResult (interp, "invalid mask handle: must be \"ALL\" or an integer 0 or greater", TCL_STATIC);
    return TCL_ERROR;
  }

    /* delete the mask */

  return (Agent_MaskDelete (interp, handle));
}

/* maskNewCmd

   Tcl syntax: mask new

      Purpose: Create a new mask

        Input: dummy  = client data (unused)
               interp = the current interpreter 
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result 
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and sets the interpreter result to the
               mask handle.
*/

int maskNewCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int handle;
  char scratch[16];

    /* check the number of arguments */

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " new\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* create the new mask */

  if ((handle = Agent_MaskNew (interp)) < 0) {
    return TCL_ERROR;
  }

    /* set the interpreter result to the mask handle */

  sprintf (scratch, "%d", handle);
  Tcl_SetResult (interp, scratch, TCL_VOLATILE);
  return TCL_OK; 
}

/* maskHandlerCmd

   Tcl syntax: mask handler <handle>
               mask handler <handle> <handler>

      Purpose: Associate a default event handler with a mask or determine the
               current default handler

        Input: dummy  = client data (unused)
               interp = the current Tcl interpreter
               argc   = the number of command arguments
               argc   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and sets the interpreter result to the
               new or current handler (depending on whether we are setting or
               checking the handler).  The procedure sets the interpreter
               result to the empty string if there is no current handler.
*/

int maskHandlerCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int handle;
  char *handleName;

    /* check the number of arguments */

  if ((argc != 3) && (argc != 4)) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " handler <handle> ?handler?\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the handle */

  if ((Tcl_GetInt (interp, argv[2], &handle) != TCL_OK) || (handle < 0)) {
    Tcl_SetResult (interp, "invalid mask handle: must be an integer 0 or greater", TCL_STATIC);
    return TCL_ERROR;
  }

    /* are we checking the default handler */

  if (argc == 3) {
    if (Agent_MaskGetDefault (interp, handle, &handleName) != TCL_OK) {
      return TCL_ERROR;
    } else if (handleName == NULL) {
      Tcl_ResetResult (interp);
      return TCL_OK;
    } else {
      Tcl_SetResult (interp, handleName, TCL_VOLATILE);
      delete (handleName);
      return TCL_OK;
    }
  }

    /* or are we setting the default handler */

  if (Agent_MaskSetDefault (interp, handle, argv[3]) != TCL_OK) {
    return TCL_ERROR;
  } else {
    Tcl_SetResult (interp, argv[3], TCL_VOLATILE);
    return TCL_OK;
  }
}

/* maskNohandlerCmd

   Tcl syntax: mask nohandle <handle>

      Purpose: Remove the default handler associated with a mask

        Input: dummy  = client data (unused)
               interp = the Tcl interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and removes the default handler
               associated with the mask.
*/

int maskNohandlerCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int handle;

    /* check the number of arguments */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " nohandler <handle>\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the handle */

  if ((Tcl_GetInt (interp, argv[2], &handle) != TCL_OK) || (handle < 0)) {
    Tcl_SetResult (interp, "invalid mask handle: must be an integer 0 or greater", TCL_STATIC);
    return TCL_ERROR;
  }

    /* remove the default handler */

  return (Agent_MaskSetDefault (interp, handle, (char *) NULL));
}



/* Agent_MaskCmd

   Tcl syntax: mask new
	       mask delete    <handle>
	       mask delete    ALL
	       mask display   <handle>
	       mask add       <handle> ALL 
	       mask add       <handle> <agent_id> ...
               mask remove    <handle> ALL
	       mask remove    <handle> <agent_id> ... 
               mask handler   <handle> <handler>
	       mask handler   <handle>
	       mask nohandler <handle>

      Purpose: Create, delete, expand, shrink or associate a default event
               handler with a mask.

        Input: dummy  = client data (unused)
               interp = the current intepreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
	       procedure returns TCL_OK and sets the interpreter result to
               the empty string (except for "mask display" which sets the
               interpreter result to a human-readable version of the mask
	       contents and "mask new" which sets the interpreter result to
               the mask handle).
*/

int Agent_MaskCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{ 
  char *op;
  int returnCode = TCL_OK;

    /* check the number of arguments */

  if (argc <= 1) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " operation [options]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* determine which operation is being performed */

  op = argv[1];
  
  if ((*op == 'a') && !strcmp(op,"add")) {
    returnCode = maskAddCmd (dummy, interp, argc, argv);
  } else if ((*op == 'r') && !strcmp(op,"remove")) {
    returnCode = maskRemoveCmd (dummy, interp, argc, argv);
  } else if ((*op == 'n') && !strcmp(op,"new")) {
    returnCode = maskNewCmd (dummy, interp, argc, argv); 
  } else if ((*op == 'd') && !strcmp(op,"delete")) {
    returnCode = maskDeleteCmd (dummy, interp, argc, argv);
  } else if ((*op == 'd') && !strcmp(op,"display")) {
    returnCode = maskDisplayCmd (dummy, interp, argc, argv);
  } else if ((*op == 'h') && !strcmp(op,"handler")) {
    returnCode = maskHandlerCmd (dummy, interp, argc, argv);
  } else if ((*op == 'n') && !strcmp(op,"nohandler")) {
    returnCode = maskNohandlerCmd (dummy, interp, argc, argv);
  } else {
    Tcl_AppendResult (interp, "invalid operation \"", argv[1], "\": should be \"new\", \"delete\", \"add\", \"remove\", \"display\", \"handler\" or \"nohandler\"", (char *) NULL);
    returnCode = TCL_ERROR;
  }

  return (returnCode);
}
 
