/* Agent Tcl
   Bob Gray
   2 August 1995
 
   tclTime.cc

   This file implements the library routine that puts an agent to sleep.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#ifndef COSMO
#include <unistd.h>
#else
#include </usr/local/lib/gcc-lib/rs6000-ibm-aix3.2.5/2.6.3/include/unistd.h>
#endif
#include "interrupt.h"
#include "tcl.h"
#include "tclAgent.h"
#include "tclAgentInt.h"
#include "tclLocation.h"
#include "tclRestrict.h"
#include "tclRestrictInt.h"
#include "timers.h"
#include "truefalse.h"

/* Agent_Elapsed

   Purpose: Get the number of seconds that have elapsed since the agent
            started executing on the current machine

     Input: interp = the current interpreter

    Output: The procedure returns -1 and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure
            returns the number of seconds that have elapsed since the agent
            started executing on the current machine.
*/

double Agent_Elapsed (Tcl_Interp *interp)
{
  clock_t deltaTime;
  clock_t currentTime;
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);

     /* return the elapsed number of seconds */

   currentTime = TIMERS::getWall ();
   deltaTime   = currentTime - locations -> startTime;
   return ((double) deltaTime / (double) TIMERS::getTicks());
}   

/* Agent_Sleep

   Purpose: Sleep for the specified number of seconds

     Input: interp  = the current interpreter
            seconds = the number of seconds

    Output: The procedure returns TCL_ERROR and sets the interpreter result
            to an appropriate error message on error.  The procedure returns
            TCL_PERMIT if a permit violation occurs.  Otherwise the procedure
            sleeps for the specified number of seconds and returns TCL_OK.
*/

int Agent_Sleep (Tcl_Interp *interp, double seconds)
{
  sigset_t oldMask;
  sigset_t newMask; 
  int first = TRUE;
  RESTRICT *restrict = RESTRICT::get_restrict (interp);

    /* make sure that the number of seconds is nonnegative */

  if (seconds < 0.0) {
    Tcl_AppendResult (interp, "number of seconds must be 0 or greater", (char *) NULL);
    return TCL_ERROR;
  } 

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
   
    /* block SIGALRM and SIGPROF */
  
  sigemptyset (&oldMask);
  sigemptyset (&newMask);
  sigaddset (&newMask, SIGALRM);
  sigaddset (&newMask, SIGPROF);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

  while (1) {

      /* check the restrictions */

    if (restrict -> hasFired() || first) {
      if (restrict -> permitViolation ()) {
        break;
      }
    }

      /* suspend and wait for signals */

    sigsuspend (&oldMask);
  }

    /* cleanup and return */
  
  sigprocmask (SIG_SETMASK, &oldMask, NULL);
  restrict -> remove ();
  return TCL_OK;
} 
