/* Agent Tcl
   Bob Gray
   2 August 1995 

   tclTimeCmd.cc

   This file implements the Tcl command that puts an agent to sleep.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <time.h>
#include "tcl.h"
#include "tclAgent.h"
#include "tclAgent.h"

/* Agent_SleepCmd

   Tcl syntax: agent_sleep seconds

      Purpose: Sleep for a certain of seconds 

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the 
               procedure tries to sleep for the specified number of seconds.
               If no interrupts occur, the procedure sleeps for the full time,
               returns TCL_OK and sets the interpreter result to 0.  If an
               interrupt occurs, the procedure returns TCL_OK and sets the
               interpreter result to the number of seconds left to sleep.
*/

int Agent_SleepCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  double seconds;       /* number of seconds to sleep      */

    /* check the number of command arguments */
  
  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " seconds\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the number of seconds */

  if (Tcl_GetDouble (interp, argv[1], &seconds) != TCL_OK) {
    Tcl_AppendResult (interp, ": number of seconds must be a real number\"", (char *) NULL);
    return TCL_ERROR;
  } 
 
    /* sleep */

  return (Agent_Sleep (interp, seconds)); 
}


/* Agent_ElapsedCmd

   Tcl syntax: agent_elapsed

      Purpose: Get the number of seconds that have elapsed since the
               agent started executing on the current machine

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
               to the empty string.
*/

int Agent_ElapsedCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  char temp[16];
  double seconds;

    /* check the number of command arguments */

  if (argc != 1) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the elapsed time */

  seconds = Agent_Elapsed (interp);

    /* put the elapsed time into the interpreter result */

  sprintf (temp, "%f", seconds);
  Tcl_SetResult (interp, temp, TCL_VOLATILE);
  return TCL_OK;
}
