/* Agent Tcl
   Bob Gray
   27 July 1995

   tclTool.cc

   This file implements the library routine that forcibly terminates an agent
   and that gets information about an agent.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <assert.h>
#include <unistd.h>
#include "fd_utilities.h"
#include "message.h"
#include "my_alloc.h"
#include "tcl.h"
#include "tclAgent.h"
#include "tclAgentInt.h"
#include "tclLocation.h"
#include "tclRestrict.h"
#include "tclRestrictInt.h"
#include "transmit.h"
#include "truefalse.h"

/* agentSpecificInfo

   Purpose: Get information about a particular agent 

     Input: interp  = the current interpreter
            seconds = the timeout interval
            id      = the server or agent identification

    Output: The procedure returns NULL and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure
            returns a pointer to a dynamically allocated AGENT_INFO structure.
*/

static void *agentSpecificInfo (Tcl_Interp *interp, double seconds, AGENT_ID *id)
{
  clock_t stop;
  int sockfd = -1;
  void *list = NULL;
  RESTRICT *restrict;
  MESSAGE *response = NULL;
  AGENT_LOCATION *locations;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (id != NULL);

    /* RESTRICT and AGENT_LOCATION */

  locations = AGENT_LOCATION::get_location (interp);
  restrict = RESTRICT::get_restrict (interp);
  assert (locations != NULL);
  assert (restrict != NULL);

    /* check whether the agent is registered */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return NULL;
  }

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);

    /* send the message */

  MESSAGE message (up_socket.messages[REQ_INFO]);
  message.elements[0].number = INFO_ID;
  message.elements[1].string = id -> name;
  message.elements[2].number = id -> id;

  if ((sockfd = message_conn_and_send (id -> server, id -> ip, message, stop, FALSE)) < 0) {

    Tcl_AppendResult (interp, "unable to send to \"", id -> server, "\"", (char *) NULL); 

  } else if ((response = message_receive (sockfd, down_socket)) == NULL) {

    Tcl_AppendResult (interp, "\"", id -> server, "\" unable to comply (bad response)", (char *) NULL);

  } else if (response -> flag != RESP_INFO) {

    Tcl_AppendResult (interp, "\"", id -> server, "\" unable to comply (server error)", (char *) NULL);

  } else {

      /* break out part of the message */

    char *actual      = response -> elements[0].string;
    UINT_32 actual_ip = response -> elements[1].number;
    char *server      = response -> elements[2].string;
    UINT_32 server_ip = response -> elements[3].number;
    char *name        = response -> elements[4].string;
    UINT_32 id        = response -> elements[5].number;

      /* set up the AGENT_INFO structure */

    AGENT_INFO *info   = new AGENT_INFO;
    info -> id         = new AGENT_ID (server, server_ip, name, id);
    info -> actual     = new MACHINE_ID (actual, actual_ip);
    info -> agentPid   = response -> elements[6].number;
    info -> blockerPid = response -> elements[7].number;
    info -> messages   = response -> elements[8].number;
    info -> meetings   = response -> elements[9].number;
    info -> events     = response -> elements[10].number;
    list               = (void *) info;
  }

    /* remove the permit */

  restrict -> remove ();

    /* cleanup */ 

  delete_check (response);
  close_check (sockfd);
  return (list);
}
 
/* agentInfo

   Purpose: Get the numeric id or symbolic name of every agent registered
            with a particular server

     Input: flag    = INFO_ALL_IDS or INFO_ALL_NAMES
            interp  = the current interpreter
            seconds = the timeout interval
            machine = the server

    Output: The procedure returns NULL and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure
            returns a pointer to either a dynamically allocated ID_LIST
            structure or a dynamically alocated NAME_LIST structure (depending
            on whether flag was INFO_ALL_IDS or INFO_ALL_NAMES). 
*/

static void *agentInfo (UINT_8 flag, Tcl_Interp *interp, double seconds, MACHINE_ID *machine)
{
  clock_t stop;
  int sockfd = -1;
  void *list = NULL;
  RESTRICT *restrict;
  UINT_8 response_flag;
  MESSAGE *response = NULL;
  AGENT_LOCATION *locations;

    /* assertions on the parameters */

  assert ((flag == INFO_ALL_IDS) || (flag == INFO_ALL_NAMES));
  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (machine != NULL);

    /* RESTRICT and AGENT_LOCATION */

  locations = AGENT_LOCATION::get_location (interp);
  restrict = RESTRICT::get_restrict (interp);
  assert (locations != NULL);
  assert (restrict != NULL);

    /* response flag */

  if (flag == INFO_ALL_IDS) {
    response_flag = RESP_IDS;
  } else {
    response_flag = RESP_NAMES;
  }

    /* check whether the agent is registered */

  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has not been registered", (char *) NULL);
    return NULL;
  }

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);

    /* send the message */

  MESSAGE message (up_socket.messages[REQ_INFO]);
  message.elements[0].number = flag;
  message.elements[1].string = NULL;
  message.elements[2].string = 0;

  if ((sockfd = message_conn_and_send (machine -> server, machine -> server_ip, message, stop, FALSE)) < 0) {
    Tcl_AppendResult (interp, "unable to send to \"", machine -> server, "\"", (char *) NULL); 
  } else if ((response = message_receive (sockfd, down_socket)) == NULL) {
    Tcl_AppendResult (interp, "\"", machine -> server, "\" unable to comply (bad response)", (char *) NULL);
  } else if (response -> flag != response_flag) {
    Tcl_AppendResult (interp, "\"", machine -> server, "\" unable to comply (server error)", (char *) NULL);
  } else {
    list = (void *) response -> elements[0].string; 
    response -> elements[0].dealloc = FALSE;
  } 

    /* remove the permit */

  restrict -> remove ();

    /* cleanup */ 

  delete_check (response);
  close_check (sockfd);
  return (list);
} 

/* Agent_InfoNames

   Purpose: Get all of the symbolic names registered with a server

     Input: interp  = the current interpreter
            seconds = the timeout interval
            machine = the server

    Output: The procedure returns NULL and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure
            returns a dynamically allocated NAME_LIST structure that contains
            all of the symbolic names registered with the server.
*/

NAME_LIST *Agent_InfoNames (Tcl_Interp *interp, double seconds, MACHINE_ID *machine)
{
  return ((NAME_LIST *) agentInfo (INFO_ALL_NAMES, interp, seconds, machine));
}

/* Agent_InfoIds

   Purpose: Get the numeric id of every agent registered with a server

     Input: interp  = the current interpreter
            seconds = the timeout interval
            machine = the server

    Output: The procedure returns NULL and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure
            returns a dynamically allocated ID_LIST structure that contains
            the numeric id of every agent registered with the server.
*/

ID_LIST *Agent_InfoIds (Tcl_Interp *interp, double seconds, MACHINE_ID *machine)
{
  return ((ID_LIST *) agentInfo (INFO_ALL_IDS, interp, seconds, machine));
} 
            
/* Agent_Info

   Purpose: Get information about a particular agent

     Input: interp  = the current interpreter
            seconds = the timeout interval
            machine = the server

    Output: The procedure returns NULL and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure
            returns a dynamically allocated AGENT_INFO structure that contains
            the agent information.
*/

AGENT_INFO *Agent_Info (Tcl_Interp *interp, double seconds, AGENT_ID *id)
{
  return ((AGENT_INFO *) agentSpecificInfo (interp, seconds, id));
}

/* Agent_Force

   Purpose: Forcibly terminate an agent

     Input: interp  = the Tcl interpreter
            seconds = number of seconds until timeout (0 or greater)
            id      = agent identification
 
    Output: The procedure returns TCL_ERROR and sets the intrepreter result to 
            an appropriate error message on error.  Otherwise the procedure
            returns TCL_OK and sets forcedId to (1) NULL if there was no agent
            with the given id or (2) a dynamically allocated AGENT_ID
            structure that contains the id of the forced agent.
*/

int Agent_Force (Tcl_Interp *interp, double seconds, AGENT_ID *id, struct AGENT_ID **forcedId)
{
  AGENT_LOCATION *locations;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (id != NULL);

    /* AGENT_LOCATION */

  locations = AGENT_LOCATION::get_location (interp);
  assert (locations != NULL);
 
    /* check whether the agent is registered and whether the agent is toplevel */
 
  if (locations -> registered == AGENT_NOT_REGISTERED) {
    Tcl_AppendResult (interp, "agent has NOT been registered", (char *) NULL);
    return TCL_ERROR;
  }

    /* send the message */

  return (agentForceEnd (interp, seconds, id, forcedId, END_FORCE));
}
