/* Agent Tcl
   Bob Gray
   27 July 1995
  
   tclToolCmd.cc

   This file implements the Tcl command that forcibly terminates an agent.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <string.h>
#include "cmd_utilities.h"
#include "tcl_utilities.h"
#include "message.h"
#include "my_alloc.h"
#include "my_sizes.h"
#include "tcl.h"
#include "tclAgent.h"
#include "tclAgentInt.h"
#include "tcpip.h"

#ifdef AGENT_TOOLS

/* Agent_InfoCmd

   Tcl syntax: agent_info
	       {-ids <machine> | -names <machine> | <identification>} 
	       [-time <seconds>]

      Purpose: Get information about the agents registered with a particular
               server

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and sets the interpreter result to
               a string that describes the agents registered with the server.
*/

int Agent_InfoCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int i = 1;
  UINT_8 flag;
  char temp[16];
  double seconds;
  ID_LIST *idList;
  AGENT_INFO *info;
  Tcl_DString *string;
  NAME_LIST *nameList;
  AGENT_ID *id = NULL;
  int code = TCL_ERROR;
  MACHINE_ID *machine = NULL;

    /* check the number of command arguments */

  if (argc < 2) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " {-ids <machine> | -names <machine> | <identification>} [-time <seconds>]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* parse the type of information */

  if (!strcmp(argv[i], "-ids")) {

    flag = INFO_ALL_IDS;

    if (++i >= argc) {
      Tcl_AppendResult (interp, "\"-ids\" must be followed by a machine name", (char *) NULL);
      return TCL_ERROR;
    }

    machine = new MACHINE_ID (argv[i], UNKNOWN_IP);

  } else if (!strcmp(argv[1], "-names")) {

    flag = INFO_ALL_NAMES;

    if (++i >= argc) {
      Tcl_AppendResult (interp, "\"-names\" must be followed by a machine name", (char *) NULL);
      return TCL_ERROR;
    }

    machine = new MACHINE_ID (argv[i], UNKNOWN_IP);
  
  } else {

    flag = INFO_ID;

    if ((id = Agent_SplitId (interp, argv[i])) == NULL) {
      return TCL_ERROR;
    }
  }

    /* parse the timeout */

  if (++i >= argc) {
    seconds = DEFAULT_CONNECT_SECONDS;
  } else if ((i = parseShortTimeout (interp, seconds, argc, argv, i)) < 0) {
    goto done;
  } else {
    Tcl_AppendResult (interp, "extraneous arguments", (char *) NULL);
    goto done;
  }     

    /* call the appropriate procedure */

  if (flag == INFO_ALL_IDS) {

    if ((idList = Agent_InfoIds (interp, seconds, machine)) == NULL) {
      goto done;
    }

    Tcl_ResetResult (interp);

    for (i = 0; i < idList -> count; i++) {
      sprintf (temp, "%u", idList -> ids[i]);
      Tcl_AppendElement (interp, temp);
    }

    delete (idList);
    code = TCL_OK;

  } else if (flag == INFO_ALL_NAMES) {

    if ((nameList = Agent_InfoNames (interp, seconds, machine)) == NULL) {
      goto done;
    }

    Tcl_ResetResult (interp);

    for (i = 0; i < nameList -> count; i++) {
      Tcl_AppendElement (interp, nameList -> names[i]);
    }

    delete (nameList);
    code = TCL_OK;
  
  } else {

    if ((info = Agent_Info (interp, seconds, id)) == NULL) {
	printf ("we have a NULL agent info\n");
      goto done;
    }

    Tcl_ResetResult (interp);
    
      /* append the agent identification */

    string = Agent_IdToString (interp, info -> id);
    Tcl_AppendElement (interp, Tcl_DStringValue (string));
    tclFreeDString (string);

      /* append the actual location */

    string = Agent_MachineToString (interp, info -> actual);
    Tcl_AppendElement (interp, Tcl_DStringValue (string));
    tclFreeDString (string);

      /* append the agent and blocker pid's */

    sprintf (temp, "%d %d", info -> agentPid, info -> blockerPid);
    Tcl_AppendElement (interp, temp);

      /* append the message, meeting and event counts */

    sprintf (temp, "%d %d %d", info -> messages, info -> meetings, info -> events);
    Tcl_AppendElement (interp, temp);
    delete (info);
    code = TCL_OK;
  }
      
done:

  delete_check (machine);
  delete_check (id);
  return (code);
}

/* Agent_ForceCmd

   Tcl syntax: agent_force [-time seconds] identification

      Purpose: Forcibly terminate an agent

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
               to the empty string.
*/

int Agent_ForceCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  double seconds;
  AGENT_ID *id = NULL;
  int code = TCL_ERROR;
  AGENT_ID *forcedId = NULL;
  Tcl_DString *string = NULL;

    /* check the number of arguments */

  if ((argc != 2) && (argc != 4)) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " identification ?-time seconds?\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the arguments */

  if ((id = Agent_SplitId (interp, argv[1])) == NULL) {
    goto cleanup;
  } else if (argc == 2) {
    seconds = DEFAULT_CONNECT_SECONDS;
  } else if (parseShortTimeout (interp, seconds, argc, argv, 2) < 0) {
    goto cleanup;
  } 

    /* forcibly terminate the agent */

  if (Agent_Force (interp, seconds, id, &forcedId) != TCL_OK) {
    goto cleanup;
  } else if (forcedId == NULL) {
    Tcl_SetResult (interp, "-1", TCL_STATIC);
  } else if ((string = Agent_IdToString (interp, forcedId)) == NULL) {
    goto cleanup;
  } else {
    Tcl_DStringResult (interp, string);
    tclFreeDString (string);
  }

  code = TCL_OK;

cleanup:

  delete_check (forcedId);
  delete_check (id);
  return (code);
}

#endif 
