/* Agent Tcl
   Bob Gray
   4 October 1995

   tkBasicCmd.cc

   This file implements the Tcl commands for Agent Tk.
*/

#include <X11/X.h>
#include <assert.h>
#include <string.h>
#include "message.h"
#include "my_strings.h"
#include "tcl.h"
#include "tk.h"
#include "tclAgentInt.h"
#include "tkAgentInt.h"
#include "tclLocation.h"
#include "tclMask.h"
#include "tcl_utilities.h"

static TK_OPTIONS options;
 
static int ArrivalTrap
	(Tcl_Interp *interp, MASK_SET *maskSet);

static int BrokenTrap
	(Tcl_Interp *interp, MASK_SET *maskSet);

struct COMMAND_INFO
{
  char *name;
  Tcl_CmdProc *proc;
};

static COMMAND_INFO agent_commands[] = 
{
  {"main"		, AgentTk_MainCmd	},
  {"exit"		, AgentTk_ExitCmd	},
  { NULL 		, (Tcl_CmdProc *) NULL	}
};

/* AgentTk_ExitCmd

   Tcl syntax: exit [code] 

      Purpose: Redefines the "exit" command to call agentForceEnd if necessary
               and destroy the application windows

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = number of command arguments
               argv   = the command argumnets

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and calls agentForceEnd if necessary.
*/

int AgentTk_ExitCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int code;
  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);

    /* check the number of arguments */

  if ((argc != 1) && (argc != 2)) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " [code]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* check the exit code -- we can not let Tk_ExitCmd do this because we  */
    /* do not want to take any action at all if the exit statement is going */
    /* to fail because of a syntax error                                    */
 
  if (argc == 2) {
    if (Tcl_GetInt(interp, argv[1], &code) != TCL_OK) {
      return TCL_ERROR;
    }
  }
 
    /* send a TRAN_END if necessary */

  if (locations -> registered == AGENT_REGISTERED) {
    agentForceEnd (interp, DEFAULT_CONNECT_SECONDS, locations -> local, NULL, END_NORMAL);
  }

    /* destroy the all windows and exit */

  Tk_ExitCmd ((ClientData) Tk_MainWindow(interp), interp, argc, argv);

    /* better not ever reach the next statement! */
 
  return TCL_ERROR;   
}

/* agentMainCreateCmd

   Tcl syntax: main create <tk_options>

      Purpose: The procedure creates the main window for the Tk applications.
 
        Input: dummy  = client data (not used)
               interp = the current interpreter
               argc   = the number of command arguments 
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and creates the main window,
*/

int agentMainCreateCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int code;
  Tk_Window tkwin;
  MAIN_WINDOW *window;
  Tk_ArgvInfo argTable[] = {
      {"-display", TK_ARGV_STRING, (char *) NULL, (char *) &options.display, (char *) NULL},
      {"-geometry", TK_ARGV_STRING, (char *) NULL, (char *) &options.geometry, (char *) NULL},
      {"-name", TK_ARGV_STRING, (char *) NULL, (char *) &options.name, (char *) NULL},
      {"-sync", TK_ARGV_CONSTANT, (char *) 1, (char *) &options.synchronize, (char *) NULL},
      {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL} 
   };

    /* check the number of command arguments */

  if (argc < 2) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " create <tk_options>\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* make sure that we do not already have a main window */

  if ((tkwin = Tk_MainWindow(interp)) != NULL) {
    Tcl_AppendResult (interp, "main window already exists", (char *) NULL);
    return TCL_ERROR;
  }

    /* parse out the options */

  argc -= 2;
  argv += 2;

  if (Tk_ParseArgv (interp, (Tk_Window) NULL, &argc, argv, argTable, TK_ARGV_DONT_SKIP_FIRST_ARG) != TCL_OK) {
    return TCL_ERROR;
  }

  window = MAIN_WINDOW::get_main (interp);
  assert (window != NULL);
  window -> update (&options); 

    /* create the main window */

  if (window -> name == NULL) {
    window -> name = my_strcpy ("unnamed");
  }

  if ((tkwin = Tk_CreateMainWindow (interp, window -> display, window -> name, window -> name)) == NULL) {
    Tcl_AppendResult (interp, ": unable to create main window", (char *) NULL);
    return TCL_ERROR;
  }

    /* set the DISPLAY environment variable */

  if (window -> display != NULL) {
    Tcl_SetVar2 (interp, "env", "DISPLAY", window -> display, TCL_GLOBAL_ONLY);
  }

    /* turn on X synchronization if desired */

  if (window -> synchronize) {
    XSynchronize (Tk_Display(tkwin), True);
  }

    /* set the initial geometry if specified */

  if (window -> geometry != NULL) {
    Tcl_SetVar (interp, "geometry", window -> geometry, TCL_GLOBAL_ONLY);
    if ((code = Tcl_VarEval (interp, "wm geometry . ", window -> geometry, (char *) NULL)) != TCL_OK) {
      Tcl_AppendResult (interp, ": unable to set geometry", (char *) NULL);
      Tk_DestroyWindow (tkwin);
      return TCL_ERROR;
    }
  }
 
    /* finish Tk initialization */

  if (Tk_Init (interp) != TCL_OK) {
    Tk_DestroyWindow (tkwin);
    return TCL_ERROR;
  }

  if (AgentTk_Init (interp, NULL) != TCL_OK) {
    Tk_DestroyWindow (tkwin);
    return TCL_ERROR;
  }

  return TCL_OK;
}

/* agentMainDestroyCmd

   Tcl syntax: main destroy

      Purpose: The procedure destroys the main window for the Tk application.

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and destroys the main window.
*/

int agentMainDestroyCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  Tk_Window tkwin;

    /* check the number of arguments */

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " destroy\"", (char *) NULL);
    return TCL_ERROR;
  } 

    /* make sure that we have a main window */ 

  if ((tkwin = Tk_MainWindow(interp)) == NULL) {
    Tcl_AppendResult (interp, "main window does not exist", (char *) NULL);
    return TCL_ERROR;
  }

    /* destroy the main window */

  Tk_DestroyWindow (tkwin);
  return TCL_OK;
}
 
/* AgentTk_MainCmd

   Tcl syntax: main create <screen> <class>
               main destroy

      Purpose: The procedure creates or destroys the main window for the 
               Tk application.

        Input: interp = the current interpreter
               dummy  = client data (unused)
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message if it is unable to create or
               destroy the main window.  Otherwise the procedure returns  
               TCL_OK and creates or destroys the main window.
*/

int AgentTk_MainCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int returnCode;

    /* check the number of arguments */

  if (argc <= 1) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " <operation> [options]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* perform the appropriate operation */

  if (!strcmp (argv[1], "create")) {
    returnCode = agentMainCreateCmd (dummy, interp, argc, argv);
  } else if (!strcmp (argv[1], "destroy")) {
    returnCode = agentMainDestroyCmd (dummy, interp, argc, argv);
  } else {
    Tcl_AppendResult (interp, "unknown option \"", argv[1], "\": should be \"create\" or \"destroy\"", (char *) NULL);
    returnCode = TCL_ERROR;
  }

  return (returnCode);
}

/* AgentTk_Init

   Purpose: Add the "main" command to a interpreter

     Input: interp  = the interpreter
            options = Tk options

    Output: The procedure adds the "main" command to the interpreter.
*/

static void agentTkCleanup (ClientData clientData, Tcl_Interp *interp) 
{
  MAIN_WINDOW *data = (MAIN_WINDOW *) clientData;
  delete_check (data);
}

int AgentTk_Init (Tcl_Interp *interp, TK_OPTIONS *options)
{
  COMMAND_INFO *ptr;
  MAIN_WINDOW *window;

    /* make sure that we have a mask set */

  MASK_SET *maskSet = MASK_SET::get_mask_set (interp);
  assert (maskSet != NULL);

    /* install the commands */

  for (ptr = agent_commands; ptr -> name != NULL; ptr++) {
    Tcl_CreateCommand 
      (interp, ptr -> name, ptr -> proc, 
	(ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  }

  if (MAIN_WINDOW::interpMap == NULL) {
    MAIN_WINDOW::interpMap = new POINTER_MAP ();
  }

    /* only do the following once */

  if ((window = MAIN_WINDOW::get_main(interp)) == NULL) {

      /* create the main window structure and the cleanup routine */

    window = new MAIN_WINDOW (interp, options);
    Tcl_CallWhenDeleted (interp, agentTkCleanup, (ClientData) window);
 
      /* set up the handlers for incoming messages, meetings and events */

    maskSet -> setEvents (ArrivalTrap, BrokenTrap);

      /* glue the tk variables */

    Tcl_GlueVar2 (interp, "tk_library", NULL, TCL_GLOBAL_ONLY);
    Tcl_GlueVar2 (interp, "tk_version", NULL, TCL_GLOBAL_ONLY);
    Tcl_GlueVar2 (interp, "tk_strictMotif", NULL, TCL_GLOBAL_ONLY);
    Tcl_GlueVar2 (interp, "tk_patchLevel", NULL, TCL_GLOBAL_ONLY);
  }

  return TCL_OK;
}

/* ArrivalHandler

   Purpose: This procedure is called whenever there is an incoming item AND
            there is an event handler for that item in the mask AND the Tk
            interpreter is at the main event loop.  The procedure executes the
            event handler.

     Input: clientData = MASK_SET structure

    Output: The procedure executes the event handler.
*/

void ArrivalHandler (ClientData clientData)
{
  int code;
  INCOMING *item;
  char scratch[32];
  Tcl_DString script;
  Tcl_DString arguments;
  MaskArrivalData *data = NULL;
  MASK_SET *maskSet = (MASK_SET *) clientData;
  Tcl_Interp *interp = maskSet -> get_interp ();

    /* get the first event in the queue */

  if ((data = maskSet -> get_event (data)) == NULL) {
    return;
  } 

    /* construct the argument list */

  item = data -> item;
  Tcl_DStringInit (&arguments);

    /* first the identifiction of the sender */

  Tcl_DString *idString = Agent_IdToString (interp, item -> id);
  Tcl_DStringAppendElement (&arguments, Tcl_DStringValue(idString));
  tclFreeDString (idString);

    /* now different things depending on the item type */ 
 
  if (item -> type == MESSAGE_ITEM) {
    INCOMING_MESSAGE *message = (INCOMING_MESSAGE *) item;
    sprintf (scratch, "%d", message -> code);
    Tcl_DStringAppendElement (&arguments, scratch);
    Tcl_DStringAppendElement (&arguments, message -> string);
  } else if (item -> type == EVENT_ITEM) { 
    INCOMING_EVENT *event = (INCOMING_EVENT *) item;
    Tcl_DStringAppendElement (&arguments, event -> tag);
    Tcl_DStringAppendElement (&arguments, event -> string);
  } else {
    INCOMING_MEETING *meeting = (INCOMING_MEETING *) item;
    Tcl_DString *locString = Tcpip_PortToString (interp, meeting -> port);
    Tcl_DString *staString = Agent_MeetingToString (interp, meeting -> status);
    Tcl_DStringAppendElement (&arguments, Tcl_DStringValue (locString));
    Tcl_DStringAppendElement (&arguments, Tcl_DStringValue (staString));
    tclFreeDString (locString);
    tclFreeDString (staString);
  }

    /* fire off the event handlers */

  do
  {
      /* first construct the appropriate script */

    Tcl_DStringInit (&script);
    Tcl_DStringAppend (&script, data -> handler, -1);
    Tcl_DStringAppend (&script, " ", -1);
    Tcl_DStringAppend (&script, Tcl_DStringValue(&arguments), -1);

      /* execute the script */

    Tcl_AllowExceptions (interp);
    code = Tcl_GlobalEval (interp, Tcl_DStringValue(&script));
    Tcl_DStringFree (&script);

      /* handle the error code */

    if ((code != TCL_OK) && (code != TCL_CONTINUE)) {
      if (code == TCL_BREAK) {
        break;
      } else {
        Tcl_AddErrorInfo (interp, "\n    (handler for an incoming item)");
        Tk_BackgroundError (interp);
        break;
      }
    }   
  } while ((data = maskSet -> get_event (data)) != NULL);

    /* clean up (including unprocessed handlers for the item) */

  while (data != NULL) {
    data = maskSet -> get_event (data);
  }

  Tcl_DStringFree (&arguments);
}

/* ArrivalTrap

   Purpose: This procedure is called whenever there is an incoming item AND
            there is an event handler for that item in the mask AND the
            Tcl interpreter is in a "safe" state.

     Input: interp  = the current interpreter
            maskSet = the MASK_SET structure associated with the interpreter

    Output: The procedure creates a Tk timer handler that will execute the 
            event handler as soon as Tk returns to the event loop.
*/

static int ArrivalTrap (Tcl_Interp *interp, MASK_SET *maskSet)
{
    /* Make a Tk timer handler that will call the event handler for the */
    /* incoming item.  The event handler can not be called right here   */
    /* since ArrivalTrap is called at arbitrary "safe" points within    */
    /* the Tcl script.  We want to wait until we are back inside the    */
    /* event loop.                                                      */

  Tk_CreateTimerHandler (0, ArrivalHandler, (ClientData) maskSet);
  return TCL_OK;
}

/* BrokenTrap

   Purpose: This procedure is called if the connection to the server breaks.
            Note that the procedure is only called when the interpreter is in
            a safe state.

     Input: interp  = the current interpreter
            maskSet = the MASK_SET structure associated with the interpreter 

    Output: The procedure fires off a background error handler.
*/

static int BrokenTrap (Tcl_Interp *interp, MASK_SET *maskSet)
{
  char *result = my_strcpy (interp -> result);
  Tcl_SetResult (interp, "broken connection to server (server crash)", TCL_STATIC);
  Tk_BackgroundError (interp);
  Tcl_SetResult (interp, result, TCL_VOLATILE);
  delete_check (result);
  return TCL_OK;
}
