/* Agent Tcl
 * Bob Gray
 * 3 October 1995
 *
 * tkMain.cc --
 *
 *      Main program for the Tk shell
 *
 * Copyright (c) 1995, Robert S. Gray Dartmouth College
 * Copyright (c) 1990-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "agent.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <tcl.h>
#include <tk.h>
#include <stdlib.h>
#include <unistd.h>
#include "agentId.h"
#include "message.h"
#include "my_sizes.h"
#include "my_strings.h"
#include "redirect.h"
#include "tclAgent.h"
#include "tkAgent.h"
#include "tclLocation.h"
#include "tclMask.h"
#include "tcpip.h"
#include "transmit.h"
#include "truefalse.h"

/*
 * Maximum number of seconds to wait between transmission attempts
 * Maximum number of tranmissions attempts
 */

const MAXIMUM_SLEEP = 512;
const MAXIMUM_TRIES = 10;
 
/*
 * Declarations for various library procedures and variables (don't want
 * to include tkInt.h or tkPort.h here, because people might copy this
 * file out of the Tk source directory to make their own modified versions).
 * Note: don't declare "exit" here even though a declaration is really
 * needed, because it will conflict with a declaration elsewhere on
 * some systems.
 */

extern int isatty _ANSI_ARGS_((int fd));

/*
 * Global variables used by the main program:
 */

static Tcl_Interp *interp;	/* Interpreter for this application. */
static Tcl_DString command;	/* Used to assemble lines of terminal input
				 * into Tcl commands. */
static int tty;			/* Non-zero means standard input is a
				 * terminal-like device.  Zero means it's
				 * a file. */
static char errorExitCmd[] = "exit 1";

/*
 * Command-line options:
 */

static char *fileName;
static char *socketName;
static struct TK_OPTIONS options;

static Tk_ArgvInfo argTable[] = {
    {"-display", TK_ARGV_STRING, (char *) NULL, (char *) &options.display,
	"Display to use"},
    {"-geometry", TK_ARGV_STRING, (char *) NULL, (char *) &options.geometry,
	"Initial geometry for window"},
    {"-name", TK_ARGV_STRING, (char *) NULL, (char *) &options.name,
	"Name to use for application"},
    {"-sync", TK_ARGV_CONSTANT, (char *) 1, (char *) &options.synchronize,
	"Use synchronous mode for display server"},
    {"-agent", TK_ARGV_STRING, (char *) NULL, (char *) &socketName,
	"Unix socket over which the agent is arriving"},
    {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL,
	(char *) NULL}
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static void		Prompt _ANSI_ARGS_((Tcl_Interp *interp, int partial));
static void		StdinProc _ANSI_ARGS_((ClientData clientData,
			    int mask));



/* Tk_ServerMain

   Purpose: Execute agents that arrive through the server
 
     Input: appInitProc = procedure to perform application initialization

    Output: The procedure never returns (it exits the process when it 
            finishes).
*/

void Tk_ServerMain (AgentTk_AppInitProc appInitProc)
{
  int sockfd;           /* socket descriptor                    */
  int return_code;      /* return_code from Tcl_Eval            */
  MESSAGE *message;     /* message carrying the script or state */
  Tcl_DString error;    /* error string from Tcl_LoadState      */
 
    /* connect to the socket */

  if ((sockfd = tcpip_unix_socket ()) < 0) {
    exit (1);
  }

  if (tcpip_unix_connect (sockfd, socketName) < 0) {
    exit (1);
  }

    /* receive the message */

  if ((message = message_receive (sockfd, to_interpreter)) == NULL) {
    exit (1);
  } 

  close (sockfd);

    /* redirect stdin, stdout and stderr */

  if (fileRedirect() < 0) {
    exit (1);
  }

    /* break out the root and local identification */

  char *root_server  = message -> elements[0].string;
  UINT_32 root_ip    = message -> elements[1].number;
  char *root_name    = message -> elements[2].string;
  UINT_32 root_id    = message -> elements[3].number;
  char *local_server = message -> elements[4].string;
  UINT_32 local_ip   = message -> elements[5].number;
  UINT_32 local_id   = message -> elements[6].number;
  char *script       = message -> elements[7].string;

    /* root and local identification */

  AGENT_ID *root  = new AGENT_ID (root_server, root_ip, root_name, root_id);
  AGENT_ID *local = new AGENT_ID (local_server, local_ip, NULL, local_id);

    /* create and the interpreter and load the state */

  if (message -> flag != INTERP_STATE) {
    interp = Tcl_CreateInterp ();
  } else {
    Tcl_DStringInit (&error);  
    if ((interp = Tcl_CreateAndLoadInterp (script, &error)) == NULL) {
      exit (1);
    }
    Tcl_DStringFree (&error);
  }

  interp -> interactive = 0;

  char *temp = my_strcpy (interp -> result);
    
  if ((*appInitProc)(interp, TRUE, &options) != TCL_OK) {
    exit (1);
  }

  Tcl_SetResult (interp, temp, TCL_DYNAMIC);
 
    /* load the locations */

  AGENT_LOCATION *locations = AGENT_LOCATION::get_location (interp);

  if (root -> server == NULL) {
    locations -> reload (local, local, AGENT_REGISTERED, AGENT_ROOT);
  } else {
    locations -> reload (root, local, AGENT_REGISTERED, AGENT_NOT_ROOT);
  }

    /* set up the background handler for messages, events and meetings */

  MASK_SET *mask_set = MASK_SET::get_mask_set (interp);
 
  if (mask_set -> sigio_on (local) < 0) {
    exit (1);
  }

    /* execute the script */

  if (message -> flag == INTERP_STATE) {
    return_code = Tcl_Eval (interp, NULL);
  } else {
    return_code = Tcl_Eval (interp, script);
  }

    /* return the result to the root agent */
 
  Agent_SendResult (interp, MAXIMUM_TRIES, MAXIMUM_SLEEP, return_code);
  
    /* we're done */

  Tcl_Eval (interp, "exit");
  return;
}

/*
 *----------------------------------------------------------------------
 *
 * AgentTk_Main --
 *
 *	Main program for Wish and most other Tk-based applications.
 *
 * Input:
 *
 *      argc        = number of arguments
 *      argv        = the arguments
 *      appInitProc = application-specific initialization procedure to call
 *                    after most initialization but before starting to
 *                    execute commands
 *  
 * Results:
 *	None. This procedure never returns (it exits the process when
 *	it's done.
 *
 * Side effects:
 *	This procedure initializes the Tk world and then starts
 *	interpreting commands;  almost anything could happen, depending
 *	on the script being interpreted.
 *
 *----------------------------------------------------------------------
 */

void AgentTk_Main (int argc, char **argv, AgentTk_AppInitProc *appInitProc)
{
  char *args, *p, *msg, *argv0;
  size_t length;
  char buf[20];
  int code;
  int prompt = 0;

    /*
     * Parse command-line arguments.  A leading "-file" argument is
     * ignored (a historical relic from the distant past).  If the
     * next argument doesn't start with a "-" then strip it off and
     * use it as the name of a script file to process.  Also check
     * for other standard arguments, such as "-geometry", anywhere
     * in the argument list.
     */

  argv0 = argv[0];

  if ((argc > 1) && !strcmp(argv[1], "-prompt")) {
    prompt = 1;
	argc--;
	argv++;
  }

  if (argc > 1) {
    length = strlen(argv[1]);
    if ((length >= 2) && (strncmp(argv[1], "-file", length) == 0)) {
       argc--;
       argv++;
    }
  }

  if ((argc > 1) && (argv[1][0] != '-')) {
    fileName = argv[1];
    argc--;
    argv++;
  } else {
    prompt = 1;
  }

  interp = Tcl_CreateInterp();

  if (Tk_ParseArgv(interp, (Tk_Window) NULL, &argc, argv, argTable, 0) != TCL_OK) {
    fprintf(stderr, "%s\n", interp->result);
    exit(1);
  }

    /*
     * The "-agent" argument means that we have an agent that is arriving
     * via the server.  Tk_ServerMain handles the agent.
     */

  if (socketName != NULL) {
    Tcl_DeleteInterp (interp);
    Tk_ServerMain (appInitProc);
  }

    /*
     * Make command-line arguments available in the Tcl variables "argc"
     * and "argv".
     */

  args = Tcl_Merge(argc-1, argv+1);
  Tcl_SetVar(interp, "argv", args, TCL_GLOBAL_ONLY);
  ckfree(args);
  sprintf(buf, "%d", argc-1);
  Tcl_SetVar(interp, "argc", buf, TCL_GLOBAL_ONLY);
  Tcl_SetVar(interp, "argv0", (fileName != NULL) ? fileName : argv0, TCL_GLOBAL_ONLY);

    /*
     * Initialize the Tk application.  If a -name option was provided,
     * use it;  otherwise, if a file name was provided, use the last
     * element of its path as the name of the application; otherwise
     * use the last element of the program name.  For the application's
     * class, capitalize the first letter of the name.
     */

  if (options.name == NULL) {
    p = (fileName != NULL) ? fileName : argv0;

    if ((options.name = strrchr((const char *) p, '/')) != NULL) {
      options.name++;
    } else {
      options.name = p;
    }
  }

    /*
     * Set the "tcl_interactive" variable.
     */

    tty = isatty(0);
    interp -> interactive = (fileName == NULL) && tty;
    Tcl_SetVar2 (interp, "tcl_interactive", NULL, interp -> interactive ? "1" : "0", TCL_GLOBAL_ONLY);
    Tcl_GlueVar2 (interp, "tcl_interactive", NULL, TCL_GLOBAL_ONLY);

    /*
     * Invoke application-specific initialization.
     */

    if ((*appInitProc)(interp, FALSE, &options) != TCL_OK) {
	fprintf(stderr, "application-specific initialization failed: %s\n",
		interp->result);
    }

    /*
     * Invoke the script specified on the command line, if any.
     */

    if (fileName != NULL) {
	code = Tcl_VarEval(interp, "source ", fileName, (char *) NULL);
	if (code != TCL_OK) {
	    goto error;
	}
    }
 
    if (prompt) {

	/*
	 * Commands will come from standard input, so set up an event
	 * handler for standard input.  Evaluate the .rc file, if one
	 * has been specified, set up an event handler for standard
	 * input, and print a prompt if the input device is a terminal.
	 */

	if (tcl_RcFileName != NULL) {
	    Tcl_DString buffer;
	    char *fullName;
	    FILE *f;
    
	    fullName = Tcl_TildeSubst(interp, tcl_RcFileName, &buffer);
	    if (fullName == NULL) {
		fprintf(stderr, "%s\n", interp->result);
	    } else {
		f = fopen(fullName, "r");
		if (f != NULL) {
		    code = Tcl_EvalFile(interp, fullName);
		    if (code != TCL_OK) {
			fprintf(stderr, "%s\n", interp->result);
		    }
		    fclose(f);
		}
	    }
	    Tcl_DStringFree(&buffer);
	}
	Tk_CreateFileHandler(0, TK_READABLE, StdinProc, (ClientData) 0);
	if (tty) {
	    Prompt(interp, 0);
	}
    } 
    fflush(stdout);
    Tcl_DStringInit(&command);
    Tcl_ResetResult(interp);

    /*
     * loop until we explicitly exit
     */

    while (1) {
      Tk_DoOneEvent (0);
    }

    /*
     * should never reach the "exit" below
     */

    exit(1);

error:
    msg = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
    if (msg == NULL) {
	msg = interp->result;
    }
    fprintf(stderr, "%s\n", msg);
    Tcl_Eval(interp, errorExitCmd);
}

/*
 *----------------------------------------------------------------------
 *
 * StdinProc --
 *
 *	This procedure is invoked by the event dispatcher whenever
 *	standard input becomes readable.  It grabs the next line of
 *	input characters, adds them to a command being assembled, and
 *	executes the command if it's complete.
 *
 * Input:
 *      clientData = not used
 *      mask       = not used
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Could be almost arbitrary, depending on the command that's
 *	typed.
 *
 *----------------------------------------------------------------------
 */

static void StdinProc(ClientData clientData, int mask)
{
#define BUFFER_SIZE 4000
    char input[BUFFER_SIZE+1];
    static int gotPartial = 0;
    char *cmd;
    int code, count;

    count = read(fileno(stdin), input, BUFFER_SIZE);
    if (count <= 0) {
	if (!gotPartial) {
	    if (tty) {
		Tcl_Eval(interp, "exit");
		exit(1);
	    } else {
		Tk_DeleteFileHandler(0);
	    }
	    return;
	} else {
	    count = 0;
	}
    }
    cmd = Tcl_DStringAppend(&command, input, count);
    if (count != 0) {
	if ((input[count-1] != '\n') && (input[count-1] != ';')) {
	    gotPartial = 1;
	    goto prompt;
	}
	if (!Tcl_CommandComplete(cmd)) {
	    gotPartial = 1;
	    goto prompt;
	}
    }
    gotPartial = 0;

    /*
     * Disable the stdin file handler while evaluating the command;
     * otherwise if the command re-enters the event loop we might
     * process commands from stdin before the current command is
     * finished.  Among other things, this will trash the text of the
     * command being evaluated.
     */

    Tk_CreateFileHandler(0, 0, StdinProc, (ClientData) 0);
    code = Tcl_RecordAndEval(interp, cmd, TCL_EVAL_GLOBAL);
    Tk_CreateFileHandler(0, TK_READABLE, StdinProc, (ClientData) 0);
    Tcl_DStringFree(&command);
    if (*interp->result != 0) {
	if ((code != TCL_OK) || (tty)) {
	    /*
	     * The statement below used to call "printf", but that resulted
	     * in core dumps under Solaris 2.3 if the result was very long.
	     */

	    puts(interp->result);
	}
    }

    /*
     * Output a prompt.
     */

    prompt:
    if (tty) {
	Prompt(interp, gotPartial);
    }
    Tcl_ResetResult(interp);
}

/*
 *----------------------------------------------------------------------
 *
 * Prompt --
 *
 *	Issue a prompt on standard output, or invoke a script
 *	to issue the prompt.
 *
 * Input:
 *      interp  = interpreter to use for prompting
 *      partial = non-zero means there already exists a partial command
 *                so use secondary prompt
 * 
 * Results:
 *	None.
 *
 * Side effects:
 *	A prompt gets output, and a Tcl script may be evaluated
 *	in interp.
 *
 *----------------------------------------------------------------------
 */

static void Prompt (Tcl_Interp *interp, int partial)
{
    char *promptCmd;
    int code;

    promptCmd = Tcl_GetVar(interp,
	partial ? "tcl_prompt2" : "tcl_prompt1", TCL_GLOBAL_ONLY);
    if (promptCmd == NULL) {
	defaultPrompt:
	if (!partial) {
	    fputs("agent-tk> ", stdout);
	}
    } else {
	code = Tcl_Eval(interp, promptCmd);
	if (code != TCL_OK) {
	    Tcl_AddErrorInfo(interp,
		    "\n    (script that generates prompt)");
	    fprintf(stderr, "%s\n", interp->result);
	    goto defaultPrompt;
	}
    }
    fflush(stdout);
}
