#!/usr/contrib/bin/agent

# Technical report search agent
#
# Original code:
# Ting Cai 
# 24 May 1995
#
# Cleanup and update to Agent Tcl 1.0:
# Bob Gray
# 27 July 1995
#
# This agent searches a distributed collection of technical reports for    
# occurences of a given keyword.  The agent sends a child agent to each 
# machine on which technical reports reside.  Each child searches the
# technical reports on its machine and returns a list of those technical 
# reports that contain the specifed keyword.
#
# The parent collects the search results and creates a WWW document that
# contains hyperlinks to all relevant technical reports.  The name of the
# document is KEYWORD.html where KEYWORD is the specified keyword.
#
# The agent incorporates a simple form of learning.  If there is already a
# file called KEYWORD.html, the agent does not perform the search again.

  # machines on which the technical reports reside

set machines "tioga.cs.dartmouth.edu \
	      muir.cs.dartmouth.edu \
	      tenaya.cs.dartmouth.edu \
	      tuolomne.cs.dartmouth.edu"

  # directories in which the technical reports reside

set directories "/a/quimby/usr/toe/grad/rgray/CODE/tioga_tr \
		 /a/quimby/usr/toe/grad/rgray/CODE/muir_tr \
		 /a/quimby/usr/toe/grad/rgray/CODE/tenaya_tr \
		 /a/quimby/usr/toe/grad/rgray/CODE/tuolomne_tr"

  # Procedure SEARCH
  #
  #  Input: keyword   = desired keyword
  #         directory = name of the directory that contains technical reports
  #
  # Output: The procedure returns a list of the technical reports that contain
  #         the specified keyword.

proc search {keyword directory} {
  
  global agent

  cd  $directory
  set result ""

    # get a list of ALL the technical reports

  if {[catch {set files [glob *]}]} {
    return ""
  }

    # check each technical report for the keyword

  foreach file $files {

      # open the technical report

    set fd [open $file]

      # check each line for the keyword
 
    while {[gets $fd line] >= 0} {
      if [regexp $keyword $line] {
        lappend result "$directory/$file"
        break
      }
    } 
  }

    # return the list of relevant technical reports

  return $result
}

  # get the search keyword

puts "Enter the search keyword: "
gets stdin keyword

  # '/' can't be part of the corresponding file name

regsub -all / $keyword | filtered

  # get a list of all the HTML files in the current directory 

if {[catch {set html_files [glob *.html]}]} {

  puts "No HTML files in current directory.  Proceeding with search ..."

} else {

    # check if there is already an HTML file for the given search keyword

  foreach file $html_files {
    if {$file == "$filtered.html"} {
      puts "Done! Results are in \"$filtered.html\"."
      return
    }
  }

    # there is no HTML file for the given search keyword

  puts "No HTML file for the given keyword.  Proceeding with search ..."
}

  # register the agent on the local machine

agent_begin

  # submit child agents to each machine
  # each child agent searchs the technical reports on its machine

set i 0

foreach machine $machines {

    # get the appropriate directory name

  set directory [lindex $directories $i]
  incr i

    # submit the child agent
    # in this case the child agent just calls procedure "search"

  agent_submit $machine -vars keyword directory \
		        -procs search \
			-script {search $keyword $directory}
}

puts "Agents are away .."

  # create the result file and write out the header

set out [open "$filtered.html" w]
puts $out "<HTML>"
puts $out "<HEAD><TITLE>Search results for keyword $keyword</TITLE></HEAD>"
puts $out "<BODY>"
puts $out "<H1>Search results for keyword $keyword</H1>"
    
  # wait for the search results to come back from each machine

set relevant_files 0

foreach machine $machines {

    # get a search result

  puts "Trying to receive ..."
  set source_id [agent_receive code status -blocking]
  puts "Result received ..."

    # abort on error

  if {$code != 0} {
    agent_end
    exec rm "$filtered.html"
    return -code error "Error in child agent: [lindex $status 0]"
  }

    # add the search results to the HTML file
    # first the list header

  puts $out "<H2>[lindex $source_id 0]<H2>"
  puts $out "<P>"
  puts $out "<UL>"

    # then each file

  foreach file $status {

     incr relevant_files

       # remove the path information

     set split_file [split $file /]
     set components [llength $split_file]
     set file_name [lindex $split_file [expr {$components - 1}]]

       # remove the .html suffix

     set split_name [split $file_name .]
     set file_name [lindex $split_name 0]
  
       # output the HTML list entry

     puts $out "<LI>"
     puts $out "<A HREF=\"$file\">$file_name</A>"
     puts $out "</LI>"
  }

    # finally the list terminator

  puts $out "</UL>"
  puts $out "<P>"
  puts "Result processed ..."
}

  # check if there were NO relevant technical reports at all

if {$relevant_files == 0} {
  exec rm "$filtered.html"
  puts "No technical reports contain the keyword $keyword!"
} else {
  puts $out </BODY>
  puts "Done! Search results are in \"$filtered.html\"." 
}
  
  # agent is done

agent_end
