/* Agent Tcl
   Bob Gray
   8 September 1995

   tclRestrict.cc

   This file implements the library routines that handle restrictions.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "interrupt.h"
#include "tcl.h"
#include "timers.h"
#include "tclRestrict.h"
#include "tclRestrictInt.h"
#include "truefalse.h"

/* restrictTimerTrap

   Purpose: This procedure is called whenever a timer fires.

     Input: timer = timer information

    Output: The procedure sets the fired flag and marks the Tcl asynchronous
            handler as ready.
*/

void restrictTimerTrap (TIMER timer)
{
  RESTRICT *restrict = (RESTRICT *) timer.data;
  Tcl_AsyncMark (restrict -> handler);
  restrict -> fired = 1;
}

/* RESTRICT::hasFired

   Purpose: Determine if a timer has fired

     Input: None

    Output: The procedure returns TRUE if a timer has fired since the previous
            call to RESTRICT::hasFired.  Otherwise the procedure returns FALSE.

      NOTE: SIGALRM and SIGPROF should be blocked when this procedure is called.
*/

int RESTRICT::hasFired (void)
{
  int flag;
  sigset_t mask;

    /* assertions on signal mask */

  assert ((sigprocmask (0, NULL, &mask), sigismember(&mask, SIGPROF) && sigismember(&mask, SIGALRM)));

    /* get the fired flag */

  flag  = fired;
  fired = FALSE;
  return (flag);
}
 
  /* initialize the pointer map in class RESTRICT */

POINTER_MAP *RESTRICT::interp_map = NULL;

  /* get the RESTRICT structure associated with the given interpreter */

RESTRICT *RESTRICT::get_restrict (Tcl_Interp *interp) {
  return ((RESTRICT *) interp_map -> lookup ((void *) interp));
}

  /* constructor for class PERMIT_EXT */

PERMIT_EXT::PERMIT_EXT (int flags, double wall, double cpu)
{
  PERMIT_EXT::flags = flags;
  PERMIT_EXT::wall  = wall;
  PERMIT_EXT::cpu   = cpu;
}

  /* constructor for class RESTRICT */

RESTRICT::RESTRICT (Tcl_Interp *interp, Tcl_AsyncHandler handler)
{
    /* add the interpreter to the mask map -- first creating the map if nec. */

  if (interp_map == NULL) {
    interp_map = new POINTER_MAP ();
  }

  interp_map -> add ((void *) interp, (void *) this);

    /* remember the interpreter and initialize the permit sets */

  RESTRICT::interp  = interp; 
  RESTRICT::sets    = NULL;
  RESTRICT::level   = 0;
  RESTRICT::handler = handler; 
} 

  /* destructor for class RESTRICT */

RESTRICT::~RESTRICT ()
{
    /* remove the interpreter from the map */

  interp_map -> remove ((void *) interp);
}

/* Restrict_AddPermit

   Purpose; Add a permit to the set of permits

     Input: interp = the current interpreter
            permit = the permit

    Output: The procedure adds the permit and returns TCL_OK.
*/

int Restrict_AddPermit (Tcl_Interp *interp, PERMIT_EXT *permit)
{
  RESTRICT *restrict;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (permit != NULL);
  restrict = RESTRICT::get_restrict (interp);
  assert (restrict != NULL);

    /* add the permit */

  restrict -> add (permit);
  return TCL_OK;
}

/* Restrict_RemovePermit

   Purpose: Remove the most recent permit from the set of permits

     Input: interp = the current interpreter

    Output: The procedure returns TCL_ERROR and sets the interpreter result to
            an appropriate error message if there are no permits.  Otherwise
            the procedure returns TCL_OK.
*/

int Restrict_RemovePermit (Tcl_Interp *interp)
{
    /* assertion on the parameter */

  assert (interp != NULL);

    /* error if there are no permits */

  RESTRICT *restrict = RESTRICT::get_restrict (interp);

  if (restrict -> getLevel() == 0) {
    Tcl_AppendResult (interp, "no permit to remove", (char *) NULL);
    return TCL_ERROR;
  }

    /* remove the most recent permit */

  restrict -> remove ();
  return TCL_OK;
}
 
/* Restrict_ParsePermit

   Purpose: Parse a string that contains a list of permits

     Input: interp = the current interpreter
            string = the string

    Output: The procedure returns NULL and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure
            returns a pointer to a PERMIT structure that contains the permit
            information.
*/

PERMIT_EXT *Restrict_ParsePermit (Tcl_Interp *interp, char *string)
{
  register int i;                  /* loop counter                  */
  char temp[10];                   /* scratch area                  */
  int listArgc;                    /* number of permits in the list */
  char **listArgv;                 /* the permits                   */
  int permArgc;                    /* number of items in a permit   */
  char **permArgv;                 /* the items                     */
  PERMIT_EXT *permit;              /* permit information            */
  double value;                    /* permit value                  */
 
    /* defaults */

  listArgv = NULL;
  permArgv = NULL; 
  permit   = new PERMIT_EXT (0, 0.0, 0.0);

    /* split apart the list of permits */

  if (Tcl_SplitList (interp, string, &listArgc, &listArgv) != TCL_OK) {
    Tcl_AppendResult (interp, ": permit list must be a valid Tcl list", (char *) NULL);
    goto error;
  } else if (listArgc == 0) {
    Tcl_AppendResult (interp, "permit list must contain at least one permit", (char *) NULL);
    goto error;
  }

    /* parse each permit */

  for (i = 0; i < listArgc; i++) {

      /* split apart the permit */

    if ((Tcl_SplitList (interp, listArgv[i], &permArgc, &permArgv) != TCL_OK) || (permArgc != 2)) {
      sprintf (temp, "%d", i);
      Tcl_AppendResult (interp, ": permit must have the form {name value}: permit ", temp, " invalid", (char *) NULL);
      goto error;
    } 
 
      /* get the permit value */

    if (Tcl_GetDouble (interp, permArgv[1], &value) != TCL_OK) {
      sprintf (temp, "%d", i);
      Tcl_AppendResult (interp, ": permit value must be a number: permit ", temp, " invalid", (char *) NULL);
      goto error;
    }

      /* put the value into the permit structure */

    if (!strcmp(permArgv[0], "wall")) {

      if (value < 0.0) {
        Tcl_AppendResult (interp, "wall time must be 0 or greater", (char *) NULL);
        goto error;
      }
 
      permit -> flags |= PERMIT_WALL;
      permit -> wall   = value;

    } else if (!strcmp(permArgv[0], "cpu")) {

      if (value < 0.0) {
        Tcl_AppendResult (interp, "cpu time must be 0 or greater", (char *) NULL);
        goto error;
      }

      permit -> flags |= PERMIT_CPU;
      permit -> cpu    = value;

    } else {

      Tcl_AppendResult (interp, "unknown permit type \"", permArgv[0], "\": should be \"wall\" or \"cpu\"", (char *) NULL);
      goto error;

    }
  } 
        
    /* success */

  delete_check (listArgv);
  delete_check (permArgv);
  return (permit);

    /* error */

error:

  delete_check (permit);
  delete_check (listArgv);
  delete_check (permArgv);
  return (NULL);
}

/* RESTRICT::add

   Purpose: Add a permit to the list of nested permits

     Input: permit = the permit

    Output: The procedure adds the permit to the list of nested permits
*/

void RESTRICT::add (PERMIT_EXT *permit_ext)
{
    /* assertions on the input parameters */

  assert (permit_ext != NULL);
 
    /* add the permit to the list of nested permits */
 
  PERMIT permit (permit_ext, timers.getTicks());
  sets = new PERMIT_SET (&permit, sets);
  level += 1;

    /* set a timer if necessary */

  if (permit.flags & PERMIT_WALL) {
    TIMER timer (permit.wall, restrictTimerTrap, (TimerClientData) this);
    timers.add (timer); 
  }
}

/* RESTRICT::remove

   Purpose: Remove the most recent permit from the list of nested permits

     Input: None

    Output: The procedure removes the most recent permit from the list of
            nested permits.
*/

void RESTRICT::remove (void)
{
    /* make sure that there is a permit to remove */

  assert (sets != NULL);

    /* remove the most recent permit */

  PERMIT_SET *temp = sets -> next;
  delete sets;
  sets = temp;
  level -= 1;
}

/* RESTRICT::permitViolation

   Purpose: Check for permit violations

     Input: None

    Output: The procedure returns a flag set that indicates which permits
            have been violated.  For example, if the wall-time limit has been
            exceeded, the PERMIT_WALL bit is set.
*/

int RESTRICT::permitCheck (PERMIT_SET *set)
{
  int flags = 0;
  clock_t wall;        /* wall time (clock ticks) */

    /* NO permit means NO violation */

  if (set == NULL) {
    return 0;
  }

    /* get the current times */

  wall = TIMERS::getWall ();

    /* see if the current times exceed the threshold */

  PERMIT *minimum = &sets -> minimum;

  if (minimum -> flags & PERMIT_WALL) {
    if (wall >= minimum -> wall) {
      flags = flags | PERMIT_WALL;
    }
  }

  return (flags);
}

int RESTRICT::permitViolation (void)
{
    /* check for violations */

  return (permitCheck (sets));
}

/* RESTRICT::permitViolationCurrent

   Purpose: See if a permit violation has occurred due to the most recent
            permit or due to a previous permit

     Input: None

    Output: The procedure returns 0 if the permit violation is due to a
            previous permit.  Otherwise the procedure returns a set of flags
            that indicates the violations arising from the current permit.
*/

int RESTRICT::permitViolationCurrent (void)
{
  int flags;
  int nextFlags;

    /* operation should only be called if there is a permit */

  assert (sets != NULL);

    /* check if the violations are at a HIGHER LEVEL */

  flags     = permitCheck (sets);
  nextFlags = permitCheck (sets -> next);

  if (nextFlags == 0) {
    return (flags);
  } else {
    return (0);
  }
}

/* PERMIT_SET::PERMIT_SET

   Purpose: This procedure is the constructor for class PERMIT_SET.

     Input: permit = amount of resource that can be used starting now
            next   = next permit in the list of nested permits
*/

PERMIT_SET::PERMIT_SET (PERMIT *permit, PERMIT_SET *next)
{
  clock_t wall;             /* wall time (clock ticks)            */
  struct tms tms_buffer;

    /* set the duration permit */

  PERMIT_SET::next     = next;
  PERMIT_SET::duration = *permit;

    /* set the threshold permit                              */
    /* first get the current times                           */
    /* then add to the duration permit to get the thresholds */ 

  wall = TIMERS::getWall ();

/*
  cpu  = tms_buffer.tms_utime + tms_buffer.tms_stime;
*/

  threshold.flags = duration.flags;

  if (threshold.flags & PERMIT_WALL) {
    threshold.wall = duration.wall + wall;
  }

/*
  if (threshold.flags & PERMIT_CPU) {
    threshold.cpu = duration.cpu + cpu;
  }
*/

    /* determine the current minimum threshold                        */
    /* just copy the current threshold if their is no previous permit */ 
    /* otherwise find the minimums                                    */

  if (next == NULL) {
    minimum = threshold;
  } else {
    if (!(threshold.flags & PERMIT_WALL)) {
      minimum.flags |= (next -> minimum.flags & PERMIT_WALL);
      minimum.wall   = next -> minimum.wall;
    } else if (threshold.wall < next -> minimum.wall) {
      minimum.flags |= PERMIT_WALL;
      minimum.wall   = threshold.wall;
    } else {
      minimum.flags |= PERMIT_WALL;
      minimum.wall   = next -> minimum.wall;
    }

/*
    if (!(threshold.flags & PERMIT_CPU)) {
      minimum.flags |= (next -> minimum.flags & PERMIT_CPU);
      minimum.cpu    = next -> minimum.cpu;
    } else if (threshold.cpu < next -> minimum.cpu) {
      minimum.flags |= PERMIT_CPU;
      minimum.cpu    = threshold.cpu;
    } else {
      minimum.flags |= PERMIT_CPU;
      minimum.cpu    = next -> minimum.cpu;
    }
*/
  }
}   

/* PERMIT::PERMIT

   Purpose: This procedure is a constructor for class PERMIT.

     Input: permit_ext = a permit in external form
*/

PERMIT::PERMIT (PERMIT_EXT *permit_ext, long ticks)
{
  flags = 0;

  if (permit_ext -> flags & PERMIT_WALL) {
    flags |= PERMIT_WALL;
    wall = (clock_t) (permit_ext -> wall * ticks);
  }  

  if (permit_ext -> flags & PERMIT_CPU) {
    flags |= PERMIT_CPU;
    wall = (clock_t) (permit_ext -> cpu * ticks);
  }
}

/* PERMIT::PERMIT 

   Purpose: This procedure is the copy constructor for class PERMIT.

     Input: permit = permit of which we are making a copy 
*/

PERMIT::PERMIT (const PERMIT &permit)
{
  flags = permit.flags;
  wall  = permit.wall;
  cpu   = permit.cpu;
}

/* PERMIT::operator= 

   Purpose: This procedure defines the assignment operator for class PERMIT.

     Input: permit = permit of which we are making a copy 
*/

PERMIT& PERMIT::operator= (const PERMIT &permit)
{
    /* make sure that we are not assigning a PERMIT to itself */

  if (this == &permit) {
    return *this;
  }

    /* otherwise assign the permit */

  flags = permit.flags;
  wall  = permit.wall;
  cpu   = permit.cpu;
  return *this;
}
