/* Agent Tcl
   Bob Gray
   25 January 1995
  
   tclRestrictCmd.cc

   This file implements the Tcl commands that handle restriction.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <sys/time.h>
#include <sys/times.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "interrupt.h"
#include "map.h"
#include "my_alloc.h"
#include "tcl.h"
#include "tclRestrict.h"
#include "tclRestrictInt.h"
#include "timers.h"
#include "truefalse.h"

#ifdef TCL_STATE_HANDLERS

static int restrictStateProc
	(Tcl_Interp *interp, ClientData data, EXTENSION_STATE *state);

#endif

struct RESTRICT_DATA
{
  RESTRICT *restrict;
  Tcl_Interp *interp;
  Tcl_AsyncHandler handler;
};

/* restrictAsyncCheck

   Purpose: Tcl asynchronously calls this procedure whenever a POTENTIAL
            permit violation has occurred.  The procedure is only called
            when the Tcl interpreter is in a "safe" state.

     Input: clientData = RESTRICT_DATA structure

    Output: The procedure returns TCL_PERMIT if a permit violation has
            occurred.  Otherwise the procedure passes through the current
            result code.
*/ 

static int restrictAsyncCheck (ClientData clientData, Tcl_Interp *interp, int code)
{
  RESTRICT_DATA *data = (RESTRICT_DATA *) clientData;

  if (data -> restrict -> permitViolation()) {

    data -> interp -> resultCode = TCL_PERMIT;

    if (data -> interp == interp) {
      return TCL_PERMIT;
    }
  }

  return (code);
} 

/* Restrict_RestrictCmd

   Tcl syntax: restrict permit [permit ...] script

      Purpose: 

*/

#define CMD_RESTRICT_BODY	1

static int Restrict_RestrictBody (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int flags;       
  char temp[60];
  Tcl_DString codeString;

     /* get the restriction information */

  RESTRICT_DATA *data = (RESTRICT_DATA *) clientData;

    /* we have just evaluated the body of the "restrict" command */
    /* so check for errors and permit violations                 */

  if (interp -> resultCode == TCL_ERROR) {

      /* set up errorInfo */

    sprintf (temp, "\n    (\"restrict\" body line %d)", interp -> errorLine);
    Tcl_AddErrorInfo (interp, temp);

  } else if (interp -> resultCode == TCL_PERMIT) {

      /* check for permit violations AT THE CURRENT LEVEL */
      /* if the permit violation is at a higher level     */
      /* we will just throw the TCL_PERMIT code           */

    if ((flags = data -> restrict -> permitViolationCurrent()) > 0) {

        /* set up errorInfo*/

      sprintf (temp, "\n    (\"restrict\" body line %d)", interp -> errorLine);
      Tcl_AddErrorInfo (interp, temp);
 
        /* set up the error string and errorCode */

      Tcl_DStringInit (&codeString);
      Tcl_DStringAppendElement (&codeString, "PERMIT");
      Tcl_SetResult (interp, "permit violation", TCL_STATIC);

      if (flags & PERMIT_WALL) {
	Tcl_AppendResult (interp, ": wall time exceeded", (char *) NULL);
        Tcl_DStringAppendElement (&codeString, "WALL");
      }

      if (flags & PERMIT_CPU) {
        Tcl_AppendResult (interp, ": cpu time exceeded", (char *) NULL);
        Tcl_DStringAppendElement (&codeString, "CPU");
      }

      Tcl_SetVar2 (interp, "errorCode", (char *) NULL, Tcl_DStringValue (&codeString), TCL_GLOBAL_ONLY);
      Tcl_DStringFree (&codeString);

        /* set up resultCode */

      interp -> resultCode = TCL_ERROR;
    }
  } 
 
    /* we are finished with the current permit */

  data -> restrict -> remove ();

    /* move on to the next command in the script */

  Stack_Pop (interp);
  return (interp -> resultCode);
}

int Restrict_RestrictCmd (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
  register PERMIT_EXT *permit;
  register int flag;

    /* if a state image has been captured and later restored, */
    /* we may be right in the middle of a "restrict" command  */

  Stack_GetFlag (interp, &flag);

  if (flag == CMD_RESTRICT_BODY) {
    return (Restrict_RestrictBody (clientData, interp, argc, argv));
  }
 
    /* check the number of arguments */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " permits script\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* parse the permit */

  if ((permit = Restrict_ParsePermit (interp, argv[1])) == NULL) {
    return TCL_ERROR;
  }

    /* add the permit to the list of nested permits */

  RESTRICT_DATA *data = (RESTRICT_DATA *) clientData;
  data -> restrict -> add (permit);    
  delete permit;

    /* push on the body */

  Stack_NewHandler (interp, CMD_RESTRICT_BODY, Restrict_RestrictBody);
  Stack_Push (interp, argv[2], TCL_STATIC);
  return TCL_OK;
}

/* Restrict_Init

   Purpose: Initialize the restrict extension

     Input: interp     = the current interpreter

    Output: The procedure returns TCL_ERROR on initialization failure
            and TCL_OK on initilization success.
*/

static void restrictCleanup (ClientData clientData, Tcl_Interp *interp) 
{
  RESTRICT_DATA *data = (RESTRICT_DATA *) clientData;
  delete_check (data -> restrict);
  Tcl_AsyncDelete (data -> handler);
}

int Restrict_Init (Tcl_Interp *interp)
{
  RESTRICT_DATA *data = new RESTRICT_DATA;
#ifdef TCL_STATE_HANDLERS
  EXTENSION_STATE *state;
#endif

    /* set up the RESTRICT structure and the interrupt handler */

  data -> handler  = Tcl_AsyncCreate (restrictAsyncCheck, (ClientData) data);
  data -> restrict = new RESTRICT (interp, data -> handler);
  data -> interp   = interp; 

#ifdef TCL_STATE_HANDLERS

    /* check for state information sent along with an agent */

  if ((state = Tcl_GetState (interp, "restrict-tcl")) != NULL) {
     /* do state processing here; don't free the "state" variable */
     Tcl_RemoveState (interp, "restrict-tcl");
  }
 
    /* register the state-capture routine */

  Tcl_RegisterState (interp, restrictStateProc, (ClientData) data);

#endif

    /* set up the cleanup routines that are called on interpreter deletion */

  Tcl_CallWhenDeleted (interp, restrictCleanup, (ClientData) data);

    /* install the restrict command */

  Tcl_CreateIntCommand (interp, "restrict", Restrict_RestrictCmd,
        (ClientData) data, (Tcl_CmdDeleteProc *) NULL);

  return TCL_OK;
}

#ifdef TCL_STATE_HANDLERS

/* restrictStateProc

   Purpose: This procedure is called during state capture in order to get the 
            internal state of the restrict extension.

     Input: interp   = the Tcl interpreter
		       (struct Tcl_Interp *)

	    data     = the associated RESTRICT structure
		       (ClientData ==> struct RESTRICT_DATA *)

	    state    = EXTENSION_STATE structure that this routine will fill
		       (struct EXTENSION_STATE *)

    Output: The procedure returns TCL_ERROR if it is unable to capture the state 
            and TCL_OK otherwise.  In both cases the procedure fills in the 
            appropriate sections of the EXTENSION_STATE structure.
*/

int restrictStateProc (Tcl_Interp *interp, ClientData data, EXTENSION_STATE *state)
{
    state -> name  = "restrict-tcl";
    state -> major = 1;
    state -> minor = 2;
    state -> type  = TCL_ALPHA;
    Tcl_DStringAppend (&state -> state, "This is the state information!", -1);
    return TCL_OK;
}

#endif
