/* Agent Tcl
   Bob Gray
   8 September 1995

   tclRestrictInt.h

   This file defines the internal routines that are called from more than one
   source file.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _TCL_RESTRICT_INT_H
#define _TCL_RESTRICT_INT_H

#include <sys/times.h>
#include <sys/times.h>
#include "map.h"
#include "my_alloc.h"
#include "tcl.h"
#include "timers.h"
#include "tclRestrict.h"

struct PERMIT
{
  int flags;     /* flags that indicate which permits are set */
  clock_t wall;  /* limit on wall time (clock ticks)          */
  clock_t cpu;   /* limit on cpu time (clock ticks)           */

    /* constructors and destructors */

  PERMIT (void) { flags = 0; };
  PERMIT (const PERMIT &permit);
  PERMIT& operator= (const PERMIT &permit);
  PERMIT (PERMIT_EXT *permit, long ticks);
};

  /* structure PERMIT_SET holds a set of permits */

struct PERMIT_SET
{
  PERMIT_SET *next;   /* next permit set                                    */
  PERMIT duration;    /* amount of resource that can be used starting now   */     
  PERMIT threshold;   /* resource threshold corresponding to that amount    */
  PERMIT minimum;     /* minumum thresholds since we can have nested limits */

    /* constructor */

  PERMIT_SET (PERMIT *duration, PERMIT_SET *next);
};

  /* called when a timer fires */

void restrictTimerTrap 
	(TIMER timer);

  /* class RESTRICT keeps track of the active "restrict" commands */

class RESTRICT 
{
    friend void restrictTimerTrap 
	(TIMER timer);

    Tcl_AsyncHandler handler;  /* Tcl asynchronous handler                   */
    TIMERS timers;             /* timers for wall and cpu time               */
    Tcl_Interp *interp;        /* interpreter associated with this instance  */
    PERMIT_SET *sets;          /* sets of permits                            */
    int level;                 /* number of nested permits                   */
    int fired;                 /* TRUE if a timer has fired since last check */

      /* map an interpreter address to the appropriate RESTRICT structure */
  
    static POINTER_MAP *interp_map;

      /* check for a permit violation */

    int permitCheck (PERMIT_SET *set);

  public:

      /* get the RESTRICT structure associated with an interpreter */

    static RESTRICT *get_restrict (Tcl_Interp *interp);

      /* get the next wall time threshold */

    inline int first_wall (clock_t &wall) { 
      if ((sets == NULL) || !(sets -> minimum.flags & PERMIT_WALL)) {
        return -1;
      } else {
        wall = sets -> minimum.wall;
        return 0;
      }
    };

      /* get the number of nested permits */

    inline int getLevel (void) { 
      return level; 
    };

      /* return TRUE if a timer has fired since the last call */

    int hasFired (void);

      /* check for permit violations */

    int permitViolation (void);
    int permitViolationCurrent (void);
 
      /* add a new permit or remove the most recent permit */

    void add (PERMIT_EXT *permit);
    void remove (void);
 
      /* constructor and destructor */

    RESTRICT (Tcl_Interp *interp, Tcl_AsyncHandler handler);
   ~RESTRICT (); 
};

  /* Tcl commands */

int Restrict_RestrictCmd 
	(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);

#endif
