# Agent Tcl
# Bob Gray
# 15 July 1995
#
# agent.tcl --
#
# Defines agent_meet and agent_accept

  # make sure that we are executing at the top level

if {[info level] != 0} {
  return -code error "\"agent.tcl\" can only be sourced at the global level"
}

  # change to the user's home directory if the agent arrived via the server

if {[info exists agent(server)] && $agent(server)} {
  cd ~
}

# agent_accept
#
# Purpose: Wait for and accept a connecton request

proc agent_accept {idVarName locVarName sockfd blockingFlag args} glue {

  upvar $idVarName source_id $locVarName source_loc

  set source_id [get_meeting source_loc $blockingFlag]

    # no request is available 

  if {$source_id == "-1"} {
    return -1 
  }

    # accept the request

  set sockfd [accept_meeting $source_id $source_loc $sockfd]
  return $sockfd
} 

# get_meeting
#
# Purpose: Wait for a meeting request  

proc get_meeting {locVarName blockingFlag args} glue {

  upvar $locVarName source_loc

    # wait for a meeting request to arrive

  set source_id [agent_getreq status source_loc $blockingFlag]

  if {$source_id != -1} {
    if {$status != "REQUEST"} {
      return -code error -errorcode "PROTOCOL" \
        "protocol error: expected REQUEST but got $status"
    }
  }

  return $source_id 
}

# reject_meeting
#
# Purpose: Reject a meeting request

proc reject_meeting source_id glue {

  agent_req $source_id -refuse
}

# accept_meeting
#
# Purpose: Accept a meeting request

proc accept_meeting {source_id source_loc sockfd} glue {

  global errorCode errorInfo

  set accept_sockfd -1

  if {[catch {

      # set up the socket and port

    if {$sockfd == "ANY"} {
      set accept_sockfd [tcpip_socket]
      set port [tcpip_bind $accept_sockfd ANY]
      tcpip_listen $accept_sockfd
    } else {
      set accept_sockfd $sockfd
      set port [lindex [tcpip_getport $accept_sockfd] 2]
    }

      # send the acknowledgement

    agent_req $source_id -connect $port

      # accept the connection

    tcpip_accept $accept_sockfd -blocking

  } result]} {      # error on CATCH

    set oldCode $errorCode
    set oldInfo $errorInfo

    if {$sockfd == "ANY"} {
      catch {tcpip_close $accept_sockfd}
    }

    return -code error -errorcode $oldCode -errorinfo $oldInfo $result

  } else {          # no error on CATCH 

    if {$sockfd == "ANY"} {
      tcpip_close $accept_sockfd
    }
 
    return $result
  } 
} 

# agent_meet
#
# Purpose: Meet with another agent

proc meet_internal dest_id glue {

  global agent errorCode errorInfo

    # send a REQUEST 

  agent_req $dest_id -request

    # wait for the response

  set dest_id [agent_getreq status dest_loc -blocking]

    # check the response

  if {$status == "REFUSED"} {

      # error message if the recipient REFUSED the meeting 

    return -code error -errorcode REFUSED \
	"recipient has refused the request for a meeting"

  } elseif {$status == "CONNECT"} {

      # connect if the recipient has provided a port

    set sockfd [tcpip_socket]
    tcpip_connect $sockfd $dest_loc
    return $sockfd

  } elseif {$status != "REQUEST"} {

      # error message if we did not get REFUSED, REQUEST or CONNECT

    return -code error -errorcode PROTOCOL \
	"protocl error: expected REFUSED, REQUEST or CONNECT but got $status"
  }

    # we have received a REQUEST from the agent to which we just sent a
    # REQUEST so decide which agent should issue "tcpip-accept" and
    # which should issue "tcpip-connect"

  set ip [lindex $dest_id 1]
  set id [lindex $dest_id 3]

  if {$agent(local-ip) < $ip} {
    set connect 1
  } elseif {$agent(local-ip) > $ip} {
    set connect 0
  } elseif {$agent(local-id) < $id} {
    set connect 1
  } elseif {$agent(local-id) > $id} {
    set connect 0
  } else {
    return -code error -errorcode SELF "an agent can not meet with itself"
  }

    # if this agent is supposed to to the connect, wait for the other
    # agent to send us a PORT number and then issue "tcpip_connect";

  if {$connect} {

    set dest_id [agent_getreq status dest_loc -blocking]

    if {$status != "CONNECT"} {
      return -code error -errorcode PROTOCOL \
	"protocol error: expected CONNECT but got $status"
    }

    set sockfd [tcpip_socket]
    tcpip_connect $sockfd $dest_loc
    return $sockfd
  } 

    # otherwise choose a port, send the port number to the other agent
    # and do the "tcpip_accept" -- i.e. just call accept_meeting

  accept_meeting $dest_id $dest_loc ANY
}

proc agent_meet dest_id glue {

  global agent errorCode errorInfo

    # set a "default" sockfd

  set sockfd -1

    # set up the mask 

  set newMask [mask new]
  mask add $newMask $dest_id
  set oldMask [mask_swap meeting $newMask]

    # catch all errors so that we can restore the mask and close the sockfd
    # if necessary

  if {[catch {

    meet_internal $dest_id

  } result]} {    # error on CATCH

    set oldCode $errorCode
    set oldInfo $errorInfo
    mask_replace meeting $oldMask
    catch {tcpip_close $sockfd}
    return -code error -errorcode $oldCode -errorinfo $oldInfo $result

  } else {        # no error on CATCH

    mask_replace meeting $oldMask
    return $result
  }
}

# mask_swap
#
# Purpose: Replace the message, meeting or event mask 
#
#   Input: name   = "message", "meeting" or "event"
#          handle = handle of the new mask 
#
#  Output: The procedure replaces and returns the handle of the old mask.

proc mask_swap {name handle} glue {

  global mask   # global array that holds the event, message and meeting masks

  set old_handle [set mask($name)]
  set mask($name) $handle
  return $old_handle 
}

# mask_replace
#
# Purpose: Replace the message, meeting or event mask
#
#   Input: name   = "message", "meeting" or "event"
#          handle = handle of the new mask
#
#  Output: The procedure replaces the old mask and returns the empty string.
#          The old mask is DELETED if it is not duplicated as one of the other
#          two special masks.

proc mask_replace {name handle} glue {

  global mask   # global array that holds the event, message and meeting masks

  set count 0
  set old $mask($name)

  foreach i {message meeting event} {
    if {$mask($i) == $old} {
      incr count
    }
  }

  if {$count == 1} {
    mask delete $old
  }

  set mask($name) $handle
  return
}
