/* Agent Tcl
   Bob Gray
   15 February 1995

   agent_data.cc

   This file implements the structures that keep track of agent data.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <limits.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "agent_data.h"
#include "agentd.h"
#include "error.h"
#include "hash.h"
#include "message.h"
#include "my_alloc.h"
#include "my_strings.h"
#include "truefalse.h"

/* AGENT_DATA::AGENT_DATA and AGENT_DATA::~AGENT_DATA
 
   Purpose: These procedures are the constructor and destructor for class
            AGENT_DATA.
*/

AGENT_DATA::AGENT_DATA (void)
{
  agent_table = new HASH (HASH_ENTRIES);
  name_table  = new HASH (HASH_ENTRIES);
  next_id     = 0;
  num_agents  = 0;
}

AGENT_DATA::~AGENT_DATA ()
{
  delete agent_table;
  delete name_table;
}

/* AGENT::AGENT and AGENT::~AGENT 

   Purpose: These procedures are the constructor and destructor for class
            AGENT.
*/

AGENT::AGENT (AGENT_ID *id)
{
  AGENT::id           = id;
  AGENT::pid          = 0;
  AGENT::blocked_pid  = 0;
  AGENT::blocked_item = FALSE;
}

AGENT::~AGENT ()
{
  register INCOMING_MESSAGE *message;
  register INCOMING_MEETING *meeting;
  register INCOMING_EVENT *event;

  while ((message = (INCOMING_MESSAGE *) messages.dequeue ()) != NULL) {
    delete message;
  }
 
  while ((meeting = (INCOMING_MEETING *) meetings.dequeue ()) != NULL) {
    delete meeting;
  }

  while ((event = (INCOMING_EVENT *) events.dequeue ()) != NULL) {
    delete event;
  }

  delete_check (id);
} 

/* AGENT_DATA::lookup_recipient

   Purpose: Lookup up the recipient agent

     Input: id   = numeric id of the destination agent
            name = name of the destination agent (NULL if not specified)

    Output: The procedure returns NULL if it can not find an agent and a 
            pointer to the appropriate AGENT structure if it does find the
            agent.  Note that the procedure will CREATE an agent if a name
            is given but there is no current agent with the name.  This new
            agent will have a numeric id of UNDEFINED_AGENT. 
*/

AGENT *AGENT_DATA::lookup_recipient (UINT_32 id, char *name)
{
  int there;
  AGENT *agent_data;
  AGENT_ID *agent_id;
  HASH_NODE *hash_node;

    /* lookup the agent using its id if no name is specified or */
    /* if the agent is a "server" agent; otherwise look up the  */
    /* agent using its name (creating a new agent if necessary) */

  if ((name == NULL) || (*name == '\0') || !strcmp(name, "server")) {
  
    if ((hash_node = agent_table -> lookup (id, NULL)) == NULL) {
      agent_data = NULL;
    } else {
      agent_data = (AGENT *) hash_node -> data;
    }

  } else {

    if ((hash_node = name_table -> lookup (0, name)) != NULL) {
      agent_data = (AGENT *) hash_node -> data;
    } else {
      agent_id = new AGENT_ID (NULL, 0, name, 0);
      agent_data = new AGENT (agent_id);
      name_table -> add (0, name, (void *) agent_data, there);
    }      
  }

  return agent_data;
}

/* AGENT_DATA::agent_info

   Purpose: Get information about the registered agents

     Input: [0] = INFO_ALL_IDS, INFO_ALL_NAMES or INFO_ID
            [1] = name of the agent of interest
            [2] = numeric id of the agent of interest

    Output: The procedure returns the response that should be sent to
            socketd.
*/

MESSAGE *AGENT_DATA::agent_info (MESSAGE *message)
{
  MESSAGE *response;
  register unsigned i;
  register AGENT *agent;
  register unsigned count;
  register ID_LIST *idList;
  register HASH_NODE *node; 
  register NAME_LIST *nameList;
  register HASH_SEARCH *search;

    /* break out the type of information */

  UINT_8 flag = (UINT_8) message -> elements[0].number;

    /* information about all numeric ids */
 
  if (flag == INFO_ALL_IDS) {

    count  = agent_table -> get_entries ();
    idList = new ID_LIST (count);
    search = new HASH_SEARCH (agent_table); 

    for (i = 0; i < count; i++) {
      node = search -> next ();
      idList -> ids[i] = (UINT_32) (node -> key);
    }      

    response = new MESSAGE (down_pipe.messages[RESP_IDS]); 
    response -> elements[0].string  = (char *) idList;
    response -> elements[0].dealloc = TRUE; 
    return (response); 
  }

    /* information about all symbolic names */

  if (flag == INFO_ALL_NAMES) {
 
    count    = name_table -> get_entries ();
    nameList = new NAME_LIST (count);
    search   = new HASH_SEARCH (name_table);

    for (i = 0; i < count; i++) {
      node = search -> next ();
      nameList -> fill (i, node -> string_key);
    }

    response = new MESSAGE (down_pipe.messages[RESP_NAMES]);
    response -> elements[0].string  = (char *) nameList;
    response -> elements[0].dealloc = TRUE;
    return (response);
  }

    /* information about a particular agent */

  if (flag == INFO_ID) {

      /* break out the symbolic name and numeric id */

    char *name = message -> elements[1].string;
    UINT_32 id = message -> elements[2].number;

      /* find the agent in the tables */

    if ((name == NULL) || (*name == '\0') || !strcmp(name,"server")) {
      node = agent_table -> lookup (id, NULL);
    } else {
      node = name_table -> lookup (0, name);
    }

      /* error if the agent was not found */

    if (node == NULL) {
      response = new MESSAGE (down_pipe.messages[RESP_ERROR]); 
      return (response); 
    }

      /* assemble the info message */
   
    agent    = (AGENT *) node -> data;
    response = new MESSAGE (down_pipe.messages[RESP_INFO]);
    response -> elements[0].string = agent -> id -> server;
    response -> elements[1].number = agent -> id -> ip;
    response -> elements[2].string = server_data -> host_name;
    response -> elements[3].number = server_data -> host_ip;
    response -> elements[4].string = agent -> id -> name;
    response -> elements[5].number = agent -> id -> id;
    response -> elements[6].number = agent -> pid;
    response -> elements[7].number = agent -> blocked_pid;
    response -> elements[8].number = (UINT_32) agent -> messages.get_count ();
    response -> elements[9].number = (UINT_32) agent -> meetings.get_count ();
    response -> elements[10].number = (UINT_32) agent -> events.get_count ();
    return (response);
  } 

    /* invalid flag */
 
  response = new MESSAGE (down_pipe.messages[RESP_ERROR]); 
  return (response); 
}
 
/* AGENT_DATA::agent_req

   Purpose: Add a meeting request to an agent's queue

     Input: [0] = name of the source server                              (string)
            [1] = IP address of the source server                        (long)
            [2] = name of the source agent                               (string)
            [3] = id of the source agent                                 (long)
            [4] = name of the destination agent                          (string)
            [5] = id of the destination agent                            (long)
	    [6] = MEET_REQUEST, MEET_ACCEPT, MEET_CONNECT or MEET_REFUSE (byte)
            [7] = name of the actual machine                             (string)
            [8] = IP address of the actual machine                       (long)
            [9] = port on the actual machine                             (long)

    Output: The procedure returns the response that should be sent to
            socketd.
*/

MESSAGE *AGENT_DATA::agent_req (MESSAGE *message)
{
  AGENT *agent;
  MESSAGE *response;
  REMOTE_PORT *port;
  AGENT_ID *agent_id;
  INCOMING_MEETING *inc_meeting;

    /* break out the pieces of the id */

  char   *src_server = message -> elements[0].string;
  UINT_32 src_ip     = message -> elements[1].number;
  char   *src_name   = message -> elements[2].string;
  UINT_32 src_id     = message -> elements[3].number;
  char   *dest_name  = message -> elements[4].string;
  UINT_32 dest_id    = message -> elements[5].number;
  UINT_8  status     = message -> elements[6].number; 
  char   *act_server = message -> elements[7].string;
  UINT_32 act_ip     = message -> elements[8].number;
  UINT_32 act_port   = message -> elements[9].number; 

    /* lookup the destination agent */

  if ((agent = lookup_recipient (dest_id, dest_name)) == NULL) {
    response = new MESSAGE (down_pipe.messages[RESP_ERROR]);
    return response;
  }

    /* construct the INCOMING_MEETING structure */

  agent_id    = new AGENT_ID (src_server, src_ip, src_name, src_id);
  port        = new REMOTE_PORT (act_server, act_ip, act_port);
  inc_meeting = new INCOMING_MEETING (agent_id, status, port);

    /* add the meeting to the meeting queue and unblock if necessary */

  agent -> meetings.enqueue ((void *) inc_meeting);

  if (agent -> blocked_item) {
    if (kill (agent -> blocked_pid, SIGUSR1) < 0) {
      char temp[128];
      sprintf (temp, "agentd: unable to send signal to process with pid %d\n", agent -> blocked_pid);
      server_data -> errorlog -> error_sys_cont (temp);
    }
    agent -> blocked_item = FALSE;
  }

    /* success */

  response = new MESSAGE (down_pipe.messages[RESP_OK]);
  return response;
}

/* AGENT_DATA::agent_begin

   Purpose: Add an agent to the table

     Input: [0] = name of the machine on which agent is running (string)
            [1] = IP address of the machine on which agent is running (long)
            [2] = process id of the agent (long)
  
    Output: The procedure finds an unused id, assigns that id to the agent
            and returns the id.  The procedure SHOULD check for the rare
            occurrence of NO id being available but does not (there are
            over 2 billion ids available on a 32-bit machine).
*/

MESSAGE * AGENT_DATA::agent_begin (MESSAGE *message)
{
  int there = TRUE;
  MESSAGE *response;
  AGENT_ID *agent_id;

    /* break out the pieces of the message */

  char *actual_host = message -> elements[0].string;
  UINT_32 actual_ip = message -> elements[1].number;
  UINT_32 pid       = message -> elements[2].number;
   
    /* construct the agent entry */

  if (actual_host == NULL) {
    agent_id = new AGENT_ID (server_data -> host_name, server_data -> host_ip, NULL, 0);
  } else {
    agent_id = new AGENT_ID (actual_host, actual_ip, NULL, 0);
  }

  AGENT *agent = new AGENT (agent_id);
  agent -> pid = pid;

    /* find an unused numeric id */

  while (there) {

    agent_table -> add (next_id, NULL, agent, there);

    if (!there) {
      agent_id -> id = next_id;
    }

    next_id = (next_id + 1) % MAX_32;
  }

    /* we now have one more agent */

  num_agents += 1;

    /* construct the response message */

  response = new MESSAGE (down_pipe.messages[RESP_ID]);
  response -> elements[0].string = server_data -> host_name;
  response -> elements[1].number = server_data -> host_ip;
  response -> elements[2].string = (char *) NULL;
  response -> elements[3].number = agent_id -> id;
  return response; 
}

/* AGENT_DATA::agent_name

   Purpose: Assign a name to an agent

     Input: [0] = numeric id of the agent                  (long)
            [1] = name that should be asigned to the agent (char)

    Output: The procedure returns the response that should be sent to
            socketd.
*/

MESSAGE *AGENT_DATA::agent_name (MESSAGE *message)
{
   int dummy;                      /* dummy integer                            */
   int count = 0;                  /* count of merged items                    */
   HASH_NODE *node;                /* node in a hash table                     */
   AGENT *id_agent;                /* pointer to the agent with the given id   */
   AGENT *name_agent;              /* pointer to the agent with the given name */
   MESSAGE *response;              /* server response                          */
   INCOMING_EVENT *inc_event;      /* merged event                             */
   INCOMING_MESSAGE *inc_message;  /* merged message                           */
   INCOMING_MEETING *inc_meeting;  /* merged meeting                           */

     /* break out the pieces of  the message */

   UINT_32 id = message -> elements[0].number;
   char *name = message -> elements[1].string;

     /* make sure that the name is nonempty and is not reserved */

  if ((name == NULL) || (*name == '\0') || !strcmp(name,"server")) {
    response = new MESSAGE (down_pipe.messages[RESP_ERROR]);
    return response;
  }

     /* lookup the id */

  if ((node = agent_table -> lookup (id, NULL)) == NULL) {
    response = new MESSAGE (down_pipe.messages[RESP_ERROR]);
    return response;
  }
  
  id_agent = (AGENT *) node -> data;

     /* add the name if the name does not exist */

  if ((node = name_table -> lookup (0, name)) == NULL) {

       /* first remove an old name if necessary */

    if (id_agent -> id -> name) {
      name_table -> remove (0, id_agent -> id -> name);
    }

       /* then set up the new name */

    id_agent -> id -> change_name (name);
    name_table -> add (0, name, id_agent, dummy);
    response = new MESSAGE (down_pipe.messages[RESP_OK]);
    return response;
  }

    /* do nothing if the name is already associated with the agent */
 
  name_agent = (AGENT *) node -> data;

  if (name_agent == id_agent) {
    response = new MESSAGE (down_pipe.messages[RESP_OK]);
    return response;
  }

    /* merge the meeting and messages lists if the name exists but is not in use */

  if (name_agent -> id -> server == NULL) {

      /* first remove an old name if necessary */

    if (id_agent -> id -> name) {
      name_table -> remove (0, id_agent -> id -> name);
    }

      /* then set up the new name */

    id_agent -> id -> change_name (name);
   
    while ((inc_message = (INCOMING_MESSAGE *) name_agent -> messages.dequeue ()) != NULL) {
       count += 1;
       id_agent -> messages.enqueue ((void *) inc_message);
    }
    
    while ((inc_meeting = (INCOMING_MEETING *) name_agent -> meetings.dequeue ()) != NULL) {
      count += 1;
      id_agent -> meetings.enqueue ((void *) inc_meeting);
    }

    while ((inc_event = (INCOMING_EVENT *) name_agent -> events.dequeue ()) != NULL) {
      count += 1;
      id_agent -> events.enqueue ((void *) inc_event);
    }

    node -> data = (void *) id_agent;

    if ((count > 0) && (id_agent -> blocked_item)) {
      if (kill (id_agent -> blocked_pid, SIGUSR1) < 0) {
        char temp[128];
        sprintf (temp, "agentd: unable to send signal to process with pid %d\n", id_agent -> blocked_pid);
        server_data -> errorlog -> error_sys_cont (temp);
      }

      id_agent -> blocked_item = FALSE;
    }

    response = new MESSAGE (down_pipe.messages[RESP_OK]);
    delete name_agent;
    return response;
  }

    /* otherwise name is in use */

  response = new MESSAGE (down_pipe.messages[RESP_ERROR]);
  return response;
}

/* AGENT_DATA::agent_end

   Purpose: Remove an agent from the table

     Input: [0] = name of the agent
            [1] = numeric id of the agent
            [2] = END_NORMAL or END_FORCE

    Output: The procedure returns the message that should be sent to 
            socketd.
*/

MESSAGE *AGENT_DATA::agent_end (MESSAGE *message)
{ 
  MESSAGE   *response;
  HASH_NODE *node;
  AGENT     *agent;

    /* break out the pieces of the message */

  char *name      = message -> elements[0].string;
  UINT_32 id      = message -> elements[1].number;
  UINT_8 end_type = message -> elements[2].number;
 	 
    /* find the agent in the tables */

  if ((name == NULL) || (*name == '\0') || !strcmp(name, "server")) {
    node = agent_table -> lookup (id, NULL); 
  } else {
    node = name_table -> lookup (0, name);
  }

    /* error if the agent was not found */

  if (node == NULL) {
    response = new MESSAGE (down_pipe.messages[RESP_ERROR]);
    return response;
  }

  agent = (AGENT *) node -> data;

    /* kill the agent if we are doing an agent_force */

  if (end_type == END_FORCE) {
 
      /* the blocker */

    if (agent -> blocked_pid != 0) {
      kill (agent -> blocked_pid, SIGKILL);
    }       

      /* the agent itself */

    if ((agent -> pid != 0) && (agent -> id -> ip == server_data -> host_ip)) {
      kill (agent -> pid, SIGKILL);
    }
  }

    /* remove the agent from the tables */

  if (agent -> id -> server) {
    agent_table -> remove (agent -> id -> id, NULL); 
  }

  if (agent -> id -> name) {
    name_table -> remove (0, agent -> id -> name);
  }

    /* construct the response message */

  response = new MESSAGE (down_pipe.messages[RESP_ID]);
  response -> elements[0].string  = server_data -> host_name;
  response -> elements[1].number  = server_data -> host_ip;
  response -> elements[2].string  = my_strcpy (agent -> id -> name);
  response -> elements[2].dealloc = TRUE;
  response -> elements[3].number  = agent -> id -> id; 

    /* remove the agent */

  delete agent;
  num_agents -= 1;

    /* return the response message */

  return response;  
}

/* AGENT_DATA::agent_event

   Purpose: Add an event to an agent's queue

     Input: [0] = source server      (string)
            [1] = source IP address  (long)
            [2] = source name        (string)
            [3] = source id          (long)
            [4] = destination name   (byte)
            [5] = destination id     (long)
            [6] = event tag          (string)
            [7] = event string       (string)

    Output: The procedure returns the message that should be sent to socketd.
*/

MESSAGE *AGENT_DATA::agent_event (MESSAGE *message)
{
  AGENT *agent;
  MESSAGE *response;
  AGENT_ID *agent_id;
  INCOMING_EVENT *inc_event;

    /* break out the pieces of the message */

  char *src_server = message -> elements[0].string;
  UINT_32 src_ip   = message -> elements[1].number;
  char *src_name   = message -> elements[2].string;
  UINT_32 src_id   = message -> elements[3].number; 
  char *name       = message -> elements[4].string;
  UINT_32 id       = message -> elements[5].number; 
  char *tag        = message -> elements[6].string;
  char *string     = message -> elements[7].string; 

    /* lookup the destination agent */

  if ((agent = lookup_recipient (id, name)) == NULL) {
    response = new MESSAGE (down_pipe.messages[RESP_ERROR]);
    return response;
  }

    /* construct the INCOMING_EVENT structure */

  agent_id  = new AGENT_ID (src_server, src_ip, src_name, src_id);
  inc_event = new INCOMING_EVENT (agent_id, tag, string);

    /* add the message to the queue and unblock if necessary */ 

  agent -> events.enqueue ((void *) inc_event);

  if (agent -> blocked_item) {
    if (kill (agent -> blocked_pid, SIGUSR1) < 0) {
      char temp[128];
      sprintf (temp, "agentd: unable to send signal to process with pid %d\n", agent -> blocked_pid);
      server_data -> errorlog -> error_sys_cont (temp);
    }
    agent -> blocked_item = FALSE;
  }

  response = new MESSAGE (down_pipe.messages[RESP_OK]);
  return response;
}
 
/* AGENT_DATA::agent_send

   Purpose: Add a message to an agent's queue

     Input: [0] = source server      (string)
            [1] = source IP address  (long)
            [2] = source name        (string)
            [3] = source id          (long)
            [4] = destination name   (byte)
            [5] = destination id     (long)
            [6] = message code       (long)
            [7] = message string     (string)

    Output: The procedure returns the message that should be sent to socketd.
*/
       
MESSAGE *AGENT_DATA::agent_send (MESSAGE *message)
{
  AGENT *agent;
  MESSAGE *response;
  AGENT_ID *agent_id;
  INCOMING_MESSAGE *inc_message;

    /* break out the pieces of the message */

  char *src_server = message -> elements[0].string;
  UINT_32 src_ip   = message -> elements[1].number;
  char *src_name   = message -> elements[2].string;
  UINT_32 src_id   = message -> elements[3].number; 
  char *name       = message -> elements[4].string;
  UINT_32 id       = message -> elements[5].number; 
  UINT_32 code     = message -> elements[6].number;
  char *string     = message -> elements[7].string; 

    /* lookup the destination agent */

  if ((agent = lookup_recipient (id, name)) == NULL) {
    response = new MESSAGE (down_pipe.messages[RESP_ERROR]);
    return response;
  }

    /* construct the INCOMING_MESSAGE structure */

  agent_id    = new AGENT_ID (src_server, src_ip, src_name, src_id);
  inc_message = new INCOMING_MESSAGE (agent_id, code, string);

    /* add the message to the queue and unblock if necessary */ 

  agent -> messages.enqueue ((void *) inc_message);

  if (agent -> blocked_item) {
    if (kill (agent -> blocked_pid, SIGUSR1) < 0) {
      char temp[128];
      sprintf (temp, "agentd: unable to send signal to process with pid %d\n", agent -> blocked_pid);
      server_data -> errorlog -> error_sys_cont (temp);
    }
    agent -> blocked_item = FALSE;
  }

  response = new MESSAGE (down_pipe.messages[RESP_OK]);
  return response;
}

/* AGENT_DATA::agent_receive

   Purpose: See if there is a result pending for an agent
   
     Input: [0] = pointer to the agent id      (long)
            [1] = pointer to the process id    (long)

    Output: The procedure returns the message that should be sent to socketd.
*/

MESSAGE *AGENT_DATA::agent_receive (MESSAGE *message)
{
  AGENT *agent;
  MESSAGE *response;
  HASH_NODE *hash_node;
  INCOMING_MESSAGE *inc_message;
  INCOMING_MEETING *inc_meeting;
  INCOMING_EVENT   *inc_event;

    /* break out the pieces of the message */

  UINT_32 id  = message -> elements[0].number;
  UINT_32 pid = message -> elements[1].number;

    /* error if there is no agent with the specified id */

  if ((hash_node = agent_table -> lookup (id, NULL)) == NULL) {
    response = new MESSAGE (down_pipe.messages[RESP_ERROR]);
    return response;
  }

    /* get the first available event, meeting or message for the agent */

  agent = (AGENT *) hash_node -> data;

  if ((inc_event = (INCOMING_EVENT *) agent -> events.dequeue ()) != NULL) {
    response = new MESSAGE (down_pipe.messages[RESP_EVENT]);
    response -> elements[0].string  = my_strcpy (inc_event -> id -> server);
    response -> elements[0].dealloc = TRUE;
    response -> elements[1].number  = inc_event -> id -> ip;
    response -> elements[2].string  = my_strcpy (inc_event -> id -> name);
    response -> elements[2].dealloc = TRUE;
    response -> elements[3].number  = inc_event -> id -> id;
    response -> elements[4].string  = my_strcpy (inc_event -> tag);
    response -> elements[4].dealloc = TRUE;
    response -> elements[5].string  = my_strcpy (inc_event -> string);
    response -> elements[5].dealloc = TRUE;
    delete inc_event;
  } else if ((inc_meeting = (INCOMING_MEETING *) agent -> meetings.dequeue ()) != NULL) {
    response = new MESSAGE (down_pipe.messages[RESP_MEETING]);
    response -> elements[0].string  = my_strcpy (inc_meeting -> id -> server);
    response -> elements[0].dealloc = TRUE;
    response -> elements[1].number  = inc_meeting -> id -> ip;
    response -> elements[2].string  = my_strcpy (inc_meeting -> id -> name);
    response -> elements[2].dealloc = TRUE;
    response -> elements[3].number  = inc_meeting -> id -> id;
    response -> elements[4].number  = inc_meeting -> status;
    response -> elements[5].string  = my_strcpy (inc_meeting -> port -> machine);
    response -> elements[6].number  = inc_meeting -> port -> ip;
    response -> elements[7].number  = inc_meeting -> port -> port;
    delete inc_meeting;
  } else if ((inc_message = (INCOMING_MESSAGE *) agent -> messages.dequeue ()) != NULL) {
    response = new MESSAGE (down_pipe.messages[RESP_MESSAGE]);
    response -> elements[0].string  = my_strcpy (inc_message -> id -> server);
    response -> elements[0].dealloc = TRUE;
    response -> elements[1].number  = inc_message -> id -> ip;
    response -> elements[2].string  = my_strcpy (inc_message -> id -> name);
    response -> elements[2].dealloc = TRUE;
    response -> elements[3].number  = inc_message -> id -> id;
    response -> elements[4].number  = inc_message -> code;
    response -> elements[5].string  = my_strcpy(inc_message -> string);
    response -> elements[5].dealloc = TRUE;
    delete inc_message;
  } else {
    response = new MESSAGE (down_pipe.messages[RESP_NONE]);

      /* will send SIGUSR1 when a message, meeting or event arrives */

    agent -> blocked_item = TRUE;
    agent -> blocked_pid  = pid;
  }

  return response;
}
