/* Agent Tcl
   Bob Gray
   9 February 1995

   agentd.cc

   This is the top-level program for the agent server.  On startup it forks
   the process "socketd" which monitors the agent port.  Then its task is to
   maintain the data structures that keep track of agents executing on its
   machine.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <setjmp.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#ifndef COSMO
#include <sys/wait.h>
#else
#include </usr/local/lib/gcc-lib/rs6000-ibm-aix3.2.5/2.6.3/include/sys/wait.h>
#endif
#ifndef COSMO
#include <unistd.h>
#else
#include </usr/local/lib/gcc-lib/rs6000-ibm-aix3.2.5/2.6.3/include/unistd.h>
#endif
#include "agent_data.h"
#include "agentd.h"
#include "daemon.h"
#include "error.h"
#include "hash.h"
#include "interp.h"
#include "interrupt.h"
#include "my_sizes.h"
#include "my_strings.h"
#include "parse.h"
#include "piped.h"
#include "redirect.h"
#include "socketd.h"
#include "stream_pipe.h"
#include "truefalse.h"
#include "tcpip.h"

   /* size of the access and interpreter tables */

const ACCESS_TABLE_SIZE = 0x100;
const INTERP_TABLE_SIZE = 0x010;

   /* maximum size of a word */

const MAX_WORD_SIZE = 2048;

   /* server and agent information */

SERVER_DATA *server_data;
AGENT_DATA  *agent_data;   

   /* child handler */

static void sigchld_handler (int signo)
{
   int status;    

    /* remove the zombie */

   while (waitpid(0, &status, WNOHANG) > 0);

    /* log an error message and quit */

  server_data -> errorlog ->
    error_app_quit 
      ("agentd: socketd has failed so agentd is terminating"); 
}

/* verify_server_ip

   Purpose: Verify that the machine specified with the -host parameter
            is the machine on which the server is running

     Input: None

    Output: The procedure returns FALSE on error or if the specified machine
            is not the machine on which the server is running.  Otherwise
            the procedure returns TRUE.
*/

int verify_server_ip (void)
{
  char *hostname;
  UINT_32 hostip;

  fprintf (stderr, "\nVerifying server IP address ...\n\n");
 
    /* get the IP number of machine specified with the -host parameter */
  
  if (tcpip_getip (server_data -> host_name, &server_data -> host_ip) < 0) {
    fprintf (stderr, "unable to get IP address of %s\n", server_data -> host_name);
    return FALSE; 
  }

    /* get the name of the the actual machine */

  if ((hostname = tcpip_hostname ()) == NULL) {
    fprintf (stderr, "unable to get actual name of host\n");
    return FALSE;
  }
  
    /* get the IP address of the actual machine */

  if (tcpip_getip (hostname, &hostip) < 0) {
    fprintf (stderr, "unable to get actual IP address of host\n");
    return FALSE;
  }

    /* compare the IP addresses */

  if (server_data -> host_ip != hostip) {
    fprintf (stderr, "machine after -host (%s) is NOT the machine on which the server is being started (%s)\n", server_data -> host_name, hostname);
    return FALSE;
  }

  fprintf (stderr, "OK!\n");
  return TRUE;
}

/* load_interpreter_list

   Purpose: Load the set of interpreters that are available through the server

     Input: None

    Output: The procedure returns FALSE if an error occurs while opening or
            reading from the access file.  Otherwise the procedure loads the
            set of interpreters and returns TRUE.
*/

static int load_interpreter_list (void)
{
  FILE *fp;
  int there;
  int i = 0;
  char *words[5];
  INTERP *interp;
  int interp_type;
  struct stat fileInfo;

  fprintf (stderr, "\nProcessing interpreter list ...\n\n");

     /* open the file that contains the interpreters */

  if ((fp = fopen (server_data -> lang, "r")) == NULL) {
    fprintf (stderr, "unable to open interpreter file \"%s\"\n", server_data -> lang);
    return FALSE;
  }

    /* create the parser and the hash table */

  server_data -> interp_table = new HASH (INTERP_TABLE_SIZE);
  PARSE parser (fp, MAX_WORD_SIZE);

    /* parse the file */

  while (parser.next_word (&words[i]) >= 0) {

      /* have we reached the end of the words */

    if (words[i] == NULL) {
      if (i != 0) {
        fprintf (stderr, "line %d: unexpected end of entry\n", parser.current_line());
        return FALSE;
      } else {
        return TRUE;
      }
    }  

      /* continue if we have not reached the end of an entry */

    if (++i != 5) {
      continue; 
    } 

      /* otherwise check the executable name */

    if (words[1][0] != '/') {
      fprintf (stderr, "line %d: interpreter name \"%s\" is not fully qualified\n", parser.current_line(), words[1]);
      return FALSE;
    }

    if (stat (words[1], &fileInfo) < 0) {
      fprintf (stderr, "line %d: interpreter \"%s\" does not exist\n", parser.current_line(), words[1]);
      return FALSE;
    }

    if (fileInfo.st_mode & S_IFDIR) {
      fprintf (stderr, "line %d: \"%s\" is a directory: should be an interpreter\n", parser.current_line(), words[1]);
      return FALSE;
    }

    if (access (words[1], X_OK) < 0) {
      fprintf (stderr, "line %d: interpreter \"%s\" is not executable\n", parser.current_line(), words[1]);
      return FALSE;
    }
 
      /* check the socket directory */

    if (words[3][0] != '/') {
      fprintf (stderr, "line %d: socket directory \"%s\" is not fully qualified\n", parser.current_line(), words[3]);
      return FALSE;
    }

    if (stat (words[3], &fileInfo) < 0) {
      fprintf (stderr, "line %d: directory \"%s\" does not exist\n", parser.current_line(), words[3]);
      return FALSE;
    }

    if (!(fileInfo.st_mode & S_IFDIR)) {
      fprintf (stderr, "line %d: \"%s\" is a file: should be a socket directory\n", parser.current_line(), words[3]);
      return FALSE;
    }


    if (access (words[3], W_OK) < 0) {
      fprintf (stderr, "line %d: socket directory \"%s\" is not writeable\n", parser.current_line(), words[3]);
      return FALSE;
    }
        

      /* otherwise check the directory name */

    
      /* otherwise add the entry to the interpreter table */

    if (!strcmp(words[4], "STATE")) {
      interp_type = STATE_INTERP;
    } else if (!strcmp(words[4], "NORMAL")) {
      interp_type = NORMAL_INTERP;
    } else {
      fprintf (stderr, "line %d: expected STATE or NORMAL\n", parser.current_line());
      return FALSE;
    }

    interp = new INTERP (words[0], words[1], words[2], words[3], interp_type);
    server_data -> interp_table -> add (0, words[0], (void *) interp, there);

    if (there) {
      fprintf (stderr, "entry ending at line %d: duplicate symbolic name \"%s\"\n", parser.current_line(), words[0]);
      return FALSE;
    } else {
      fprintf (stderr, "Adding \"%s\" to interpreter list\n", words[0]);
    }

      /* clean up for next go */

    for (int j = 0; j < 5; j++) {
      delete (words[j]);
    }

    i = 0;
  }

    /* error when reading a word */

  fprintf (stderr, "line %d: word is too long\n", parser.current_line());
  return FALSE;
} 
            
/* load_access_list

   Purpose: Load the list of machines that are allowed to use the agent
            server

     Input: None

    Output: The procedure returns FALSE if an error occurs while opening
            or reading from the access file.  Otherwise the procedure 
            reads each line of the file, converts the host name on that
            line into an IP address and adds the IP address to the access
            table.  The procedure then returns TRUE.
*/

static int load_access_list (void)
{
  FILE *fp;
  int dummy;
  char *machine;
  int count = 0;      /* number of machine names in the access file */
  struct in_addr address;

  fprintf (stderr, "\nProcessing access list ...\n\n");

    /* open the access file */

  if ((fp = fopen (server_data -> access, "rb")) == NULL)
  {
    fprintf (stderr, "agentd: unable to open access file %s\n", server_data -> access);
    return FALSE;
  }
 
    /* create the hash table and the parser */

  server_data -> access_table = new HASH (ACCESS_TABLE_SIZE);
  PARSE parser (fp, MAX_WORD_SIZE);

    /* read each machine */

  while (parser.next_word (&machine) >= 0) {

       /* out of machines */

    if (machine == NULL) {
      if (count == 0) { 
        fprintf (stderr, "warning: no valid machines in the access file\n");
      }
      return TRUE;
    }

      /* add the machine to the access list */
 
    if (tcpip_getip (machine, (UINT_32 *) &address.s_addr) < 0) {
      fprintf (stderr, "line %d: unable to get IP address of %s\n", parser.current_line(), machine);
    } else {
      fprintf (stdout, "Adding %15s (%s) to access list\n", inet_ntoa(address), machine);
      server_data -> access_table -> add (address.s_addr, NULL, NULL, dummy); 
      count += 1;
    }

      /* clean up for the next go around */

    delete machine;
  }

    /* error when reading a machine name */

  fprintf (stderr, "line %d: machine name is too long\n", parser.current_line());
  return FALSE;
} 

/* print_usage

   Purpose: Print usage instructions to standard output

     Input: name = name that was used to start the program

    Output: The procedure outputs usage instructions to standard output.
*/

void print_usage (char *name)
{
  int i;
  int len      = strlen(name);
  char *spaces = new char [len + 1];

    /* initialize the spacing string */

  for (i = 0; i < len; i++) {
    spaces[i] = ' ';
  }

  spaces[i] = '\0';

    /* print usage instructions */

  printf ("\n");
  printf ("USAGE:\n\n");
  printf ("%s -host <hostname>\n", name);
  printf ("%s -log <filename> -lock <filename>\n", spaces);
  printf ("%s -access <filename> -lang <filename>\n", spaces);
  printf ("%s [-tracker] [-nodaemon]\n\n", spaces);
}

/* get_server_data

   Purpose: Get the command line parameters

     Input: argc = number of command line parameters
            argv = the command line arguments

    Output: The procedure returns FALSE if the command line parameters are
            specified incorrectly.  Otherwise the procedure fills in the
            the SERVER_DATA structure and returns TRUE.
*/

int get_server_data (int argc, char **argv)
{
  int access_found = FALSE;      /* access file specified      */
  int log_found    = FALSE;      /* log file specified         */
  int host_found   = FALSE;      /* host specified             */
  int lock_found   = FALSE;      /* lock file specified        */
  int lang_found   = FALSE;      /* interpreter file specified */
  unsigned numArgs = argc - 1;   /* number of arguments        */ 
  unsigned curArg  = 1;          /* current argument           */

    /* assign the default values */

  server_data -> daemon  = defaultDaemon;
  server_data -> tracker = defaultTracker;
 
    /* get the arguments from the command line */

  while (curArg <= numArgs) {
    if (!strcmp(argv[curArg], "-host")) {

      if (++curArg > numArgs) {
        return FALSE;
      }  

      server_data -> host_name = my_strcpy (argv[curArg]);
      host_found            = TRUE;

    } else if (!strcmp(argv[curArg], "-access")) {

      if (++curArg > numArgs) {
        return FALSE;
      }

      server_data -> access = my_strcpy (argv[curArg]);
      access_found       = TRUE;

    } else if (!strcmp(argv[curArg], "-lang")) {

      if (++curArg > numArgs) {
        return FALSE;
      }

      server_data -> lang = my_strcpy (argv[curArg]);
      lang_found       = TRUE;

    } else if (!strcmp(argv[curArg], "-lock")) {

      if (++curArg > numArgs) {
        return FALSE;
      }  

      server_data -> lock = my_strcpy (argv[curArg]);
      lock_found       = TRUE;

    } else if (!strcmp(argv[curArg], "-log")) {

      if (++curArg > numArgs) {
        return FALSE;
      }  

      server_data -> error = my_strcpy (argv[curArg]);
      log_found         = TRUE;

    } else if (!strcmp(argv[curArg], "-nodaemon")) {
      server_data -> daemon = FALSE;
    } else if (!strcmp(argv[curArg], "-tracker")) {
      server_data -> tracker = TRUE;
    } else {
      return FALSE;
    }

    curArg++;
  }

  return (host_found && log_found && lock_found && access_found && lang_found);
}

/* redirect_output

   Purpose: Close all open files except for stdin, stdout and stderr.
            Redirect stdin, stdout and stderr to /dev/null.

     Input: None

    Output: The procedure halts the server and logs an error message if
            it is unable to redirect stdin, stdout and stderr.
*/

static void redirect_output (void)
{
  if (fileRedirect() < 0) {
    server_data -> errorlog ->
      error_sys_quit ("agentd: unable to redirect stdin, stdout and stderr");
  }
}

int main (int argc, char **argv)
{
  pid_t childPid;               /* pid of the child process            */
  int fd[2];                    /* file descriptors                    */

    /* create sever_data and agent_data */

  server_data = new SERVER_DATA ();
  agent_data  = new AGENT_DATA ();

    /* get the command line parameters */

  if (!get_server_data (argc, argv)) {
    print_usage (argv[0]);
    exit (1);  
  }

    /* make sure that the filenames are fully specified */

  if ((server_data -> error[0] != '/') || (server_data -> lock[0] != '/')) {
    fprintf (stderr, "ERROR: lock and log filenames must be fully qualified\n");
    exit (1);
  }

    /* load the interpreter and access lists */

  if (!load_interpreter_list ()) {
    exit (1);
  }

  if (!load_access_list ()) {
    exit (1);
  }

    /* get the IP address of the current machine */

  if (!verify_server_ip()) {
    exit (1);
  }

    /* print the tracking status */

  fprintf (stderr, "\nTracking status ...\n\nThe tracker agent is ");

  if (server_data -> tracker) {
    fprintf (stderr, "ON.\n");
  } else {
    fprintf (stderr, "OFF.\n");
  }
  
  fprintf (stderr, "\n");
 
    /* turn the server into a daemon */
 
  if (server_data -> daemon) {
    server_data -> errorlog = new ERROR (server_data -> error);
    make_daemon ();
    redirect_output ();
  } else {
    server_data -> errorlog = new ERROR (NULL);
  }

    /* ignore SIGPIPE so that we do not die on pipe breakdown */
    /* catch  SIGCHLD so that we can detect if socketd dies   */

  if (install_signal (SIGPIPE, (void (*)(...)) SIG_IGN, 0) < 0) { 
    server_data -> errorlog ->
      error_sys_quit 
        ("agentd: unable to ignore SIGPIPE");
  }

  if (install_signal (SIGCHLD, (void (*)(...)) sigchld_handler, SA_NOCLDSTOP) < 0) {
    server_data -> errorlog ->
      error_sys_quit 
        ("agentd: unable to install SIGCHLD handler");
  }

    /* create the pipes */ 

  if (stream_pipe (fd) < 0) {
    server_data -> errorlog ->
      error_sys_quit ("agentd: unable to create the stream pipe\n");
  }

  server_data -> streamSocketd = fd[0];
  server_data -> streamAgentd  = fd[1]; 

    /* open the socket descriptor that brings in external requests */

  if ((server_data -> sockfd = tcpip_setup (SERVER_TCP_PORT, NULL)) < 0)
    server_data -> errorlog ->
      error_sys_quit ("agentd: unable to set up external socket");

    /* fork off socketd */
 
  if ((childPid = fork()) < 0) {

      /* FORK failed */

    server_data -> errorlog -> 
      error_sys_quit 
        ("agentd: unable to fork \"socketd\"");

  } else if (childPid == 0) {
      
      /* CHILD = socketd */

    close (server_data -> streamSocketd);       
    socketd ();

  } else {

       /* PARENT = agentd */

    char temp[64];
    sprintf (temp, "agentd: socketd has been started (process id %ld)", (long) childPid);
    server_data -> errorlog -> 
      error_app_cont 
	(temp);

    close (server_data -> sockfd);
    close (server_data -> streamAgentd);
    piped ();
  }

    /* should never get here -- prevents a compiler warning with CC */

  return 1;   
}
