/* Agent Tcl
   Bob Gray
   9 February 1995

   tcl_receive.cc

   This file implements the routines that the server uses to RECEIVE a 
   client request.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h> 
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#ifndef COSMO
#include <unistd.h>
#else
#include </usr/local/lib/gcc-lib/rs6000-ibm-aix3.2.5/2.6.3/include/unistd.h>
#endif
#include "agentd.h"
#include "error.h"
#include "hash.h"
#include "interrupt.h"
#include "message.h"
#include "my_alloc.h"
#include "piped.h"
#include "tcl_receive.h"
#include "tcl_run.h"
#include "tcpip.h"
#include "transmit.h"
#include "truefalse.h"

typedef int (*Action) (int, MESSAGE *);

static Action jumpTable[NUM_REQUESTS] =
{
  &tcl_send,		/* REQ_BEGIN	*/
  &tcl_send,		/* REQ_NAME	*/
  &tcl_send,		/* REQ_END	*/
  &tcl_run,		/* REQ_SCRIPT	*/
  &tcl_run,		/* REQ_STATE	*/
  &tcl_send,		/* REQ_MESSAGE	*/
  &tcl_send,		/* REQ_MEETING  */
  &tcl_send,		/* REQ_EVENT	*/
  &tcl_receive,		/* REQ_GET	*/
  &tcl_encrypted,	/* REQ_ENCRYPT  */
  &tcl_send,		/* REQ_INFO     */
  &tcl_sign             /* REQ_SIGN     */
};

/* tcl_action

   Purpose: Handle the incoming transmission

     Input: sockfd = socket descriptor

    Output: The first byte of the transmission must be one of

	    REQ_BEGIN
            REQ_NAME
            REQ_END
            REQ_SCRIPT
            REQ_STATE
            REQ_MESSAGE
            REQ_MEETING
            REQ_EVENT
            REQ_GET
            REQ_ENCRYPT
            REQ_SIGN

            The procedure reads the transmission from the socket, checks
            the first byte and takes the appropriate action.  The procedure
            returns -1 on error and 0 otherwise.
*/

int tcl_action (int sockfd)
{
  UINT_32 port;
  int return_code = 0;
  struct in_addr address;
  MESSAGE *message = NULL;

     /* install the SIGCHLD handler */

  if (install_signal (SIGCHLD, (void (*) (...)) SIG_DFL, SA_NOCLDSTOP) < 0) {
    server_data -> errorlog ->
      error_sys_quit
	("tcl_action: unable to install default SIGCHLD");
  }

    /* check that the request comes from an allowed host */

  if (tcpip_getpeername (sockfd, (UINT_32 *) &address.s_addr, &port) < 0) {

    server_data -> errorlog -> 
      error_sys_cont
        ("tcl_action: unable to determine identity of connected client");

    return_code = -1;

  } else if (server_data -> access_table -> lookup (address.s_addr, NULL) == NULL) {

    char temp[256];
    sprintf (temp, "tcl_action: attemp to connect from disallowed host.  IP address was %s", inet_ntoa (address));
    server_data -> errorlog ->
      error_app_cont
        (temp);

    return_code = -1;

  } else if ((message = message_receive (sockfd, up_socket)) == NULL) {

    server_data -> errorlog -> 
      error_app_cont
        ("tcl_action: invalid or interrupted message\n");

    return_code = -1;
  }

    /* send an error message if necessary */

  if (return_code < 0) {
    MESSAGE error_message (down_socket.messages[RESP_ERROR]);
    message_send (sockfd, error_message);
    return return_code;
  }

    /* take the appropriate action */

  return_code = (*jumpTable[message -> flag]) (sockfd, message); 
  delete_check (message);
  return return_code;
}
