/* Agent Tcl
   Bob Gray
   9 February 1995

   tcl_run.cc

   This file implements the functions that the server uses to HANDLE a client
   request.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <sys/file.h>
#include <sys/wait.h>
#include <sys/time.h>
#ifdef AIX
#include <sys/select.h>
#endif
#ifdef SYSV
#ifndef SOLARIS
#include <bstring.h>
#endif
#endif
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef COSMO
#include <unistd.h>
#else
#include </usr/local/lib/gcc-lib/rs6000-ibm-aix3.2.5/2.6.3/include/unistd.h>
#endif
#include <errno.h>
#include "agentd.h"
#include "error.h"
#include "interp.h"
#include "locks.h"
#include "message.h"
#include "my_alloc.h"
#include "my_strings.h"
#include "piped.h"
#include "tcpip.h"
#include "tcl_run.h"
#include "transmit.h"
#include "truefalse.h"

/* respond_with_error

   Purpose: Send a RESP_ERROR message down the socket

     Input: sockfd = socket descriptor
            string = error string

    Output: The procedure constructs a RESP_ERROR message containing the
            given string and sends the RESP_ERROR message down the socket.
*/

static void respond_with_error (int sockfd, char *string)
{
  MESSAGE message (down_socket.messages[RESP_ERROR]);
  message_send (sockfd, message);
}

/* send_message

   Purpose: Send a message from a server agent to a second agent.  The second
            agent must be on the same machine as the server.

     Input: server_id  = numeric id of the server agent
            recip_name = name of the recipient
            recip_id   = numeric id of the recipient
            code       = message code
            string     = message string

    Output: The procedure returns -1 on error.  Otherwise the procedure
            sends the given message to the specified recipient.
*/

static int send_message (UINT_32 server_id, char *recip_name, UINT_32 recip_id, int code, char *string)
{ 
  int return_code = 0;
  MESSAGE *response = NULL;

    /* construct the message */

  MESSAGE message (up_pipe.messages[REQ_MESSAGE]);
  message.elements[0].string = server_data -> host_name;
  message.elements[1].number = server_data -> host_ip;
  message.elements[2].string = "server";
  message.elements[3].number = server_id;
  message.elements[4].string = recip_name;
  message.elements[5].number = recip_id;
  message.elements[6].number = code;
  message.elements[7].string = string;

    /* send the message to the server */

  get_lock (server_data -> lockfd, server_data -> errorlog);

  if (message_send (server_data -> streamAgentd, message) >= 0) {
    response = message_receive (server_data -> streamAgentd, down_pipe);
  }

  release_lock (server_data -> lockfd, server_data -> errorlog);

    /* return -1 on server error */

  if ((response == NULL) || (response -> flag == RESP_ERROR)) {
    return_code = -1;
  }

  delete_check (response);
  return (return_code);
}
 
/* get_item

   Purpose: Get the next available message, event or meeting request for a
            particular agent

     Input: id = numeric id of the agent

    Output: The procedure returns a dynamically allocated MESSAGE structure
            which contains a RESP_ERROR, RESP_MEETING, RESP_MESSAGE or
            RESP_EVENT message.
*/

static MESSAGE *get_item (UINT_32 id)
{
  sigset_t newmask;
  sigset_t oldmask;
  MESSAGE *response = NULL;
 
    /* set up the pipe message */

  MESSAGE message (up_pipe.messages[REQ_GET]);
  message.elements[0].number = id;
  message.elements[1].number = (UINT_32) getpid();

    /* initialize the signal masks */

  sigemptyset (&oldmask);
  sigemptyset (&newmask);
  sigaddset   (&newmask, SIGUSR1);

    /* block SIGUSR1 */
 
  if (sigprocmask (SIG_BLOCK, &newmask, &oldmask) < 0) {
    server_data -> errorlog ->
      error_sys_quit 
	("tcl_receive: unable to block SIGUSR1");
  }

    /* iterate */

  while (1) {

      /* see if we have an available item */

    get_lock (server_data -> lockfd, server_data -> errorlog);

    if (message_send (server_data -> streamAgentd, message) >= 0) {
      response = message_receive (server_data -> streamAgentd, down_pipe);
    }

    release_lock (server_data -> lockfd, server_data -> errorlog);

      /* break on server error or a response */

    if (response == NULL) {
      response = new MESSAGE (down_pipe.messages[RESP_ERROR]);
      break;
    } else if (response -> flag != RESP_NONE) {
      break;
    } 

      /* wait for an incoming item */

    if (sigsuspend (&oldmask) != -1) {
      server_data -> errorlog ->
        error_sys_quit 
          ("tcl_run: unable to pause and wait for SIGUSR1");
    }

    delete response;
  }

    /* unblock SIGUSR1 */

  if (sigprocmask (SIG_SETMASK, &oldmask, NULL) < 0) {
    server_data -> errorlog ->
      error_sys_quit 
	("tcl_run: unable to unblock SIGUSR1");
  }

  return (response);
}

/* remove_id

   Purpose: Remove an agent

     Input: id = numeric id of the agent

    Output: The procedure returns -1 on error.  Otherwise the procedure
            removes the agent from the tables and returns 0.
*/

static int remove_id (UINT_32 id)
{
  int code = 0;
  MESSAGE *response = NULL;

    /* construct the REQ_END message */

  MESSAGE message (up_pipe.messages[REQ_END]);
  message.elements[0].string = NULL;
  message.elements[1].number = id;
  message.elements[2].number = END_NORMAL;

    /* send the message to the server */

  get_lock (server_data -> lockfd, server_data -> errorlog);

  if (message_send (server_data -> streamAgentd, message) >= 0) {
    response = message_receive (server_data -> streamAgentd, down_pipe);
  }

  release_lock (server_data -> lockfd, server_data -> errorlog);

    /* return -1 on error */

  if ((response == NULL) || (response -> flag != RESP_ID)) {
    code = -1;
  }

  delete_check (response);
  return (code);
}

/* get_id

   Purpose: Get a local identification for the agent

     Input: pid = process id of the agent interpreter

    Output: The procedure returns NULL on error and returns the new local
            identification of the agent on success.  The new local
            identification is dynamically allocated.
*/

static AGENT_ID *get_id (pid_t pid)
{
  AGENT_ID *id = NULL;
  MESSAGE *response = NULL; 

    /* set up the message */

  MESSAGE message (up_pipe.messages[REQ_BEGIN]);
  message.elements[0].string = NULL;
  message.elements[1].number = 0;
  message.elements[2].number = pid;

    /* send the message to the server */

  get_lock (server_data -> lockfd, server_data -> errorlog);

  if (message_send (server_data -> streamAgentd, message) >= 0) {
    response = message_receive (server_data -> streamAgentd, down_pipe);
  }

  release_lock (server_data -> lockfd, server_data -> errorlog);

    /* did the server respond */

  if ((response != NULL) && (response -> flag == RESP_ID)) {
    char *server       = response -> elements[0].string;
    UINT_32 ip         = response -> elements[1].number;
    UINT_32 numeric_id = response -> elements[3].number;
    id                 = new AGENT_ID (server, ip, NULL, numeric_id);
  }

    /* return the id */

  delete_check (response);
  return (id);
}
 
  /* get a local identification from the server */

static AGENT_ID *register_with_agentd (int sockfd, pid_t pid) 
{
  AGENT_ID *id;
  MESSAGE *message;

    /* get an identification for the agent */

  if ((id = get_id (pid)) == NULL) {
    message = new MESSAGE (down_socket.messages[RESP_ERROR]);
  } else {
    message = new MESSAGE (down_socket.messages[RESP_ID]);
    message -> elements[0].string = id -> server;
    message -> elements[1].number = id -> ip;
    message -> elements[2].string = (char *) NULL;
    message -> elements[3].number = id -> id;
  }

    /* send the identification to the client */

  message_send (sockfd, *message);
  delete (message);
  return id;
}

  /* send a message to the root agent on abnormal interpreter termination */

static int send_result (AGENT_ID *local, AGENT_ID *root)
{
  int sockfd;
  int sleep_time = 1;

    /* do not send to ourselves */

  if (root -> server == NULL) {
    return 0;
  } 

     /* set up the message */

  MESSAGE mesg_to_root (up_socket.messages[REQ_MESSAGE]);
  mesg_to_root.elements[0].string = local -> server;
  mesg_to_root.elements[1].number = local -> ip;
  mesg_to_root.elements[2].string = local -> name;
  mesg_to_root.elements[3].number = local -> id;
  mesg_to_root.elements[4].string = (char *) NULL; 
  mesg_to_root.elements[5].number = root -> id;
  mesg_to_root.elements[6].number = 1;
  mesg_to_root.elements[7].string = "abnormal interpreter exit";

    /* send the result */

  sockfd = message_conn_and_send (NULL, root -> ip, mesg_to_root, -1, FALSE);

  while ((sockfd < 0) && (sleep_time < MAXIMUM_SLEEP)) {
    sleep (sleep_time);
    sleep_time *= 2;
    sockfd = message_conn_and_send (NULL, root -> ip, mesg_to_root, -1, FALSE);
  }

    /* log any error */

  if (sockfd < 0) {
    server_data -> errorlog ->
      error_app_cont 
	("tcl_run: unable to send error message to root agent");

    return -1;
  }

  close (sockfd);
  return 0;
} 

/* tcl_run

   Purpose: This procedure handles the messages

            REQ_SCRIPT
            REQ_STATE

     Input: sockfd  = socket on which the script is arriving
            message = the message

    Output: The procedure returns -1 if it is unable to start execution.
            Otherwise the procedure returns 0.
*/

int tcl_run (int sockfd, MESSAGE *message)
{
  pid_t pid;           /* process id of the interpreter                 */
  int status;          /* exit code from the interpreter                */
  UINT_8 flag;         /* flag for the message sent to the interpreter  */
  int basefd;          /* descriptor for the UNIX socket                */
  int interpfd;        /* descriptor once the interpreter has connected */
  char temp[256];      /* scratch area for constructing error messages  */
  AGENT_ID *local;     /* new identification for the arriving agent     */
  HASH_NODE *node;     /* an entry in the interpreter table             */
  char *name = NULL;   /* name of the UNIX socket                       */

    /* extract the root identification and language name */

  char *root_server = message -> elements[0].string;
  UINT_32 root_ip   = message -> elements[1].number;
  char *root_name   = message -> elements[2].string;
  UINT_32 root_id   = message -> elements[3].number;
  char *src_server  = message -> elements[4].string;
  UINT_32 src_ip    = message -> elements[5].number;
  char *src_name    = message -> elements[6].string;
  UINT_32 src_id    = message -> elements[7].number; 
  char *language    = message -> elements[8].string;
  AGENT_ID *root    = new AGENT_ID (root_server, root_ip, root_name, root_id);
  AGENT_ID *source  = new AGENT_ID (src_server, src_ip, src_name, src_id);

    /* find the language in the list of supported interpreters */

  if ((node = server_data -> interp_table -> lookup (0, language)) == NULL) {
    sprintf (temp, "no interpreter for language \"%s\"", language); 
    respond_with_error (sockfd, temp);
    return -1;
  }

  INTERP *interp = (INTERP *) node -> data;

    /* check that the langauge supports state images if we have a state image */

  if ((message -> flag == REQ_STATE) && (interp -> interp_type != STATE_INTERP)) {
    sprintf (temp, "interpreter for language \"%s\" does not accept state images", language);
    respond_with_error (sockfd, temp);
    return -1;
  }

    /* open a named socket */

  if ((name = tempnam (interp -> sock_dir, "intrp")) == NULL) {
    sprintf (temp, "unable to create a temporary file in \"%s\" (errno = %d)", interp -> sock_dir, errno);
    respond_with_error (sockfd, temp); 
    return -1;
  } 

  if ((basefd = tcpip_unix_setup (name)) < 0) {
    sprintf (temp, "unable to create Unix domain socket \"%s\" (errno = %d)", name, errno);
    respond_with_error (sockfd, temp);
    unlink (name);
    return -1;
  }

    /* fork the child that will then exec the interpreter */

  if ((pid = fork()) < 0) {       /* ERROR */

    sprintf (temp, "unable to fork child process (errno = %d)", errno);
    respond_with_error (sockfd, temp);
    close (basefd);
    unlink (name);
    return -1;

  } else if (pid == 0) {          /* CHILD */

    if (execl (interp -> executable, interp -> argv0, "-agent", name, (char *) 0) < 0) {
      exit (1);
    }
  }

    /* get a new agent identification */
 
  if ((local = register_with_agentd (sockfd, pid)) == NULL) {
    close (basefd);
    unlink (name);
    return -1;
  }
    
    /* wait for the interpreter to connect */

  flag = (message -> flag == REQ_STATE) ? INTERP_STATE : INTERP_SCRIPT;
  MESSAGE mesg_to_interp (to_interpreter.messages[flag]);
  mesg_to_interp.elements[0].string = root -> server;
  mesg_to_interp.elements[1].number = root -> ip;
  mesg_to_interp.elements[2].string = root -> name;
  mesg_to_interp.elements[3].number = root -> id;
  mesg_to_interp.elements[4].string = local -> server;
  mesg_to_interp.elements[5].number = local -> ip;
  mesg_to_interp.elements[6].number = local -> id;
  mesg_to_interp.elements[7].string = message -> elements[9].string;

  if ((interpfd = tcpip_accept (basefd)) < 0) {
    close (basefd);
    unlink (name);
    return -1;
  }

  if (message_send (interpfd, mesg_to_interp) < 0) {
    close (interpfd);
    close (basefd);
    unlink (name);
    return -1;
  }

  close (interpfd);
  close (basefd);
  unlink (name);
 
    /* wait for the interpreter to finish */

  waitpid (pid, &status, 0);

  if (WIFSIGNALED(status) || (WIFEXITED(status) && (WEXITSTATUS(status) != 0))) {
    send_result (local, root);
    return -1;
  }
  
  return 0;
}

/* tcl_send

   Purpose: This procedure handles the messages

            REQ_BEGIN
            REQ_NAME
            REQ_END
            REQ_MESSAGE
            REQ_MEETING
            REQ_EVENT

     Input: sockfd  = socket descriptor
            message = the message
 
    Output: The procedure returns -1 on error and 0 otherwise.
*/

int tcl_send (int sockfd, MESSAGE *message)
{
  int return_code = -1;
  MESSAGE *mesg_from_agentd = NULL;
  MESSAGE *mesg_to_client   = NULL;

    /* construct the pipe message */

  MESSAGE *mesg_to_agentd = map_message_to_pipe (message);

    /* send the message up the pipe */

  get_lock (server_data -> lockfd, server_data -> errorlog);

  if (message_send (server_data -> streamAgentd, *mesg_to_agentd) < 0) {
    mesg_to_client = new MESSAGE (down_socket.messages[RESP_ERROR]);
  } else if ((mesg_from_agentd = message_receive (server_data -> streamAgentd, down_pipe)) == NULL) {
    mesg_to_client = new MESSAGE (down_socket.messages[RESP_ERROR]);
  }

  release_lock (server_data -> lockfd, server_data -> errorlog);

  if (mesg_to_client == NULL) {
    mesg_to_client = map_message_to_socket (mesg_from_agentd);
    return_code    = 0;
  }

    /* send the response to the client */

  message_send (sockfd, *mesg_to_client);

    /* clean up */

  delete_check (mesg_from_agentd);
  delete mesg_to_agentd;
  delete mesg_to_client;
  close (sockfd);
  return return_code;
}

/* tcl_receive

   Purpose: This procedure handles the message

            REQ_GET

     Input: sockfd   = socket descriptor
            message  = the message
 
    Output: The procedure returns -1 and sends a RESP_ERROR message
            along the socket if an error occurs.  Otherwise the procedure
            loops forver and sends all incoming messages, meetings and
            events down the socket to the agent.
*/

int tcl_receive (int sockfd, MESSAGE *message)
{
  char buf;
  int code;
  fd_set readset; 
  struct timeval timeout;
  sigset_t newmask, oldmask;
  MESSAGE *mesg_from_agentd = NULL;
  MESSAGE *mesg_to_client   = NULL;

    /* break out the id */

  UINT_32 id = message -> elements[0].number;
 
    /* set up the pipe message */

  MESSAGE *mesg_to_agentd = map_message_to_pipe (message);
  mesg_to_agentd -> elements[1].number = (UINT_32) getpid();

    /* turn on asynchronous I/O for the socket */

  if (tcpip_asynchronous (sockfd) < 0) {
    server_data -> errorlog ->
      error_sys_quit
        ("tcl_receive: unable to turn on asychronous I/O for agent socket");
  }
 
    /* initialize the signal masks */

  sigemptyset (&oldmask);
  sigemptyset (&newmask);
  sigaddset   (&newmask, SIGUSR1);
  sigaddset   (&newmask, SIGIO);

    /* initiailize the "select" timeout */

  timeout.tv_sec  = 0;
  timeout.tv_usec = 0;

    /* block SIGUSR1 and SIGIO */
 
  if (sigprocmask (SIG_BLOCK, &newmask, &oldmask) < 0) {
    server_data -> errorlog ->
      error_sys_quit 
	("tcl_receive: unable to block SIGUSR1 and SIGIO");
  }

    /* iterate */

  while (1) {

      /* make sure the agent socket is still connected */

    FD_ZERO (&readset);
    FD_SET (sockfd, &readset);

    do {
      code = select (sockfd + 1, &readset, NULL, NULL, &timeout);
    } while ((code < 0) && (errno == EINTR));

    if (code > 0) {

        /* end the agent if the socket is not going down cleanly */

      if (read (sockfd, &buf, 1) <= 0) {
        remove_id (id);
      }

      break;
    }

      /* see if we have an available item */

    get_lock (server_data -> lockfd, server_data -> errorlog);

    if (message_send (server_data -> streamAgentd, *mesg_to_agentd) >= 0) {
      mesg_from_agentd = message_receive (server_data -> streamAgentd, down_pipe);
    }

    release_lock (server_data -> lockfd, server_data -> errorlog);

      /* break on server error */

    if ((mesg_from_agentd == NULL) || (mesg_from_agentd -> flag == RESP_ERROR)) {
      mesg_to_client = new MESSAGE (down_socket.messages[RESP_ERROR]); 
      message_send (sockfd, *mesg_to_client);
      delete mesg_to_client;
      break;
    }

      /* send the message down to the client OR block until signal arrives */

    if (mesg_from_agentd -> flag != RESP_NONE) {
    
      mesg_to_client = map_message_to_socket (mesg_from_agentd); 
      message_send (sockfd, *mesg_to_client);
      delete mesg_to_client;

    } else {

      if (sigsuspend (&oldmask) != -1) {
        server_data -> errorlog ->
          error_sys_quit 
  	    ("tcl_run: unable to pause and wait for SIGUSR1");
      }
    }

    delete mesg_from_agentd; 
  }

    /* unblock SIGUSR1 and SIGIO */

  if (sigprocmask (SIG_SETMASK, &oldmask, NULL) < 0) {
    server_data -> errorlog ->
      error_sys_quit 
	("tcl_run: unable to unblock SIGUSR1 and SIGIO");
  }

    /* clean up */

  delete_check (mesg_from_agentd);
  delete mesg_to_agentd;
  close (sockfd);
  return -1;
}

/* tcl_encrypted

   Purpose: This procedure handles the message

            REQ_ENCRYPTED

     Input: sockfd  = socket on which the message arrived
            message = the message  

    Output: The procedure returns -1 and sends a RESP_ERROR message along the
            socket if an error occurs.  Otherwise the procedure decrypts
            the embedded message, handles the message, sends the handler
            response down the socket and returns 0.
*/

int tcl_encrypted (int sockfd, MESSAGE *message)
{
  AGENT_ID *server;
  MESSAGE *response;

    /* for right now we just print the components */

  printf ("REQ_ENCRYPED\n");
  printf ("1. Server name ==> %s\n", message -> elements[0].string);
  printf ("2. Server IP   ==> %x\n", message -> elements[1].number);
  printf ("3. Public key  ==> %s\n", message -> elements[2].string);
  printf ("4. Filename    ==> %s\n", message -> elements[3].string);

  close (sockfd);

    /* and test out the server agents (this has nothing to do with   */
    /* encryption but was a very convenient place for the test code) */

  if ((server = get_id (0)) == NULL) {
    printf ("unable to an id for the handler\n");
    return -1;
  }

  if (send_message (server -> id, "tracker", 0, 0, "HELLO") < 0) {
    printf ("unable to send message to \"tracker\"\n");
    return -1;
  }

  if (((response = get_item (server -> id)) == NULL) || (response -> flag != RESP_MESSAGE)) {
    printf ("no response from tracker\n");
    return -1;
  }

  printf ("The \"tracker\" says %s\n", response -> elements[5].string);

  if (remove_id (server -> id) < 0) {
    printf ("unable to remove the handler id\n");
    return -1;
  }

  return 0;
}

/* tcl_sign

   Purpose: Sign a message with the server private key

     Input: sockfd  = the socket on which the message arrived
            message = the message

    Output: The procedure returns -1 on error.  Otherwise the procedure
            reads a binary stream off the socket, encrypts the stream,
            sends the encrypted stream back down the socket and returns 0.
*/

int tcl_sign (int sockfd, MESSAGE *message)
{
    /* for now just print a message and bail */

  printf ("****** REQ_SIGNED ******\n");
  close (sockfd);
  return 0;
} 
