/* Agent Tcl
   Bob Gray
   24 June 1995

   tclTcpip.cc

   This file implements the TCP/IP library routines.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifndef COSMO
#include <unistd.h>
#else
#include </usr/local/lib/gcc-lib/rs6000-ibm-aix3.2.5/2.6.3/include/unistd.h>
#endif
#include "fd_utilities.h"
#include "my_strings.h"
#include "tcl.h"
#include "tclRestrictInt.h"
#include "tclTcpip.h"
#include "tcpip.h"
#include "truefalse.h"

  /* size of copy buffer */

const BUFFER_SIZE = 1024;

/* Tcpip_Getport

   Purpose: Get the IP address and port number associated with either the
            local or remote end of a socket

     Input: interp = the current interpreter
            sockfd = the socket descriptor
            peer   = LOCAL_PEER or REMOTE_PEER

    Output: The procedure returns NULL and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure
            returns a pointer to a dynamically allocated REMOTE_PORT structure
            that contains the IP address and port number.
*/

REMOTE_PORT *Tcpip_Getport (Tcl_Interp *interp, int sockfd, int peer)
{
  UINT_32 ip;
  UINT_32 port;
  char *string;
  struct in_addr inaddr;

    /* get the port and IP address */

  if (peer == LOCAL_PEER) {
    if (tcpip_getsockname (sockfd, &ip, &port) < 0) {
      Tcl_AppendResult (interp, "unable to get local socket address: invalid descriptor", (char *) NULL);
      return NULL;
    }
  } else {
    if (tcpip_getpeername (sockfd, &ip, &port) < 0) {
      Tcl_AppendResult (interp, "unable to get remote socket address: invalid descriptor or not connected", (char *) NULL);
      return NULL;
    }
  }

    /* convert the IP address to a string */

  inaddr.s_addr = ip;
  string = inet_ntoa (inaddr);

    /* construct the REMOTE_PORT structure */

  REMOTE_PORT *port_address = new REMOTE_PORT (string, ip, port);
  return port_address;
}   
     
/* Tcpip_Getip

   Purpose: Get the IP number of a machine.

     Input: interp  = the current interpreter
            machine = name of the machine

    Output: The procedure returns -1 and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure
            returns 0 and sets *ip to the IP number.
*/

int Tcpip_Getip (Tcl_Interp *interp, char *machine, UINT_32 *ip)
{
  int return_code;

  if ((return_code = tcpip_getip (machine, ip)) < 0) {
    Tcl_AppendResult (interp, "unable to get IP address of \"", machine, "\"", (char *) NULL);
    return -1;
  }

  return 0;
}
      
/* Tcpip_Hostname

   Purpose: Get the name of the current host

     Input: interp = the current interpreter

    Output: The procedure sets the interpreter result to an error message
            and returns NULL on error.  Otherwise the procedure returns
            a pointer to a dynamically allocated string that contains the
            name of the current host.
*/

char *Tcpip_Hostname (Tcl_Interp *interp)
{
  char *name;  /* host name */

  if ((name = tcpip_hostname ()) == NULL)
    Tcl_AppendResult (interp, "unable to get local host name", (char *) NULL);

  return name;
}  

/* Tcpip_Socket

   Purpose: Create a socket

     Input: interp = the current interpreter

    Output: The procedure returns -1 and sets the interpreter result to an
            error message on error.  Otherwise the procedure returns the
            socket descriptor.
*/

int Tcpip_Socket (Tcl_Interp *interp)
{
  int sockfd;    /* socket descriptor */

  if ((sockfd = tcpip_socket ()) < 0)
  {
    Tcl_AppendResult (interp, tcpip_error_to_string(sockfd), (char *) NULL);
    return -1;
  }

  return sockfd;
}
   
/* Tcpip_Bind

   Purpose: Bind a socket to a port

     Input: interp = the current interpreter
            sockfd = the socket descriptor
            port   = the port number

    Output: The procedure sets the interpreter result to a error message
            and returns -1 on error.  Otherwise the procedure returns the
            number of the bound port.  The bound port will be the same
            as the requested port unless the requested port was TCPIP_ANY in
            which case the procedure lets the system choose a port. 
*/

int Tcpip_Bind (Tcl_Interp *interp, int sockfd, int port)
{
  int return_code;
  int selected_port;

  if ((return_code = tcpip_bind (sockfd, port, &selected_port)) < 0)
  {
    Tcl_AppendResult (interp, tcpip_error_to_string(return_code), (char *) NULL);
    return -1;
  }

  return selected_port;
}

/* Tcpip_Listen

   Purpose: Listen to a socket

     Input: interp = the current interpreter
            sockfd = socket descriptor

    Output: The procedure returns TCL_ERROR and sets the intepreter result
            to an error message on error.  Otherwise the procedure returns
            TCL_OK.
*/

int Tcpip_Listen (Tcl_Interp *interp, int sockfd)
{
  int return_code;

  if ((return_code = tcpip_listen (sockfd)) < 0)
  {
    Tcl_AppendResult (interp, tcpip_error_to_string(return_code), (char *) NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}    
   
/* Tcpip_Close

   Purpose: Close a socket

     Input: interp = the current interpreter
            sockfd = socket descriptor

    Output: The procedure closes the socket and returns TCL_OK.
*/

int Tcpip_Close (Tcl_Interp *interp, int sockfd)
{
   close (sockfd);
   return TCL_OK;
}

/* Tcpip_Accept 
  
   Purpose: Accept a connection on a socket

     Input: interp  = the current interpreter
            seconds = timeout interval
            sockfd  = socket descriptor
                      
            The procedure waits forever if seconds is less than 0.
            Otherwise the procedure waits for the specified number of
            seconds.  If there is no incoming connection at the end of that
            time, the procedure returns with a time out error.
   
    Output: The procedure returns -1 and sets the interpreter result to an 
            error message on error.  Otherwise the procedure returns the new 
            socket descriptor for the connection.
*/

int Tcpip_Accept (Tcl_Interp *interp, double seconds, int sockfd)
{
  UINT_32 ip;
  UINT_32 port;
  clock_t stop;
  int new_sockfd;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (sockfd >= 0);

    /* RESTRICT */

  RESTRICT *restrict = RESTRICT::get_restrict (interp);
  assert (restrict != NULL);

    /* make sure that the socket is unconnected */

  if (tcpip_getpeername (sockfd, &ip, &port) >= 0) {
    Tcl_AppendResult (interp, "socket is already connected", (char *) NULL);
    return -1;
  }

    /* add the restriction */

  if (seconds >= 0.0) {
    PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
    restrict -> add (&permit);
  } 

    /* do a select if we have a time limit */

  new_sockfd = 0;

  if (restrict -> first_wall (stop) >= 0) {
    if (fileWaitForRead (interp, sockfd, stop) != TCL_OK) {
      new_sockfd = -1;
    }
  }

    /* do the accept */

  if (new_sockfd >= 0) {
    if ((new_sockfd = tcpip_accept (sockfd)) < 0) {
      Tcl_AppendResult (interp, tcpip_error_to_string(new_sockfd), (char *) NULL);
    }
  }

    /* remove the restriction */

  if (seconds >= 0.0) {
    restrict -> remove ();
  }

  return (new_sockfd);
}

/* Tcpip_Connect

   Purpose: Connect to a socket on a remote machine

     Input: interp  = the current interpreter
            seconds = timeout interval 
            sockfd  = socket descriptor
            port    = name and port on remote machine 

    Output: The procedure returns TCL_ERROR and sets the interpreter result
            to an error message on error.  Otherwise the procedure returns
            TCL_OK.
*/

int Tcpip_Connect (Tcl_Interp *interp, double seconds, int sockfd, REMOTE_PORT *port)
{
  clock_t stop;
  int errorCode;
  int returnCode = TCL_OK;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (seconds >= 0.0);
  assert (sockfd >= 0);
  assert (port != NULL);

    /* RESTRICT */

  RESTRICT *restrict = RESTRICT::get_restrict (interp);
  assert (restrict != NULL);

    /* add the restriction */

  PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
  restrict -> add (&permit);
  restrict -> first_wall (stop);

    /* attempt to connect */

  if ((errorCode = tcpip_connect (sockfd,  port -> machine, port -> ip, port -> port, stop)) < 0) {

    Tcl_AppendResult (interp, tcpip_error_to_string (errorCode), (char *) NULL);

    if (errorCode == TCPIP_TIMEOUT) {
      Tcl_SetErrorCode (interp, "TIMEOUT", (char *) NULL);
    }

    returnCode = TCL_ERROR;
  }

    /* remove the restriction */

  restrict -> remove ();
  return (returnCode);
} 
   
/* Tcpip_WriteFromFile

   Purpose: Copy the contents of a file onto a socket

     Input: interp = the current interpreter
            sockfd = the socket descriptor
            fptr   = the file pointer

    Output: The procedure copies the file onto the socket.
*/

int Tcpip_WriteFromFile (Tcl_Interp *interp, int sockfd, FILE *fptr)
{
  register int nreq;
  register int nread;
  register int length;
  char buffer[BUFFER_SIZE];
  register int tread = 0;

    /* assertions on the parameters */

  assert (interp != NULL);
  assert (sockfd >= 0);
  assert (fptr != NULL);

    /* get the length of the file */

  fseek (fptr, 0, SEEK_END); 
  length = (int) ftell (fptr);
  fseek (fptr, 0, SEEK_SET);

  if (ferror (fptr)) {
    Tcl_AppendResult (interp, "error while reading file: ", Tcl_PosixError (interp), (char *) NULL);
    return TCL_ERROR;
  }

    /* write the length */

  if (tcpip_write_long (sockfd, length) < 0) {
    Tcl_AppendResult (interp, "error while writing socket: ", Tcl_PosixError (interp), (char *) NULL);
    return TCL_ERROR;
  }

    /* copy the file onto the socket */

  while (tread < length) {

    nreq = length - tread;
    nreq = (nreq < BUFFER_SIZE) ? nreq : BUFFER_SIZE;
  
       /* read the first buffer of data from the file */

    if ((nread = fread (buffer, sizeof(char), nreq, fptr)) < nreq) {
      Tcl_AppendResult (interp, "error while reading file: ", Tcl_PosixError (interp), (char *) NULL);
      return TCL_ERROR;
    }

       /* write out the buffer */

    if (tcpip_writen (sockfd, buffer, nread) < 0) {
      Tcl_AppendResult (interp, "error while writing socket: ", Tcl_PosixError (interp), (char *) NULL);
      return TCL_ERROR;
    }

    tread += nread;
  }

    /* everything okay */

  return TCL_OK;
}
   
/* Tcpip_Write

   Purpose: Write a string onto a socket

     Input: interp = the current interpreter
            sockfd = the socket descriptor
            string = the string

    Output: The procedure sets the interpreter results to an error message
            and returns TCL_ERROR on error.  Otherwise the procedure writes
            the string onto the socket and returns TCL_OK.
*/

int Tcpip_Write (Tcl_Interp *interp, int sockfd, char *string)
{
  int return_code;

  if ((return_code = tcpip_write_string (sockfd, string)) < 0) {
    Tcl_AppendResult (interp, "unable to write to socket", (char *) NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

/* Tcpip_ReadToFile

   Purpose: Copy a file off a socket

     Input: interp  = the current interpreter
            seconds = timeout interval
            sockfd  = socket descriptor
            fptr    = file pointer
 
    Output: The procedure copies the file off the socket.
*/

int Tcpip_ReadToFile (Tcl_Interp *interp, double seconds, int sockfd, FILE *fptr)
{
  int nleft;
  int nread;
  int nwrite;
  UINT_32 ip;
  UINT_32 port;
  clock_t stop;
  int requested;
  UINT_32 length;
  UINT_32 total = 0;
  int fileOkay = TRUE;
  int code = TCL_ERROR;
  char buffer[BUFFER_SIZE];
 
    /* assertions on the parameters */

  assert (interp != NULL);
  assert (sockfd >= 0);
  assert (fptr != NULL);

    /* RESTRICT */

  RESTRICT *restrict = RESTRICT::get_restrict (interp);
  assert (restrict != NULL);

    /* make sure that the socket is connected */

  if (tcpip_getpeername (sockfd, &ip, &port) < 0) {
    Tcl_AppendResult (interp, "socket is not connected", (char *) NULL);
    return TCL_ERROR;
  }

    /* add the restriction */

  if (seconds >= 0.0) {
    PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
    restrict -> add (&permit);
  } 

    /* do a select if we have a time limit */

  if (restrict -> first_wall (stop) >= 0) {
    if (fileWaitForRead (interp, sockfd, stop) != TCL_OK) {
      goto done;
    }
  } 

    /* read the file length */ 

  if (tcpip_read_long (sockfd, &length) < 0) {
    Tcl_AppendResult (interp, "unable to read from socket", (char *) NULL);
    goto done;
  }

    /* copy off the file */

  while (total < length) {

      /* get the next buffer of data */

    nleft     = length - total;
    requested = (nleft < BUFFER_SIZE) ? nleft : BUFFER_SIZE;

    if ((nread = tcpip_readn (sockfd, buffer, requested)) < requested) {
      Tcl_SetResult (interp, "broken socket", TCL_STATIC);
      goto done;
    }

       /* write onto the file */

    if (fileOkay) {
      if ((nwrite = fwrite (buffer, sizeof(char), nread, fptr)) < nread) {
        Tcl_AppendResult (interp, "error writing file: ", Tcl_PosixError (interp), (char *) NULL);
        fileOkay = FALSE; 
      }
    }

    total += nread;
  }

  code = fileOkay ? TCL_OK : TCL_ERROR;

done:

  fflush (fptr);
 
  if (seconds >= 0.0) {
    restrict -> remove ();
  }

  return (code);
}

/* Tcpip_Read

   Purpose: Read a string from a socket

     Input: interp  = the current interpreter
            seconds = timeout interval     
            sockfd  = the socket descriptor
                      
            The procedure waits forever if seconds is less than 0.
            Otherwise the procedure waits for the specified number of
            seconds.  If there is no incoming connection at the end of that
            time, the procedure returns with a time out error.
   
    Output: The procedure sets the interpreter result to an error message
            and returns NULL on error.  Otherwise the procedure returns a
            pointer to a dynamically allocated character array that contains
            the string.
*/

char *Tcpip_Read (Tcl_Interp *interp, double seconds, int sockfd)
{
  UINT_32 ip;
  UINT_32 port;
  clock_t stop;
  int code = TCL_OK;
  char *string = NULL;
  
    /* assertions on the parameters */

  assert (interp != NULL);
  assert (sockfd >= 0);

    /* RESTRICT */

  RESTRICT *restrict = RESTRICT::get_restrict (interp);
  assert (restrict != NULL);

    /* make sure that the socket is connected */

  if (tcpip_getpeername (sockfd, &ip, &port) < 0) {
    Tcl_AppendResult (interp, "socket is not connected", (char *) NULL);
    return NULL;
  }

    /* add the restriction */

  if (seconds >= 0.0) {
    PERMIT_EXT permit (PERMIT_WALL, seconds, 0.0);
    restrict -> add (&permit);
  } 

    /* do a select if we have a time limit */

  if (restrict -> first_wall (stop) >= 0) {
    code = fileWaitForRead (interp, sockfd, stop);
  } 

    /* read a string from the socket */

  if (code != TCL_ERROR) {
    if ((string = tcpip_read_string (sockfd)) == NULL) {
      Tcl_AppendResult (interp, "unable to read from socket", (char *) NULL);
    }
  }

    /* remove the restriction */

  if (seconds >= 0.0) {
    restrict -> remove ();
  }

  return string;
}

/* Tcpip_SplitPort

   Purpose: Split apart a string that contains a port identification

     Input: interp = Tcl interpreter
            string = the string

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns a pointer to a dynamically allocated REMOTE_PORT
            structure that contains the port identification.
*/

REMOTE_PORT *Tcpip_SplitPort (Tcl_Interp *interp, char *string)
{
  REMOTE_PORT *port = NULL;  /* port information                   */
  int listArgc;              /* number of components in the string */
  char **listArgv;           /* components of the string           */
  UINT_32 ip;                /* IP address                         */
  int port_number;           /* port number                        */

    /* must have a valid Tcl list */

  if (Tcl_SplitList (interp, string, &listArgc, &listArgv) != TCL_OK) {
    Tcl_AppendResult (interp, "port identification must be a valid Tcl list", (char *) NULL);
    return NULL;
  }
 
    /* two arguments means <machine port> or <IP port> */
    /* three arguments means <machine IP port>         */
    /* more means an invalid specification             */

  if (listArgc == 2) {

    if (Tcpip_StringToIp (interp, listArgv[0], &ip) < 0) {
      Tcl_ResetResult (interp);
      ip = UNKNOWN_IP;
    }
  
    if ((Tcl_GetInt (interp, listArgv[1], &port_number) != TCL_OK) || (port_number < 0)) {
      Tcl_ResetResult (interp);
      Tcl_AppendResult (interp, "invalid port number \"", listArgv[1], "\": must be an integer 0 or greater", (char *) NULL);
    } else {
      port = new REMOTE_PORT (listArgv[0], ip, port_number);
    }

  } else if (listArgc == 3) {

    if (Tcpip_StringToIp (interp, listArgv[1], &ip) < 0) {
      Tcl_ResetResult (interp);
      Tcl_AppendResult (interp, "invalid IP address \"", listArgv[1], (char *) NULL);
    } else if ((Tcl_GetInt (interp, listArgv[2], &port_number) != TCL_OK) || (port_number < 0)) {
      Tcl_ResetResult (interp);
      Tcl_AppendResult (interp, "invalid port number \"", listArgv[1], "\": must be an integer 0 or greater", (char *) NULL);
    } else {
      port = new REMOTE_PORT (listArgv[0], ip, port_number);
    }

  } else {

    Tcl_AppendResult (interp, "must be {IP port}, {machine port} or {machine IP port}", (char *) NULL);

  }

  delete listArgv;
  return (port);
}

/* Tcpip_StringToIp

   Purpose: Convert a dotted decimal IP string into an IP number

     Input: interp = the current interpreter
            string = the string

    Output: The procedure returns -1 and sets the intepreter result to an
            appropriate error message on error.  Otherwise the procedure
            returns 0 and fills in the "ip" parameter.
*/

int Tcpip_StringToIp (Tcl_Interp *interp, char *string, UINT_32 *ip)
{
   if ((*ip = (UINT_32) inet_addr(string)) == INADDR_NONE)
   {
     Tcl_AppendResult (interp, "invalid IP address", (char *) NULL);
     return -1;
   }

   return 0;
}

/* Tcpip_IpToString

   Purpose: Convert an IP number into a string

     Input: interp = the current interpreter
            ip     = the IP number

    Output: The procedure returns a pointer to a dyanmically allocated
            string that contains the IP number in dotted decimal form.
*/

char *Tcpip_IpToString (Tcl_Interp *interp, UINT_32 ip)
{
  struct in_addr inaddr;
  inaddr.s_addr = ip;
  return (my_strcpy (inet_ntoa (inaddr)));
}

/* Tcpip_PortToString

   Purpose: Convert a port identification into a string

     Input: interp = Tcl interpreter
            port   = port identification

    Output: The procedure returns NULL on error or a pointer to a dynamically
            allocated Tcl_DString.
*/

Tcl_DString *Tcpip_PortToString (Tcl_Interp *interp, REMOTE_PORT *port)
{
  char temp[16];
  Tcl_DString *string = new Tcl_DString;
  Tcl_DStringInit (string);

  Tcl_DStringAppendElement (string, port -> machine);

  if (port -> ip == UNKNOWN_IP)
  {
    Tcl_DStringAppendElement (string, "");
  }
  else
  { 
    char *ip_string = Tcpip_IpToString (interp, port -> ip);
    Tcl_DStringAppendElement (string, ip_string);
    delete ip_string;
  }
 
  sprintf (temp, "%d", port -> port);
  Tcl_DStringAppendElement (string, temp);
  return string;  
}
