/* Agent Tcl
   Bob Gray
   24 June 1995

   tclTcpipCmd.cc

   This file implements the TCP/IP commands and the Tcpip_Init procedure.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "my_sizes.h"
#include "cmd_utilities.h"
#include "tcl.h"
#include "tcpip.h"
#include "tclTcpip.h"
#include "tcl_utilities.h"
#include "truefalse.h"

  /* general */

int Tcpip_BindCmd     (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
int Tcpip_ListenCmd   (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
int Tcpip_HostnameCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
int Tcpip_CloseCmd    (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
int Tcpip_GetipCmd    (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
int Tcpip_GetportCmd  (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
int Tcpip_AcceptCmd   (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
int Tcpip_ReadCmd     (ClientData dummy, Tcl_Interp *interp, int argc, char **argv); 
int Tcpip_WriteCmd    (ClientData dummy, Tcl_Interp *interp, int argc, char **argv); 

  /* agent protocol */

int Tcpip_SocketCmd   (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
int Tcpip_ConnectCmd  (ClientData raw, Tcl_Interp *interp, int argc, char **argv);
int Tcpip_AcceptCmd   (ClientData raw, Tcl_Interp *interp, int argc, char **argv);

  /* raw protocl */

int Raw_ConnectCmd    (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
int Raw_AcceptCmd     (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);

  /* agent protocol (Unix domain) */

int Unix_SocketCmd    (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
int Unix_ConnectCmd   (ClientData raw, Tcl_Interp *interp, int argc, char **argv);
int Unix_BindCmd      (ClientData dummy, Tcl_Interp *interp, int argc, char **argv);

  /* command info */

struct COMMAND_INFO
{
  char *name;
  Tcl_CmdProc *proc;  
};

COMMAND_INFO tcpip_commands[] = 
{
  {"tcpip_socket"       , Tcpip_SocketCmd       },
  {"tcpip_bind"         , Tcpip_BindCmd         },
  {"tcpip_listen"       , Tcpip_ListenCmd       },
  {"tcpip_close"        , Tcpip_CloseCmd        },
  {"tcpip_hostname"     , Tcpip_HostnameCmd     },
  {"tcpip_getip"        , Tcpip_GetipCmd        },
  {"tcpip_getport"      , Tcpip_GetportCmd      },
  {"tcpip_read"         , Tcpip_ReadCmd         },
  {"tcpip_write"        , Tcpip_WriteCmd        },
  {"tcpip_connect"      , Tcpip_ConnectCmd      },
  {"tcpip_accept"       , Tcpip_AcceptCmd       },
  {"raw_connect"	, Raw_ConnectCmd        },
  {"raw_accept"		, Raw_AcceptCmd         },
  {"unix_connect"	, Unix_ConnectCmd       },
  {"unix_bind"		, Unix_BindCmd          },
  {"unix_socket"	, Unix_SocketCmd        },
  { NULL 		, (Tcl_CmdProc *) NULL  }
};

int Unix_SocketCmd (ClientData clientData,Tcl_Interp *interp,int argc,char ** argv )
{
  int sockfd;
  char temp[16];

    /* check the number of arguments */

  if (argc != 1) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the socket */

  sockfd = tcpip_unix_socket ();

  if (sockfd < 0) {
    Tcl_AppendResult (interp, "unable to create socket: ", Tcl_PosixError (interp), (char *) NULL);
    return TCL_ERROR;
  }

    /* return the socket descriptor */

  sprintf(temp,"%d",sockfd);
  Tcl_SetResult(interp,temp,TCL_VOLATILE);
  return TCL_OK;
}

int Unix_ConnectCmd (ClientData clientData,Tcl_Interp *interp,int argc,char ** argv )
{ 
  int result, sockfd;

    /* check the number of arguments */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " <sockfd> <filename>\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the socket descriptor */

   result = Tcl_GetInt(interp,argv[1],&sockfd);

   if (result != TCL_OK) {
     return TCL_ERROR;
   }

     /* connect the socket */

   result = tcpip_unix_connect(sockfd,argv[2]);

   if (result < 0) {
     Tcl_AppendResult (interp, "unable to connect socket: ", Tcl_PosixError (interp), (char *) NULL);
     return TCL_ERROR;
   }

     /* everything okay */

   return TCL_OK;
}

int Unix_BindCmd (ClientData clientData,Tcl_Interp *interp,int argc,char ** argv )
{
  int sockfd,result;

    /* check the number of arguments */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " <sockfd> <filename>\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the socket descriptor */

  result = Tcl_GetInt(interp,argv[1],&sockfd);

  if (result != TCL_OK) {
    return TCL_ERROR;
  }

    /* bind the socket */

  result = tcpip_unix_bind(sockfd,argv[2]);

  if (result < 0) {
    Tcl_AppendResult (interp, "unable to bind socket: ", Tcl_PosixError (interp), (char *) NULL);
    return TCL_ERROR;
  }

    /* everything okay */

  Tcl_SetResult(interp, argv[2], TCL_VOLATILE);
  return TCL_OK;
}

/* Tcpip_GetportCmd

   Tcl syntax: tcpip_getport sockfd [-peer]
    
      Purpose: Get the local or remote IP address and port number of a
               connected socket

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments 
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
               to a {machine IP port} triple.
*/

int Tcpip_GetportCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int sockfd;
  int peer = LOCAL_PEER;
  REMOTE_PORT *port;
  Tcl_DString *port_string;
  
    /* check the number of arguments */
 
  if (argc == 3)
  {
    if (!strcmp (argv[2], "-peer"))
    {
      peer = REMOTE_PEER;
    }
    else
    {
      Tcl_AppendResult (interp, "second argument must be \"-peer\" if specified", (char *) NULL);
      return TCL_ERROR;
    }
  }
  else if (argc != 2)
  {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], "\" sockfd [-peer]", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the socket descriptor */

  if ((Tcl_GetInt (interp, argv[1], &sockfd) != TCL_OK) || (sockfd < 0))
  {
    Tcl_AppendResult (interp, "first argument must be a sockfd", (char *) NULL);
    return TCL_ERROR;
  }
  
    /* get the port information */

  if ((port = Tcpip_Getport (interp, sockfd, peer)) == NULL)
  {
    return TCL_ERROR;
  }

    /* convert the port information to a string */

  if ((port_string = Tcpip_PortToString (interp, port)) == NULL)
  {
    delete port;
    return TCL_ERROR;
  }

  Tcl_SetResult (interp, Tcl_DStringValue (port_string), TCL_VOLATILE);
  tclFreeDString (port_string);
  delete port;
  return TCL_OK; 
}

/* Tcpip_GetipCmd

   Tcl syntax: tcpip_getip hostname

      Purpose: Get the IP number of a host

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
               to the IP number in dotted decimal form.
*/

int Tcpip_GetipCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  UINT_32 ip;
  char *string;

   /* check the number of arguments */

  if (argc != 2)
  {
    Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " machine\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the IP number */

  if (Tcpip_Getip (interp, argv[1], &ip) < 0)
    return TCL_ERROR;

    /* set the intepreter result */

  if ((string = Tcpip_IpToString (interp, ip)) == NULL) 
    return TCL_ERROR;

  Tcl_SetResult (interp, string, TCL_DYNAMIC);
  return TCL_OK;
}  

/* Tcpip_HostnameCmd

   Tcl syntax: tcpip_hostname

      Purpose: Get the official name of the current host

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               results to an error message on error.  Otherwise the procedure
               returns TCL_OK and sets the interpreter result to the name
               of the current host.
*/

int Tcpip_HostnameCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv) 
{
  char *name;

    /* check the number of arguments */

  if (argc != 1)
  {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the hostname */

  if ((name = Tcpip_Hostname (interp)) == NULL)
    return TCL_ERROR;

  Tcl_SetResult (interp, name, TCL_DYNAMIC);
  return TCL_OK;  
}

/* Tcpip_SocketCmd

   Tcl syntax: tcpip_socket

      Purpose: Create a socket

        Input: dummy  = client data (unused)
               interp = the current interpreter 
               argc   = number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns the socket descriptor.
*/

int Tcpip_SocketCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  char temp[16];       /* scratch area      */
  int sockfd;          /* socket descriptor */

    /* check the number of arguments */

  if (argc != 1)
  {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* create the socket */

  if ((sockfd = Tcpip_Socket (interp)) < 0)
    return TCL_ERROR;

    /* put the socket descriptor into the interpreter result */

  sprintf (temp, "%d", sockfd);
  Tcl_SetResult (interp, temp, TCL_VOLATILE);
  return TCL_OK;
}
 
/* Tcpip_BindCmd

   Tcl syntax: tcpip_bind sockfd port

               The system will select an usued port if the given port is 0
               or the string "ANY".
 
      Purpose: Bind a socket to a port number.

        Input: dummy  = client data (unused)
               interp = the current interpreter 
               argc   = number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure returns TCL_OK and sets the interpreter result
               to the port number.  This will be same as the given port unless
               the given port was 0 or ANY.
*/
     
int Tcpip_BindCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  char temp[16];       /* scratch area          */
  int sockfd;          /* socket descriptor     */
  int port;            /* requested port number */
  int actual_port;     /* actual port number    */

    /* check the number of arguments */

  if (argc != 3)
  {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " sockfd port\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the port number */

  if (!strcmp(argv[2], "ANY"))
  {
    port = TCPIP_ANY_PORT;
  }
  else if ((Tcl_GetInt (interp, argv[2], &port) != TCL_OK) || (port < 0))
  {
    Tcl_AppendResult (interp, "port must be a nonnegative integer", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the socket descriptor */

  if ((Tcl_GetInt (interp, argv[1], &sockfd) != TCL_OK) || (sockfd < 0))
  {
    Tcl_AppendResult (interp, "sockfd must be a nonnegative integer", (char *) NULL);
    return TCL_ERROR;
  }
 
    /* bind to the port */

  if ((actual_port = Tcpip_Bind (interp, sockfd, port)) < 0)
    return TCL_ERROR;

    /* put the actual port number into the result */

  sprintf (temp, "%d", actual_port);
  Tcl_SetResult (interp, temp, TCL_VOLATILE);
  return TCL_OK;
}

/* Tcpip_ListenCmd

   Tcl syntax: tcpip_listen sockfd

      Purpose: Listen to a socket

        Input: dummy  = client data (unused)
               interp = the current interpreter 
               argc   = number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an appropriate error message on error.  Otherwise
               the procedure sets the interpreter result to the empty string
               and returns TCL_OK.
*/
     
int Tcpip_ListenCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int sockfd;          /* socket descriptor     */

    /* check the number of arguments */

  if (argc != 2)
  {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " sockfd\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the socket descriptor */

  if ((Tcl_GetInt (interp, argv[1], &sockfd) != TCL_OK) || (sockfd < 0))
  {
    Tcl_AppendResult (interp, "sockfd must be a nonnegative integer", (char *) NULL);
    return TCL_ERROR;
  }

    /* open the port */

  if (Tcpip_Listen (interp, sockfd) != TCL_OK)
    return TCL_ERROR;

    /* empty out the result */

  Tcl_ResetResult (interp);
  return TCL_OK;
}

/* Tcpip_ConnectCmd and Raw_ConnectCmd

   Tcl syntax: tcpip_connect sockfd {machine port} [-time seconds]
               raw_connect   sockfd {machine port} [-time seconds]

      Purpose: Connect to a socket to a remote port

        Input: raw    = non-NULL if we want raw mode
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter 
               result to an error message on error.  Otherwise the procedure
	       sets the interpreter result to the empty string and returns
               TCL_OK.
*/

int Raw_ConnectCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  return (Tcpip_ConnectCmd ((ClientData) 0x1, interp, argc, argv));
}

int Tcpip_ConnectCmd (ClientData raw, Tcl_Interp *interp, int argc, char **argv)
{
  FILE *fp;
  int sockfd;
  char temp[32];
  double seconds;
  int returnCode;
  REMOTE_PORT *port;

    /* get the timeout if it is specified */

  if (argc == 3) {
    seconds = DEFAULT_TCPIP_SECONDS;
  } else if (argc != 5) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " sockfd {machine port} [-time seconds]\"", (char *) NULL);
    return TCL_ERROR;
  } else if (parseShortTimeout (interp, seconds, argc, argv, 3) < 0) {
    return TCL_ERROR;
  }

    /* get the socket descriptor */

  if ((Tcl_GetInt (interp, argv[1], &sockfd) != TCL_OK) || (sockfd < 0)) { 
    Tcl_SetResult (interp, "sockfd must be 0 or greater", TCL_STATIC);
    return TCL_ERROR;
  }

    /* split out the machine name and port number */

  if ((port = Tcpip_SplitPort(interp, argv[2])) == NULL) {
    return TCL_ERROR;
  }

    /* connect to the remote port */
  
  returnCode = Tcpip_Connect (interp, seconds, sockfd, port);
  delete port;

    /* set up the new file descriptor if we have done a raw connect */

  if ((raw) && (returnCode == TCL_OK)) {
    fp = fdopen (sockfd, "r+");
    setbuf (fp, (char *) NULL);
    Tcl_EnterFile (interp, fp, TCL_FILE_READABLE | TCL_FILE_WRITABLE);
    sprintf (temp, "file%d", sockfd);
    Tcl_SetResult (interp, temp, TCL_VOLATILE);
  }

  return (returnCode);
}

/* Tcpip_AcceptCmd

   Tcl_Syntax: tcpip_accept sockfd {-blocking | -time seconds | -nonblocking}
 
      Purpose: Accept a connection on a socket

        Input: raw    = non-NULL if we want raw mode
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an error message on error.  Otherwise the procedure sets
               the interpreter result to the socket descriptor for the new
               connection and returns TCL_OK.
*/

int Raw_AcceptCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  return (Tcpip_AcceptCmd ((ClientData) 0x1, interp, argc, argv));
}

int Tcpip_AcceptCmd (ClientData raw, Tcl_Interp *interp, int argc, char **argv)
{
  int i;
  FILE *fp;
  int sockfd;    
  char temp[16];  
  int new_sockfd; 
  double seconds;

    /* check the number of arguments */

  if ((argc < 3) || (argc > 4)) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " sockfd {-blocking | -time seconds | -nonblocking}\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the time out information  */

  if ((i = parseTimeout (interp, seconds, argc, argv, 2)) < 0) {
    return TCL_ERROR;
  } else if (i < argc) {
    Tcl_AppendResult (interp, "extraneous arguments", (char *) NULL);
    return TCL_ERROR;
  }      

    /* get the socket descriptor */

  if ((Tcl_GetInt (interp, argv[1], &sockfd) != TCL_OK) || (sockfd < 0)) {
    Tcl_SetResult (interp, "first argument must be a sockfd", TCL_STATIC);
    return TCL_ERROR;
  }

    /* accept a connection */

  if ((new_sockfd = Tcpip_Accept (interp, seconds, sockfd)) < 0) { 
    return TCL_ERROR;
  }

    /* add to the Tcl tables if this is a raw connection */

  if (raw) {
    fp = fdopen (new_sockfd, "r+");
    setbuf (fp, (char *) NULL);
    Tcl_EnterFile (interp, fp, TCL_FILE_READABLE | TCL_FILE_WRITABLE);
    sprintf (temp, "file%d", new_sockfd);
  } else {
    sprintf (temp, "%d", new_sockfd);
  }

    /* return the new sockfd */

  Tcl_SetResult (interp, temp, TCL_VOLATILE);
  return TCL_OK;
}

/* Tcpip_ReadCmd

   Tcl syntax: tcpip_read sockfd [ to <fd> ] {-blocking | -time seconds | -nonblocking}

      Purpose: Read a string from a socket

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to an appropriate error message on error.  Otherwise the
               procecedure returns TCL_OK and sets the interpreter result to
               the string read from the socket.
*/
     
int Tcpip_ReadCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{ 
  int i;
  int sockfd;     
  FILE *fptr;
  char *string;  
  int makeFile;
  double seconds;

    /* check the number of arguments */

  if (argc < 3) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " sockfd [ to <fd> ] {-blocking | -time seconds | -nonblocking}\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the socket descriptor */

  if ((Tcl_GetInt (interp, argv[1], &sockfd) != TCL_OK) || (sockfd < 0)) {
    Tcl_ResetResult (interp);
    Tcl_AppendResult (interp, "invalid sockfd \"", argv[1], "\": must be an integer 0 or greater", (char *) NULL);
    return TCL_ERROR;
  }

    /* see if we are reading a file or string */

  if (argc >= 5) {   /* FILE */

      /* index of the timout part */

    i        = 4;
    makeFile = TRUE;
  
      /* check for the "to" parameter */

    if (strcmp (argv[2], "to")) {
      Tcl_AppendResult (interp, "invalid option \"", argv[2], "\": expected \"to\"", (char *) NULL);
      return TCL_ERROR;
    } 

      /* get the file descriptor */

    if (Tcl_GetOpenFile (interp, argv[3], TRUE, TRUE, &fptr) != TCL_OK) {
      return TCL_ERROR;
    }

  } else {   /* STRING */

      /* index of the timeout part */

    i        = 2;
    makeFile = FALSE;
  } 
      
    /* get the time out information  */

  if ((i = parseTimeout (interp, seconds, argc, argv, i)) < 0) {
    return TCL_ERROR;
  } else if (i < argc) {
    Tcl_AppendResult (interp, "extraneous arguments", (char *) NULL);
    return TCL_ERROR;
  }

    /* read into a file if a file was specified */

  if (makeFile) {
    return (Tcpip_ReadToFile (interp, seconds, sockfd, fptr)); 
  } 

    /* otherwise read into a string */

  if ((string = Tcpip_Read (interp, seconds, sockfd)) == NULL) {
    return TCL_ERROR;
  } else {
    Tcl_SetResult (interp, string, TCL_DYNAMIC);
    return TCL_OK;
  }
}

/* Tcpip_WriteCmd 

   Tcl syntax: tcpip_write <sockfd> [ <string> | from <fd> ]

      Purpose: Write a string onto a socket

        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = number of command arguments
               argv   = the command arguments

       Output: The procedure returns TCL_ERROR and sets the interpreter
               result to an error message on error.  Otherwise the procedure
               returns TCL_OK and writes the string onto the socket. 
*/     

int Tcpip_WriteCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  FILE *fptr;
  int sockfd;           

    /* check the number of arguments */

  if (argc < 3) {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " <sockfd> [ <string> | from <fd> ]\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the socket descriptor */

  if ((Tcl_GetInt (interp, argv[1], &sockfd) != TCL_OK) || (sockfd < 0)) {
    Tcl_ResetResult (interp);
    Tcl_AppendResult (interp, "invalid socket descriptor \"", argv[1], "\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* write the string if we are writing a string */

  if (argc == 3) {
    return (Tcpip_Write (interp, sockfd, argv[2]));
  }

    /* otherwise we are writing from a file */

  if (argc != 4) {
    Tcl_AppendResult (interp, "extraneous arguments", (char *) NULL);
    return TCL_ERROR;
  } else if (strcmp(argv[2], "from")) {
    Tcl_AppendResult (interp, "invalid option \"", argv[2], "\": expected \"from\"", (char *) NULL);
    return TCL_ERROR;
  }

    /* get the file pointer */

  if (Tcl_GetOpenFile (interp, argv[3], FALSE, TRUE, &fptr) != TCL_OK) {
    return TCL_ERROR;
  }

    /* write from the file */

  return (Tcpip_WriteFromFile (interp, sockfd, fptr));
}

/* Tcpip_Close

   Tcl syntax: tcpip_close sockfd

      Purpose: Close a socket
  
        Input: dummy  = client data (unused)
               interp = the current interpreter
               argc   = the number of command arguments
               argv   = the command arguments  

       Output: The procedure returns TCL_ERROR and sets the interpreter result
               to error message on error.  Otherwise the procedure returns
               TCL_OK.
*/

int Tcpip_CloseCmd (ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
  int sockfd;   /* socket desriptor */

    /* check the number of arguments */

  if (argc != 2)
  {
    Tcl_AppendResult (interp, "wrong # of arguments: should be \"", argv[0], " sockfd\"", (char *) NULL);
    return TCL_ERROR; 
  }

    /* get the socket descriptor */

  if ((Tcl_GetInt (interp, argv[1], &sockfd) != TCL_OK) || (sockfd < 0))
  {
    Tcl_AppendResult (interp, "socket fd must be a nonnegative integer", (char *) NULL);
    return TCL_ERROR; 
  }

    /* close the socket */

  return Tcpip_Close (interp, sockfd);
}
  
/* Tcpip_Init

   Purpose: Initialize the TCP/IP  TCL package

     Input: interp = the current interpreter

    Output: The procedure returns TCL_ERROR on initialization failure.
            The procedure returns TCL_OK    on initialization success.
*/
             
int Tcpip_Init (Tcl_Interp *interp)
{
  COMMAND_INFO *ptr;

  for (ptr = tcpip_commands; ptr -> name != NULL; ptr++)
    Tcl_CreateCommand 
      (interp, ptr -> name, ptr -> proc, 
	(ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  return TCL_OK;
}

