/* Agent Tcl
   Bob Gray
   25 January 1995

   agentId.cc

   This file implements the structures REMOTE_PORT and AGENT_ID.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "agentId.h"
#include "my_alloc.h"
#include "my_sizes.h"
#include "my_strings.h"

/* AGENT_INFO::AGENT_INFO and AGENT_INFO::~AGENT_INFO

   Purpose: These procedures are the constructor and destructor for class
            AGENT_INFO.
*/

AGENT_INFO::AGENT_INFO (void)
{
  id         = NULL;
  actual     = NULL;
  agentPid   = 0;
  blockerPid = 0;
  messages   = 0;
  meetings   = 0;
  events     = 0;
}

AGENT_INFO::~AGENT_INFO ()
{
  delete_check (id);
  delete_check (actual);
}

/* NAME_LIST::fill

   Purpose: Fill in a name

     Input: i    = index
            name = the name

    Output: The procedure fills in the name.
*/

void NAME_LIST::fill (int i, char *name)
{
    /* assertions on the parameters */

  assert ((i >= 0) && (i < count));

    /* delete the old name */

  delete_check (names[i]);

    /* fill in the new name */

  names[i] = my_strcpy (name);

  if (name != NULL) {
    lengths[i] = strlen(name) + 1;
  } else {
    lengths[i] = 0;
  }
}

/* NAME_LIST::NAME_LIST

   Purpose: This procedure is the constructor for class NAME_LIST.

     Input: count = number of names in the list
*/

NAME_LIST::NAME_LIST (int count)
{
  NAME_LIST::count   = count;
  NAME_LIST::names   = NULL;
  NAME_LIST::lengths = NULL;

  if (count != 0) {

    names   = new char * [count];
    lengths = new UINT_32 [count];
  
    for (int i = 0; i < count; i++) {
      names  [i] = NULL;
      lengths[i] = 0;
    }
  } 
}

/* NAME_LIST::~NAME_LIST

   Purpose: This procedure is the destructor for class NAME_LIST.
*/

NAME_LIST::~NAME_LIST ()
{
  for (int i = 0; i < count; i++) {
    delete_check (names[i]);
  }

  delete_check (lengths);
  delete_check (names);
} 
  
/* ID_LIST::ID_LIST

   Purpose: This procedure is the constructor for class ID_LIST.

     Input: count = number of ids in the list
*/

ID_LIST::ID_LIST (int count)
{
  ID_LIST::count = count;

  if (count != 0) {
    ids = new UINT_32 [count];
  } else {
    ids = NULL;
  }
}

/* ID_LIST::~ID_LIST

   Purpose: This procedure is the denstructor for class ID_LIST.
*/

ID_LIST::~ID_LIST ()
{
  delete_check (ids);
}

/* MACHINE_ID::MACHINE_ID and MACHINE_ID::~MACHINE_ID
 
   Purpose: These procedures are the constructor and destructor for class
            MACHINE_ID.
*/

MACHINE_ID::MACHINE_ID (char *server, UINT_32 server_ip) 
{
  MACHINE_ID::server    = my_strcpy (server);
  MACHINE_ID::server_ip = server_ip;
}

MACHINE_ID::MACHINE_ID (MACHINE_ID &id)
{
  MACHINE_ID::server    = my_strcpy (id.server);
  MACHINE_ID::server_ip = id.server_ip;
}

MACHINE_ID::~MACHINE_ID ()
{
  delete_check (server);
}
  
/* AGENT_ID::change_name

   Purpose: Change the name of an agent

     Input: new_name = new name of the agent

    Output: The procedure replaces the new name of the agent with the old name.
*/

void AGENT_ID::change_name (char *new_name)
{
  delete_check (name);
  name = my_strcpy (new_name);
}

/* AGENT_ID::AGENT_ID and AGENT_ID::~AGENT_ID

   Purpose: These procedures are the constructor and destructor for class
            AGENT_ID.
*/

AGENT_ID::AGENT_ID (const char *server, UINT_32 ip, const char *name, INT_32 id)
{
  AGENT_ID::ip     = ip;
  AGENT_ID::id     = id;
  AGENT_ID::server = my_strcpy (server);
  AGENT_ID::name   = my_strcpy (name);
}

AGENT_ID::AGENT_ID (const AGENT_ID &id)
{
  AGENT_ID::ip     = id.ip;
  AGENT_ID::id     = id.id;
  AGENT_ID::server = my_strcpy (id.server);
  AGENT_ID::name   = my_strcpy (id.name);
}

AGENT_ID::AGENT_ID (AGENT_ID *id)
{
  AGENT_ID::ip     = id -> ip;
  AGENT_ID::id     = id -> id;
  AGENT_ID::server = my_strcpy (id -> server);
  AGENT_ID::name   = my_strcpy (id -> name);
}

AGENT_ID::AGENT_ID (void)
{
  AGENT_ID::server = ANY_SERVER;
  AGENT_ID::ip     = ANY_IP;
  AGENT_ID::name   = ANY_NAME;
  AGENT_ID::id     = ANY_ID;
}
 
AGENT_ID::~AGENT_ID ()
{
  delete_check (server);
  delete_check (name);
}

/* REMOTE_PORT::REMOTE_PORT and REMOTE_PORT::~REMOTE_PORT

   Purpose: These procedures are the constructor and destructor for class
            REMOTE_PORT.
*/

REMOTE_PORT::REMOTE_PORT (char *machine, UINT_32 ip, int port)
{
  REMOTE_PORT::machine = my_strcpy (machine);
  REMOTE_PORT::ip      = ip;
  REMOTE_PORT::port    = port;
}

REMOTE_PORT::REMOTE_PORT (const REMOTE_PORT &port)
{
  REMOTE_PORT::machine = my_strcpy (port.machine);
  REMOTE_PORT::ip      = port.ip;
  REMOTE_PORT::port    = port.port;
}

REMOTE_PORT::~REMOTE_PORT ()
{
  delete_check (machine);
}

/* INCOMING::INCOMING and INCOMING::~INCOMING

   Purpose: These procedures are the constructor and destructor for class
            INCOMING.
*/

INCOMING::INCOMING (AGENT_ID *id)
{
  INCOMING::count = 0;
  INCOMING::id    = id;
}

INCOMING::~INCOMING () 
{
  delete_check (id);
}

/* INCOMING_MESSAGE::INCOMING_MESSAGE and INCOMING_MESSAGE::~INCOMING_MESSAGE

   Purpose: These procedures are the constructor and destructor for class
            INCOMING_MESSAGE.
*/

INCOMING_MESSAGE::INCOMING_MESSAGE (AGENT_ID *id, UINT_32 code, char *string)

  : INCOMING (id)

{
  INCOMING_MESSAGE::type   = MESSAGE_ITEM;
  INCOMING_MESSAGE::code   = code;
  INCOMING_MESSAGE::string = my_strcpy (string);
}

INCOMING_MESSAGE::~INCOMING_MESSAGE ()
{
  delete_check (string);
}

/* INCOMING_MEETING::INCOMING_MEETING and INCOMING_MEETING::~INCOMING_MEETING

   Purpose: These procedures are the constructor and destructor for class
            INCOMING_MEETING.
*/

INCOMING_MEETING::INCOMING_MEETING (AGENT_ID *id, UINT_8 status, REMOTE_PORT *port)

  : INCOMING (id)

{
  INCOMING_MEETING::type   = MEETING_ITEM;
  INCOMING_MEETING::status = status;
  INCOMING_MEETING::port   = port;
}

INCOMING_MEETING::~INCOMING_MEETING ()
{
  delete_check (port);
}

/* INCOMING_EVENT::INCOMING_EVENT and INCOMING_EVENT::~INCOMING_EVENT

   Purpose: These procedures are the constructor and destructor for class
            INCOMING_EVENT.
*/

INCOMING_EVENT::INCOMING_EVENT (AGENT_ID *id, char *tag, char *string)

  : INCOMING (id)

{
  INCOMING_EVENT::type   = EVENT_ITEM;
  INCOMING_EVENT::tag    = my_strcpy (tag);
  INCOMING_EVENT::string = my_strcpy (string);
}

INCOMING_EVENT::~INCOMING_EVENT ()
{
  delete_check (tag);
  delete_check (string);
}

/* MESSAGE_EXT::MESSAGE_EXT and MESSAGE_EXT::~MESSAGE_EXT

   Purpose: These procedures are the constructor and destructor for class
            MESSAGE_EXT.  Note that the constructor deletes the contents
            of the INCOMING_MESSAGE structure.
*/

MESSAGE_EXT::MESSAGE_EXT (INCOMING_MESSAGE *message)
{
    /* copy */

  id     = message -> id;
  code   = message -> code;
  string = message -> string;
  
    /* empty out the INCOMING_MESSAGE */
  
  message -> id     = NULL;
  message -> string = NULL; 
}

MESSAGE_EXT::~MESSAGE_EXT ()
{
  delete_check (id);
  delete_check (string);
}

/* MEETING_EXT::MEETING_EXT and MEETING_EXT::~MEETING_EXT

   Purpose: These procedures are the constructor and destructor for class
            MEETING_EXT.  Note that the constructor deletes the contents
            of the INCOMING_MEETING structure.
*/

MEETING_EXT::MEETING_EXT (INCOMING_MEETING *meeting)
{
    /* copy */

  id     = meeting -> id;
  status = meeting -> status;
  port   = meeting -> port;
  
    /* empty out the INCOMING_MESSAGE */
  
  meeting -> id   = NULL;
  meeting -> port = NULL; 
}

MEETING_EXT::~MEETING_EXT ()
{
  delete_check (id);
  delete_check (port);
}

/* EVENT_EXT::EVENT_EXT and EVENT_EXT::~EVENT_EXT

   Purpose: These procedures are the constructor and destructor for class
            EVENT_EXT.  Note that constructor deletes the contents of the
            INCOMING_EVENT structure.
*/

EVENT_EXT::EVENT_EXT (INCOMING_EVENT *event)
{
    /* copy */

  id     = event -> id;
  tag    = event -> tag;
  string = event -> string;
  
    /* empty out the INCOMING_EVENT structure */

  event -> id     = NULL;
  event -> tag    = NULL;
  event -> string = NULL;
}

EVENT_EXT::~EVENT_EXT ()
{
  delete_check (id);
  delete_check (tag);
  delete_check (string);
}
