/* Agent Tcl
   Bob Gray
   25 January 1995

   agentId.h

   This file defines the structures REMOTE_PORT and AGENT_ID.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _AGENT_ID_H
#define _AGENT_ID_H

#include "my_sizes.h"

#ifdef __cplusplus

#define EXTERN extern "C"

#define ANY_SERVER	NULL
const ANY_IP		= 0;
#define ANY_NAME	NULL
const ANY_ID		= 0;

const MEETING_ITEM	= 0;
const MESSAGE_ITEM	= 1;
const EVENT_ITEM	= 2;
const NUM_ITEMS		= 3;

#else

#define EXTERN extern

#define ANY_SERVER	NULL
#define ANY_IP		0
#define ANY_NAME	NULL
#define ANY_ID		0

#define MEETING_ITEM	0
#define MESSAGE_ITEM	1
#define EVENT_ITEM	2
#define NUM_ITEMS	3
 
#endif

struct ID_LIST	      /* list of numeric ids   */
{
  int count;          /* number of numeric ids */
  UINT_32 *ids;       /* the numeric ids       */

#ifdef __cplusplus

    /* constructor and destructor */

  ID_LIST (int count);
 ~ID_LIST ();

#endif
};
  
struct NAME_LIST      /* list of symbolic names   */
{
  int count;          /* number of symbolic names */
  UINT_32 *lengths;   /* lengths of the names     */
  char **names;       /* the symbolic names       */

#ifdef __cplusplus

    /* fill in a name */

  void fill (int i, char *name);

    /* constructor and destructor */

  NAME_LIST (int count);
 ~NAME_LIST ();

#endif
};
 
struct AGENT_ID       /* id of an agent */
{
  char    *server;    /* server name             */
  UINT_32 ip;         /* server IP address       */
  char    *name;      /* name of the agent       */ 
  INT_32 id;          /* numeric id of the agent */  

#ifdef __cplusplus

     /* change the name of the agent */

  void change_name (char *name);

    /* constructor and destructor */

  AGENT_ID (const char *server, UINT_32 ip, const char *name, INT_32 id);
  AGENT_ID (const AGENT_ID &id);
  AGENT_ID (AGENT_ID *id);
  AGENT_ID (void);
 ~AGENT_ID ();

#endif
};

struct MACHINE_ID        /* id of a machine */
{
  char    *server;       /* name of the server                        */
  UINT_32  server_ip;    /* IP address of the server (or UNKNOWN_IP)  */

#ifdef __cplusplus 

    /* constructor and destructor */

  MACHINE_ID (char *server, UINT_32 server_ip);
  MACHINE_ID (MACHINE_ID &id);
 ~MACHINE_ID ();

#endif
};

struct AGENT_INFO  
{
  AGENT_ID *id;	       /* agent identification                 */
  MACHINE_ID *actual;  /* actual agent location                */
  int agentPid;	       /* process id of the running agent      */
  int blockerPid;      /* process id of message handler        */
  int messages;        /* number of messages waiting at server */
  int meetings;        /* number of meetings waiting at server */
  int events;	       /* number of events waiting at server   */

#ifdef __cplusplus

    /* constructor and destructor */

  AGENT_INFO (void);
  ~AGENT_INFO ();

#endif
};

struct REMOTE_PORT     /* port identification */
{
  char *machine;       /* machine name              */
  UINT_32 ip;          /* IP address of the machine */
  int port;            /* port number               */

#ifdef __cplusplus

  REMOTE_PORT (char *machine, UINT_32 ip, int port);
  REMOTE_PORT (const REMOTE_PORT &port);
  ~REMOTE_PORT ();

#endif
};

#ifdef __cplusplus

struct INCOMING    /* incoming message, meeting or event */
{
  int type;        /* MEETING_ITEM, MESSAGE_ITEM or EVENT_ITEM */
  int count;       /* count of pointers to the instance        */
  AGENT_ID *id;    /* sender identification                    */

    /* constructor and destructor */

  INCOMING (AGENT_ID *id);
 ~INCOMING ();
};

struct INCOMING_MESSAGE: public INCOMING     /* incoming message */
{
  UINT_32 code;      /* message code   */
  char *string;      /* message string */

    /* constructor and destructor */

  INCOMING_MESSAGE (AGENT_ID *id, UINT_32 code, char *string);
 ~INCOMING_MESSAGE ();
};

struct INCOMING_MEETING: public INCOMING    /* incoming meeting */
{
  UINT_8 status;      /* status flag      */
  REMOTE_PORT *port;  /* port information */

    /* constructor and destructor */

  INCOMING_MEETING (AGENT_ID *id, UINT_8 status, REMOTE_PORT *port);
 ~INCOMING_MEETING ();
};

struct INCOMING_EVENT: public INCOMING     /* incoming event */
{
  char *tag;     /* event name    */
  char *string;  /* event message */

    /* constructor and destructor */

  INCOMING_EVENT (AGENT_ID *id, char *tag, char *string);
 ~INCOMING_EVENT ();
};

#endif

  /* MESSAGE_EXT, MEETING_EXT and EVENT_EXT are alternative structures */
  /* for holding meetings, message and events.  There are two sets of  */
  /* structures since it was necessary to have both polymorphism and   */ 
  /* compatibility with C.                                             */

struct MESSAGE_EXT
{
  struct AGENT_ID *id;   /* sender identification */
  int code;              /* message code          */  
  char *string;          /* message string        */

#ifdef __cplusplus

    /* constructor and destructor */

  MESSAGE_EXT (struct INCOMING_MESSAGE *message);
 ~MESSAGE_EXT (); 

#endif
};

struct MEETING_EXT
{
  struct AGENT_ID *id;       /* sender identification */
  UINT_8 status;             /* meeting status        */
  struct REMOTE_PORT *port;  /* port information      */

#ifdef __cplusplus
 
    /* constructor and destructor */

  MEETING_EXT::MEETING_EXT (INCOMING_MEETING *meeting);
 ~MEETING_EXT ();

#endif
};

struct EVENT_EXT
{
  struct AGENT_ID *id;   /* sender identification */
  char *tag;             /* event tag             */
  char *string;          /* event string          */

#ifdef __cplusplus

    /* constructor and destructor */

  EVENT_EXT::EVENT_EXT (INCOMING_EVENT *event);
 ~EVENT_EXT ();

#endif
};

#endif
