/* Agent Tcl
   Bob Gray
   10 March 1995

   error.cc

   This file implements class ERROR which provides an error logging
   facility.  ERROR is based on the error logging facilities in
   "UNIX Network Programming" by W. Richard Stevens [Prentice-Hall, 1990].

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <sys/file.h>
#include <errno.h> 
#include <fcntl.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "error.h"
#include "locks.h"
#include "my_strings.h"

/* ERROR::open_log and ERROR::close_log

   Purpose: These procedures open and close the error log.
*/ 

int ERROR::open_log (void)
{
  if (use_stderr)
  {
    logfd = STDERR_FILENO;
  } 
  else if ((logfd = open (logname, O_RDWR | O_APPEND | O_CREAT, 0600)) >= 0)
  {
    get_lock (logfd, NULL);
  }

  return (logfd >= 0);
}

void ERROR::close_log (void)
{
  if (!use_stderr) 
  {
    release_lock (logfd, NULL); 
    close (logfd);
  }
}

/* ERROR::ERROR

   Purpose: This procedure is the constructor for class ERROR.
*/

ERROR::ERROR (char *logname)
{
  ERROR::logname    = my_strcpy (logname);
  ERROR::use_stderr = (logname == NULL);

  if (!open_log())
  {
    fprintf (stderr, "ERROR: unable to open error log %s\n", logname);
    exit (1);
  }

  close_log ();
}

/* ERROR::ERROR

   Purpose: This procedure is the constructor for class ERROR.
*/

ERROR::~ERROR ()
{
  delete logname;
}

/* ERROR::error_date

   Purpose: This procedure logs the date and time.
*/

void ERROR::error_date (void)
{
  time_t current_time;
  char *ascii_time;

  current_time = time (NULL);
  ascii_time   = ctime (&current_time);
  
  write (logfd, ascii_time, strlen(ascii_time));
}
 
/* ERROR::error_app_cont

   Purpose: This procedure logs a NON-FATAL application error.
*/

void ERROR::error_app_cont (char *message)
{
  if (open_log())
  {
    error_date ();
    write (logfd, message, strlen(message)); 
    write (logfd, "\n", 1);
    close_log ();
  }
}

/* ERROR::error_app_quit

   Purpose: This procedure logs a fatal application error.
*/

void ERROR::error_app_quit (char *message)
{
  error_app_cont (message);
  exit (1);
}

/* ERROR::error_sys_cont

   Purpose: This procedure logs a NON-FATAL system error.
*/

void ERROR::error_sys_cont (char *message)
{
  char temp[32];
 
  if (open_log ())
  {
    sprintf (temp, "%d", errno);
    error_date ();
    write (logfd, message, strlen(message)); 
    write (logfd, " (errno = ", 10);
    write (logfd, temp, strlen(temp));
    write (logfd, ")\n", 2);
    close_log ();
  }
}

/* ERROR::error_sys_quit

   Purpose: This procedure logs a FATAL system error.
*/

void ERROR::error_sys_quit (char *message)
{
  error_sys_cont (message);
  exit (1);
}
