/* Agent Tcl
   Bob Gray
   5 April 1995

   hash.cc

   This file implements the hash table classes.   

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#ifndef SOLARIS
#include <strings.h>
#endif
#include "hash.h"
#include "my_alloc.h"
#include "my_strings.h"
#include "truefalse.h"

/* HASH_SEARCH::HASH_SEARCH

   Purpose: This procedure is the constructor for class HASH_SEARCH.

     Input: hash = hash table that we are searching
*/

HASH_SEARCH::HASH_SEARCH (HASH *hash)
{
    /* assertions on the parameters */

  assert (hash != NULL);

    /* initialize */

  HASH_SEARCH::hash   = hash;
  HASH_SEARCH::bucket = 0;
  HASH_SEARCH::node   = hash -> table[0];
}

/* HASH_SEARCH::next

   Purpose: Get the next HASH_NODE in the hash table

     Input: None

    Output: The procedure returns NULL if there are no more entries in the
            hash table.  Otherwise the procedure returns a pointer to the next
            node in the hash table.  This node should NOT be deleted by the
            caller.
*/

HASH_NODE *HASH_SEARCH::next (void)
{ 
  HASH_NODE *nextNode;

    /* is there another node in the same bucket */

  if (node != NULL) {
    nextNode = node;
    node = node -> next;
    return (nextNode);
  }

    /* no so move to the next nonempty bucket */

  while (bucket < hash -> max_index) {

      /* next bucket */

    bucket += 1;

      /* is there a node in that bucket */

    if ((node = hash -> table[bucket]) != NULL) {
      nextNode = node;
      node = node -> next;
      return (nextNode);
    }
  }

    /* there are no more nonempty buckets */

  return (NULL);
}

/* HASH_NODE::HASH_NODE

   Purpose: This procedure is the constructor for class HASH_NODE.

     Input: key        = the integer key
            string_key = the string key
            data       = the entry data
            next       = next node in a list
*/

HASH_NODE::HASH_NODE (long key, char *string_key, void *data, HASH_NODE *next)
{
  HASH_NODE::key        = key;
  HASH_NODE::data       = data;
  HASH_NODE::next       = next;
  HASH_NODE::string_key = my_strcpy (string_key);
}

/* HASH_NODE::~HASH_NODE

   Purpose: This procedure is the destructor for class HASH_NODE.
*/

HASH_NODE::~HASH_NODE ()
{
  delete_check (string_key);
} 

/* HASH::HASH

   Purpose: This procedure is the constructor for class HASH.

     Input: buckets = number of buckets in the hash table

            buckets must be a power of two since the hash function is
            just a bit-wise AND.
*/

HASH::HASH (unsigned buckets)
{
  HASH::max_index = buckets - 1;
  HASH::entries   = 0;
  HASH::table     = new HASH_NODE * [buckets];

  for (unsigned i = 0; i <= max_index; i++) {
    table[i] = NULL;
  }
}

/* HASH::~HASH

   Purpose: This procedure is the destructor for class HASH.
*/

HASH::~HASH ()
{
  register HASH_NODE *current;
  register HASH_NODE *next;

    /* delete the table entries */

  for (unsigned i = 0; i <= max_index; i++) {
    for (current = table[i]; current != NULL; current = next) {
      next = current -> next;
      delete current;
    } 
  }

    /* delete the table */

  delete table;
}

/* HASH::add

   Purpose; Add an entry to the hash table

     Input: key        = entry integer key
            string_key = entry string key 
            data       = entry data

    Output: If the entry is in the hash table, the procedure returns a
            pointer to HASH_NODE structure and sets "there" to TRUE.
            If the entry is not in the hash table, the procedure adds the
            entry to the table, returns a pointer to the new HASH_NODE
            structure and sets "there" to FALSE.
*/

HASH_NODE *HASH::add (long key, char *string_key, void *data, int &there)
{
  register HASH_NODE *current;

    /* check if the entry is in the table */

  long index = hash (key, string_key);

  for (current = table[index]; current != NULL; current = current -> next)
    if (current -> key == key)
      if ((string_key == NULL) || (!strcmp (current -> string_key, string_key))) {
        there = TRUE;
        return current;
      }

    /* add a new entry to the table */

  current      = new HASH_NODE (key, string_key, data, table[index]);
  table[index] = current;
  entries     += 1;
  there        = FALSE;
    
     /* return the new entry */

  return current;
}

/* HASH::lookup

   Purpose: Lookup an entry in the hash table

     Input: key        = entry integer key
            string_key = entry string key
 
    Output: The procedure returns NULL if the entry is not in the hash table.
            Otherwise the procedure returns a pointer to the HASH_NODE
            structure.
*/

HASH_NODE *HASH::lookup (long key, char *string_key)
{
  register HASH_NODE *current;

    /* check if the entry is in the table */

  long index = hash(key, string_key);

  for (current = table[index]; current != NULL; current = current -> next)
    if (current -> key == key)
      if ((string_key == NULL) || (!strcmp (current -> string_key, string_key)))
        return current;

   return NULL;
}

/* HASH::remove

   Purpose: Remove an entry in the hash table

     Input: key        = entry integer key
            string key = entry string key
 
    Output: The procedure returns FALSE if the entry is not in the hash
            table.  Otherwise the procedure removes the entry and returns
            TRUE.
*/

int HASH::remove (long key, char *string_key)
{
  register HASH_NODE *current;
  register HASH_NODE *previous;

     /* check if the entry is in the table */

 long index = hash(key, string_key);

  for (previous = NULL, current = table[index]; current != NULL; previous = current, current = current -> next)
    if (current -> key == key)
      if ((string_key == NULL) || (!strcmp(current -> string_key, string_key))) {
        if (previous == NULL) {
          table[index] = current -> next;
        } else {
           previous -> next = current -> next;
        }

        delete current;
        entries -= 1;
        return TRUE;
      }

  return FALSE;
} 
