/* Agent Tcl
   Bob Gray
   9 February 1995

   message.cc

   This file implements the functions associated with messages.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <stdio.h>
#include <stdlib.h>
#include "message.h"
#include "my_alloc.h"
#include "truefalse.h"

  /* messages going up the socket */

static MESSAGE_TYPE up_socket_types[NUM_REQUESTS] = 
{
  {REQ_BEGIN   , 3, {MESG_TYPE_STR,       /* actual machine name           */
                     MESG_TYPE_IP,        /* actual machine IP address     */
		     MESG_TYPE_LONG,      /* process id                    */
		     0, 0, 0, 0, 0, 0, 0, 0}}, 
  {REQ_NAME    , 2, {MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_STR,       /* requested name                */
		     0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_END     , 3, {MESG_TYPE_STR,       /* local name                    */
		     MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_BYTE,      /* END_NORMAL or END_FORCE       */
		     0, 0, 0, 0, 0, 0, 0, 0}}, 
  {REQ_SCRIPT  ,10, {MESG_TYPE_STR,       /* root server                   */
                     MESG_TYPE_IP,        /* root IP address               */
                     MESG_TYPE_STR,       /* root name                     */
                     MESG_TYPE_LONG,      /* root id                       */
		     MESG_TYPE_STR,       /* local server                  */
		     MESG_TYPE_IP,        /* local IP address              */
		     MESG_TYPE_STR,       /* local name                    */
                     MESG_TYPE_LONG,      /* local id                      */
		     MESG_TYPE_STR,	  /* language name                 */
                     MESG_TYPE_STR, 0}},  /* script                        */
  {REQ_STATE   ,10, {MESG_TYPE_STR,       /* root server                   */
                     MESG_TYPE_IP,        /* root IP address               */
                     MESG_TYPE_STR,       /* root name                     */
                     MESG_TYPE_LONG,      /* root id                       */
		     MESG_TYPE_STR,       /* source server                 */
		     MESG_TYPE_IP,        /* source IP address             */
		     MESG_TYPE_STR,       /* source name                   */
		     MESG_TYPE_LONG,      /* source id                     */
		     MESG_TYPE_STR,       /* language name                 */
                     MESG_TYPE_STR, 0}},  /* state image                   */
  {REQ_MESSAGE , 8, {MESG_TYPE_STR,       /* local server                  */
                     MESG_TYPE_IP,        /* local IP address              */
                     MESG_TYPE_STR,       /* local name                    */
                     MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_STR,       /* destination name              */
                     MESG_TYPE_LONG,      /* destination id                */
                     MESG_TYPE_LONG,      /* message code                  */
                     MESG_TYPE_STR,       /* message string                */
		     0, 0, 0}},
  {REQ_MEETING ,10, {MESG_TYPE_STR,       /* local server                  */
		     MESG_TYPE_IP,        /* local IP address              */
                     MESG_TYPE_STR,       /* local name                    */
                     MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_STR,       /* destination name              */
                     MESG_TYPE_LONG,      /* destination id                */
                     MESG_TYPE_BYTE,      /* MEET_ACCEPT, MEET_REFUSED ... */
		     MESG_TYPE_STR,       /* actual machine                */
		     MESG_TYPE_IP,        /* actual IP address             */
		     MESG_TYPE_LONG, 0}},  /* port number                   */
  {REQ_EVENT   , 8, {MESG_TYPE_STR,       /* local server                  */
                     MESG_TYPE_IP,        /* local server IP               */
                     MESG_TYPE_STR,       /* local name                    */
                     MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_STR,       /* destination name              */
                     MESG_TYPE_LONG,      /* destination id                */
                     MESG_TYPE_STR,       /* event tag                     */
                     MESG_TYPE_STR,       /* event string                  */
		     0, 0, 0}},
  {REQ_GET     , 1, {MESG_TYPE_LONG,      /* local id                      */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_ENCRYPT , 4, {MESG_TYPE_STR,       /* local server                  */
                     MESG_TYPE_IP,        /* local server IP               */
                     MESG_TYPE_STR,       /* local server public key       */
                     MESG_TYPE_BINARY,    /* encrpyted message             */
		     0, 0, 0, 0, 0, 0, 0}},
  {REQ_INFO    , 3, {MESG_TYPE_BYTE,      /* INFO_ALL_IDS ...              */
                     MESG_TYPE_STR,       /* local name                    */
                     MESG_TYPE_LONG,      /* local id                      */
		     0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_SIGN    , 0, {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}}                   
};

  /* mapping a socket message to a pipe message */

static INT_8 socket_to_pipe_map[NUM_REQUESTS][MAX_ELEMENTS] =
{
  { 0,  1,  2, -2, -2, -2, -2, -2, -2, -2, -2},       /* REQ_BEGIN   */
  { 0,  1, -2, -2, -2, -2, -2, -2, -2, -2, -2},       /* REQ_NAME    */
  { 0,  1,  2, -2, -2, -2, -2, -2, -2, -2, -2},       /* REQ_END     */
  {-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2},       /* REQ_SCRIPT  */
  {-2  -2, -2, -2, -2, -2, -2, -2, -2, -2, -2},       /* REQ_STATE   */
  { 0,  1,  2,  3,  4,  5,  6,  7, -2, -2, -2},       /* REQ_MESSAGE */
  { 0,  1,  2,  3,  4,  5,  6,  7,  8,  9, -2},       /* REQ_MEETING */
  { 0,  1,  2,  3,  4,  5,  6,  7, -2, -2, -2},       /* REQ_EVENT   */
  { 0, -1, -2, -2, -2, -2, -2, -2, -2, -2, -2},       /* REQ_GET     */
  {-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2},       /* REQ_ENCRYPT */
  { 0,  1,  2, -2, -2, -2, -2, -2, -2, -2, -2},	      /* REQ_INFO    */
  {-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2}        /* REQ_SIGN    */
}; 

  /* messages going up the pipe */

static MESSAGE_TYPE up_pipe_types[NUM_REQUESTS] = 
{
  {REQ_BEGIN   , 3, {MESG_TYPE_STR,       /* actual machine name           */
                     MESG_TYPE_IP,        /* actual machine IP address     */
                     MESG_TYPE_LONG,      /* process id                    */
		     0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_NAME    , 2, {MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_STR,       /* requested name                */
		     0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_END     , 3, {MESG_TYPE_STR,       /* local name                    */
		     MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_BYTE,      /* END_NORMAL or END_FORCE       */
		     0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_SCRIPT  , 0, {0,                   /* ****** UNUSED MESSAGE ******  */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_STATE   , 0, {0,                   /* ****** UNUSED MESSAGE ******  */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_MESSAGE , 8, {MESG_TYPE_STR,       /* local server                  */
                     MESG_TYPE_IP,        /* local IP address              */
                     MESG_TYPE_STR,       /* local name                    */
                     MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_STR,       /* destination name              */
                     MESG_TYPE_LONG,      /* destination id                */
                     MESG_TYPE_LONG,      /* message code                  */
                     MESG_TYPE_STR,       /* message string                */
		     0, 0, 0}}, 
  {REQ_MEETING ,10, {MESG_TYPE_STR,       /* local server                  */
		     MESG_TYPE_IP,        /* local IP address              */
                     MESG_TYPE_STR,       /* local name                    */
                     MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_STR,       /* destination name              */
                     MESG_TYPE_LONG,      /* destination id                */
                     MESG_TYPE_BYTE,      /* MEET_ACCEPT, MEET_REFUSED ... */
		     MESG_TYPE_STR,       /* actual machine                */
		     MESG_TYPE_IP,        /* actual IP address             */
		     MESG_TYPE_LONG, 0}},    /* port number                   */
  {REQ_EVENT   , 8, {MESG_TYPE_STR,       /* local server                  */
                     MESG_TYPE_IP,        /* local server IP               */
                     MESG_TYPE_STR,       /* local name                    */
                     MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_STR,       /* destination name              */
                     MESG_TYPE_LONG,      /* destination id                */
                     MESG_TYPE_STR,       /* event tag                     */
                     MESG_TYPE_STR,       /* event string                  */
		     0, 0, 0}},
  {REQ_GET     , 2, {MESG_TYPE_LONG,      /* local id                      */
                     MESG_TYPE_LONG,      /* local pid                     */
		     0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_ENCRYPT , 0, {0,                   /* ****** UNUSED MESSAGE ******  */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_INFO    , 3, {MESG_TYPE_BYTE,      /* INFO_ALL_IDS ...              */
                     MESG_TYPE_STR,       /* local name                    */
                     MESG_TYPE_LONG,      /* local id                      */
		     0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_SIGN    , 0, {0,                  /* ****** UNUSED MESSAGE ******  */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}}
};

  /* messages going down the pipe */

static MESSAGE_TYPE down_pipe_types[NUM_RESPONSES] =
{
  {RESP_ID     , 4, {MESG_TYPE_STR,       /* server                        */
                     MESG_TYPE_IP,        /* address                       */
		     MESG_TYPE_STR,       /* symbolic name                 */
                     MESG_TYPE_LONG,      /* numeric id                    */
		     0, 0, 0, 0, 0, 0, 0}},
  {RESP_OK     , 0, {0,
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_ERROR  , 0, {0,
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_MESSAGE, 6, {MESG_TYPE_STR,       /* source server                 */
		     MESG_TYPE_IP,        /* source IP address             */
                     MESG_TYPE_STR,       /* source name                   */
                     MESG_TYPE_LONG,      /* source id                     */
                     MESG_TYPE_LONG,      /* message code                  */
                     MESG_TYPE_STR,       /* message string                */
		     0, 0, 0, 0, 0}},
  {RESP_MEETING, 8, {MESG_TYPE_STR,       /* source server                 */
                     MESG_TYPE_IP,        /* source IP address             */
                     MESG_TYPE_STR,       /* source name                   */
                     MESG_TYPE_LONG,      /* source id                     */
                     MESG_TYPE_BYTE,      /* MEET_ACCEPT, MEET_REFUSED ... */
                     MESG_TYPE_STR,       /* actual machine                */
                     MESG_TYPE_IP,        /* actual IP address             */ 
                     MESG_TYPE_LONG,      /* port number                   */
		     0, 0, 0}},
  {RESP_EVENT  , 6, {MESG_TYPE_STR,       /* source server                 */
		     MESG_TYPE_IP,        /* source IP address             */
                     MESG_TYPE_STR,       /* source name                   */
                     MESG_TYPE_LONG,      /* source id                     */
                     MESG_TYPE_STR,       /* event tag                     */
                     MESG_TYPE_STR,       /* event string                  */
		     0, 0, 0, 0, 0}},
  {RESP_NONE   , 0, {0,
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_IDS    , 1, {MESG_TYPE_IDS,       /* ID_LIST                        */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_NAMES  , 1, {MESG_TYPE_NAMES,     /* NAME_LIST                      */ 
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_INFO   , 11, {MESG_TYPE_STR,       /* actual server                  */
                     MESG_TYPE_IP,        /* actual server IP               */
                     MESG_TYPE_STR,       /* agent server name              */
                     MESG_TYPE_IP,        /* agent server IP               */
                     MESG_TYPE_STR,       /* agent name                     */
                     MESG_TYPE_LONG,      /* agent numeric id               */
                     MESG_TYPE_LONG,      /* process id (agent interpreter) */
                     MESG_TYPE_LONG,      /* process id (agent handler)     */
                     MESG_TYPE_LONG,      /* messages in queue              */
                     MESG_TYPE_LONG,      /* meetings in queue              */
                     MESG_TYPE_LONG      /* events in queue                */
		     }}
}; 

  /* mapping a socket message to a pipe message */

static INT_8 pipe_to_socket_map[NUM_RESPONSES][MAX_ELEMENTS] =
{
  { 0,  1,  2,  3, -2, -2, -2, -2, -2, -2, -2},    /* RESP_ID       */
  {-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2},    /* RESP_OK       */
  {-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2},    /* RESP_ERROR    */
  { 0,  1,  2,  3,  4,  5, -2, -2, -2, -2, -2},    /* RESP_MESSAGE  */
  { 0,  1,  2,  3,  4,  5,  6,  7, -2, -2, -2},    /* RESP_MEETING  */
  { 0,  1,  2,  3,  4,  5, -2, -2, -2, -2, -2},    /* RESP_EVENT    */
  {-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2},    /* RESP_NONE     */
  { 0, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2},    /* RESP_IDS      */
  { 0, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2},    /* RESP_NAMES    */
  { 0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10}     /* RESP INFO     */
};

static MESSAGE_TYPE down_socket_types[NUM_RESPONSES] =
{
  {RESP_ID     , 4, {MESG_TYPE_STR,       /* server                        */
                     MESG_TYPE_IP,        /* address                       */
		     MESG_TYPE_STR,	  /* symbolic name                 */
                     MESG_TYPE_LONG,      /* numeric id                    */
		     0, 0, 0, 0, 0, 0, 0}},
  {RESP_OK     , 0, {0,
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_ERROR  , 0, {0,
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_MESSAGE, 6, {MESG_TYPE_STR,       /* source server                 */
		     MESG_TYPE_IP,        /* source IP address             */
                     MESG_TYPE_STR,       /* source name                   */
                     MESG_TYPE_LONG,      /* source id                     */
                     MESG_TYPE_LONG,      /* message code                  */
                     MESG_TYPE_STR,       /* message string                */
		     0, 0, 0, 0, 0}},
  {RESP_MEETING, 8, {MESG_TYPE_STR,       /* source server                 */
                     MESG_TYPE_IP,        /* source IP address             */
                     MESG_TYPE_STR,       /* source name                   */
                     MESG_TYPE_LONG,      /* source id                     */
                     MESG_TYPE_BYTE,      /* MEET_ACCEPT, MEET_REFUSED ... */
                     MESG_TYPE_STR,       /* actual machine                */
                     MESG_TYPE_IP,        /* actual IP address             */ 
                     MESG_TYPE_LONG,      /* port number                   */
		     0, 0, 0}},
  {RESP_EVENT  , 6, {MESG_TYPE_STR,       /* source server                 */
		     MESG_TYPE_IP,        /* source IP address             */
                     MESG_TYPE_STR,       /* source name                   */
                     MESG_TYPE_LONG,      /* source id                     */
                     MESG_TYPE_STR,       /* event tag                     */
                     MESG_TYPE_STR,       /* event string                  */
		     0, 0, 0, 0, 0}},
  {RESP_NONE   , 0, {0,
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_IDS    , 1, {MESG_TYPE_IDS,      /* ID_LIST                       */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_NAMES  , 1, {MESG_TYPE_NAMES,    /* NAME_LIST                     */ 
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_INFO   , 11, {MESG_TYPE_STR,       /* actual server                  */
                     MESG_TYPE_IP,        /* actual server IP               */
                     MESG_TYPE_STR,       /* agent server name              */
                     MESG_TYPE_IP,        /* agent server IP               */
                     MESG_TYPE_STR,       /* agent name                     */
                     MESG_TYPE_LONG,      /* agent numeric id               */
                     MESG_TYPE_LONG,      /* process id (agent interpreter) */
                     MESG_TYPE_LONG,      /* process id (agent handler)     */
                     MESG_TYPE_LONG,      /* messages in queue              */
                     MESG_TYPE_LONG,      /* meetings in queue              */
                     MESG_TYPE_LONG       /* events in queue                */
		     }}
}; 

  /* messages going to an interpreter */

static MESSAGE_TYPE to_interpreter_types[NUM_INTERP] = 
{
  {INTERP_SCRIPT, 8, {MESG_TYPE_STR,       /* root server                  */
                      MESG_TYPE_IP,        /* root IP address              */
                      MESG_TYPE_STR,       /* root name                    */
                      MESG_TYPE_LONG,      /* root id                      */
		      MESG_TYPE_STR,       /* local server                 */
                      MESG_TYPE_IP,        /* local IP address             */
                      MESG_TYPE_LONG,      /* local id                     */
                      MESG_TYPE_STR,       /* the script                   */
		      0, 0, 0}},
  {INTERP_STATE, 8,  {MESG_TYPE_STR,       /* root server                  */
                      MESG_TYPE_IP,        /* root IP address              */
                      MESG_TYPE_STR,       /* root name                    */
                      MESG_TYPE_LONG,      /* root id                      */
		      MESG_TYPE_STR,       /* local server                 */
                      MESG_TYPE_IP,        /* local IP address             */
                      MESG_TYPE_LONG,      /* local id                     */
                      MESG_TYPE_STR,       /* the script                   */
		      0, 0, 0}},
};

MESSAGE_SET up_socket      = {NUM_REQUESTS , up_socket_types     };
MESSAGE_SET up_pipe        = {NUM_REQUESTS , up_pipe_types       };
MESSAGE_SET down_pipe      = {NUM_RESPONSES, down_pipe_types     };
MESSAGE_SET down_socket    = {NUM_RESPONSES, down_socket_types   };
MESSAGE_SET to_interpreter = {NUM_INTERP   , to_interpreter_types};

/* MESSAGE::~MESSAGE

   Purpose: This procedure is the destructor for class MESSAGE.
*/

MESSAGE::~MESSAGE ()
{
  for (int i = 0; i < n; i++) {
    if (elements[i].dealloc) {
      delete_check (elements[i].string);
    }
  }

  delete_check (buffer);
}
 
/* MESSAGE::MESSAGE

   Purpose: This procedure is a constructor for class MESSAGE.

     Input: type = number and types of the message elements
*/

MESSAGE::MESSAGE (const MESSAGE_TYPE &type)
{
  buffer = NULL;
  flag   = type.flag;
  n      = type.n;

  for (int i = 0; i < n; i++) {
    types[i]            = type.types[i];
    elements[i].dealloc = FALSE;
    elements[i].string  = NULL;
    elements[i].number  = 0;
  }
}

/* map_message_to_pipe

   Purpose: Given a message that has arrived over the socket, create the
            message that should go up the pipe, copying over all relevant
            components of the message.
*/
 
MESSAGE *map_message_to_pipe (MESSAGE *message)
{ 
  int i;
  int j; 
  UINT_8 flag = message -> flag;

  MESSAGE *pipe_message = new MESSAGE (up_pipe.messages[flag]);

  for (i = 0; i < pipe_message -> n; i++) {
    if ((j = socket_to_pipe_map[flag][i]) != -1) {
      pipe_message -> elements[i].number = message -> elements[j].number;
      pipe_message -> elements[i].string = message -> elements[j].string;
    }
  }

  return pipe_message;
}

/* map_message_to_socket

   Purpose: Given a message that has arrived over the pipe, create the
            message that should go down the socket, copying over all
            relevant components of the message.
*/

MESSAGE *map_message_to_socket (MESSAGE *message)
{
  int i;
  int j;
  UINT_8 flag = message -> flag;

  MESSAGE *sock_message = new MESSAGE (down_socket.messages[flag]);

  for (i = 0; i < sock_message -> n; i++) {
    if ((j = pipe_to_socket_map[flag][i]) != -1) {
      sock_message -> elements[i].number = message -> elements[j].number;
      sock_message -> elements[i].string = message -> elements[j].string;
    }
  }

  return sock_message;	 
}
