/* Agent Tcl
   Bob Gray
   29 September 1995

   sigio.cc

   This file implements the class which handles the SIGIO interrupt.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <assert.h>
#include <errno.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include "interrupt.h"
#include "sigio.h"
#include "truefalse.h"

  /* static data */

int            SIGIO_HANDLER::count      = 0;
SIGIO_HANDLER *SIGIO_HANDLER::head       = NULL;

/* sigio_handler

   Purpose: This procedure is called when SIGIO fires.

     Input: signo = SIGIO

    Output: The procedure sets the event flag in class SIGIO and calls all of
            the client functions specified in the instances of class SIGIO.
*/

void sigio_handler (int signo)
{
  int oldErrno;
  register SIGIO_HANDLER *next;
  register SIGIO_HANDLER *handle;
    
    /* save errno */

  oldErrno = errno;
 
    /* call the client functions */

  for (handle = SIGIO_HANDLER::head; handle != NULL; handle = next) {

      /* The function could delete the instance of SIGIO_HANDLER to which   */
      /* handle refers.  So we get the "next" field now rather than later.  */
      /* Note that the function should NOT delete this instace since that   */
      /* would involve calls to "free" or "malloc" which are not reentrant. */

    next = handle -> next;

      /* call the appropriate function */

    if (handle -> function != NULL) {
      (handle -> function)(handle -> data);
    }
  }

    /* restore errno */

  errno = oldErrno;
}

/* SIGIO_HANDLER::SIGIO_HANDLER

   Purpose: This procedure is the constructor for class SIGIO_HANDLER.

     Input: function = function to be called when SIGIO fires
            data     = client data to be passed to the function

      NOTE: This procedure will lead to race conditions unless it is used as
            follows.

            1. Block SIGIO
            2. Create an instance of SIGIO_HANDLER
            3. Turn on aysnchronous I/O for the file descriptor of interest
            4. Unblock SIGIO 
*/

SIGIO_HANDLER::SIGIO_HANDLER (SigioFunc function, SigioClientData data)
{
  sigset_t oldMask;     /* old interrupt mask      */
  sigset_t newMask;     /* new interrupt mask      */

    /* assertions on the parameters */

  assert (function != NULL);

    /* block SIGIO */

  sigemptyset (&newMask);
  sigaddset (&newMask, SIGIO);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

     /* is this the first handler */

  if (count == 0) {
    head     = this;
    next     = NULL;
    previous = NULL;
    install_signal_intr (SIGIO, (void (*) (...)) sigio_handler, 0);
  } else {
    head -> previous = this;
    next             = head;
    previous         = NULL;
    head             = this;
  }

    /* set the function and data */

  SIGIO_HANDLER::function = function;
  SIGIO_HANDLER::data     = data;
  SIGIO_HANDLER::count   += 1;

    /* unblock SIGIO */

  sigprocmask (SIG_SETMASK, &oldMask, NULL);
}

/* SIGIO_HANDLER::~SIGIO_HANDLER

   Purpose: This procedure is the destructor for class SIGIO_HANDLER.
*/

SIGIO_HANDLER::~SIGIO_HANDLER (void)
{
  sigset_t oldMask;     /* old interrupt mask      */
  sigset_t newMask;     /* new interrupt mask      */

    /* block SIGIO */

  sigemptyset (&newMask);
  sigaddset (&newMask, SIGIO);

    /* splice this instance out of the list */

  if (count == 1) {

    head = NULL;
    install_signal_intr (SIGIO, (void (*) (...)) SIG_IGN, 0);

  } else {

    if (next != NULL) {
      next -> previous = previous;
    }

    if (previous != NULL) {
      previous -> next = next;
    } else {
      head = next;
    }
  } 

  SIGIO_HANDLER::count -= 1;

    /* unblock SIGIO */

  sigprocmask (SIG_SETMASK, &oldMask, NULL);
}
