/* Agent Tcl
   Bob Gray
   26 June 1995

   tcpip.cc

   This file implements the routines that handle TCP/IP connections.  These
   routines are based on the examples in [W. Richard Stevens, Unix Network
   Programming, Prentice Hall, 1990] and on the code that Saurab Nog
   developed as part of his A.I. project.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#ifdef AIX
#include <sys/select.h>
#endif
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/un.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <errno.h>
#ifdef SYSV
#ifndef SOLARIS 
#include <bstring.h>
#endif
#endif
#ifndef COSMO
#include <unistd.h>
#else
#include </usr/local/lib/gcc-lib/rs6000-ibm-aix3.2.5/2.6.3/include/unistd.h>
#endif
#include "timers.h"
#include "my_strings.h"
#include "tcpip.h"
#include "truefalse.h"

const MAX_NAME_LENGTH = 128;

/* tcpip_asynchronous

   Purpose: Turn on asynchronous I/O for a socket

     Input: sockfd = the socket descriptor

    Output: The procedure returns TCPIP_FCNTL on error and 0 otherwise.
*/

int tcpip_asynchronous (int sockfd)
{
  int flags;

  if (fcntl (sockfd, F_SETOWN, getpid()) < 0) {
    return TCPIP_FCNTL;
  }

  if ((flags = fcntl (sockfd, F_GETFL)) < 0) {
    return TCPIP_FCNTL;
  }

  if (fcntl (sockfd, F_SETFL, flags | FASYNC) < 0) {
    return TCPIP_FCNTL;
  }

  return 0;
}

/* tcpip_getsockname

   Purpose: Get the IP address and the port number associated with the
            LOCAL end of a socket

     Input: sockfd = socket descriptor

    Output: The procedure returns TCPIP_UNABLE_TO_GET_PORT on error.
            Otherwise the procedure fills in *ip and *port with the IP 
            address and port number.
*/

int tcpip_getsockname (int sockfd, UINT_32 *ip, UINT_32 *port)
{
  struct sockaddr_in address;  
  int size = sizeof(sockaddr_in);

  if (getsockname(sockfd, (struct sockaddr *) &address, &size) < 0)
    return TCPIP_UNABLE_TO_GET_PORT;

  *ip   = (UINT_32) address.sin_addr.s_addr;
  *port = ntohs (address.sin_port);
  return 0;
}

/* tcpip_getpeername

   Purpose: Get the IP address and the port number associated with the
            REMOTE end of a CONNECTED socket

     Input: sockfd = socket descriptor

    Output: The procedure returns TCPIP_UNABLE_TO_GET_PORT on error.
            Otherwise the procedure fills in *ip and *port with the IP 
            address and port number.
*/

int tcpip_getpeername (int sockfd, UINT_32 *ip, UINT_32 *port)
{
  struct sockaddr_in address;  
  int size = sizeof(sockaddr_in);

  if (getpeername(sockfd, (struct sockaddr *) &address, &size) < 0)
    return TCPIP_UNABLE_TO_GET_PORT;

  *ip   = (UINT_32) address.sin_addr.s_addr;  
  *port = ntohs (address.sin_port);
  return 0;
}

/* tcpip_getip

   Purpose: Get the IP number of a machine

     Input: machine = name of the machine

    Output: The procedure returns TCPIP_UNABLE_TO_CONVERT_NAME on error.
            Otherwise the procedure returns 0 and fills in *ip with the IP
            number of the machine. 
*/

int tcpip_getip (char *machine, UINT_32 *ip)
{
  char *name;
  struct hostent *serverInfo;
  int return_code = TCPIP_UNABLE_TO_CONVERT_NAME;
 
    /* strip the whitespace off the machine name and lookup IP address */

  if ((name = strip_whitespace (machine)) != NULL) { 
    if (*name != '\0') {  
      if ((*ip = (UINT_32) inet_addr(name)) != INADDR_NONE) {
        return_code = 0;
      } else if ((serverInfo = gethostbyname(name)) != NULL) {
#ifdef SOLARIS
	memcpy ((void *) ip, (void *) serverInfo -> h_addr, sizeof(UINT_32));
#else
        bcopy (serverInfo -> h_addr, (char *) ip, sizeof(UINT_32));
#endif
        return_code = 0;
      }
    }

    delete name;
  }

  return return_code;
}
      
/* tcpip_hostname

   Purpose: Get the name of the current host

     Input: None

    Output: The procedure returns NULL if it is unable to get the host name.
            Otherwise the procedure returns a pointer to a dyanmically
            allocated string that contains the hostname.
*/

char *tcpip_hostname (void)
{
  char *name_copy;
  char name[MAX_NAME_LENGTH];

  if (gethostname (name, MAX_NAME_LENGTH) < 0) {
    return NULL;
  }

  name_copy = new char [strlen(name) + 1];
  strcpy (name_copy, name);
  return name_copy;
}

/* tcpip_socket

   Purpose: This function creates a socket.

     Input: None

   Purpose: The procedure returns TCPIP_UNABLE_TO_CREATE if it is unable to
            create the socket.  Otherwise the procedure returns the sockfd.
*/

int tcpip_socket (void)
{
  int sockfd;
  int flag = 1;

    /* create the socket */

  if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
    return TCPIP_UNABLE_TO_CREATE;
  }

    /* turn on the TCP_NODELAY flag */

  if (setsockopt (sockfd, IPPROTO_TCP, TCP_NODELAY, (char *) &flag, sizeof(int)) < 0) {
    close (sockfd);
    return TCPIP_UNABLE_TO_CREATE;
  }

  return sockfd;
}

/* tcpip_unix_socket

   Purpose: This function creates a Unix domain socket.

     Input: None

    Output: The procedure returns TCPIP_UNABLE_TO_CREATE if it is unable to
            create the socket.  Otherwise the procedure returns the sockfd.
*/

int tcpip_unix_socket (void)
{
  int sockfd;

  if ((sockfd = socket (AF_UNIX, SOCK_STREAM, 0)) < 0) {
    return TCPIP_UNABLE_TO_CREATE;
  }

  return sockfd;
}
 
/* tcpip_connect 

   Purpose: Connect to a socket on a remote machine

     Input: sockfd    = socket descriptor
            server    = remote machine
            server_ip = IP address of the remote machine (or UNKNOWN_IP)
            port      = port on the remote machine 
            stop      = stop time (in clock ticks)
						
    Output: The procedure returns TCPIP_UNABLE_TO_CONVERT_NAME if it can
            not determine the network address of the machine.  The procedure
            returns TCPIP_UNABLE_TO_CONNECT if it is unable to connect to 
            the specified port on that machine.  The procedure returns
            TCPIP_TIMEOUT if the stop time is reached before the connection
            is established.  Otherwise the procedure returns 0.
*/

int tcpip_connect (int sockfd, char *server, UINT_32 server_ip, int port, clock_t stop)
{
  int code;
  int mode;
  int new_mode;
  fd_set writeSet;
  struct timeval timeout; 
  struct timeval *timeoutPtr;
  struct sockaddr_in serverAddress;    

    /* convert the machine name to an IP number */

  if (server_ip == UNKNOWN_IP) {
    if ((code = tcpip_getip (server, &server_ip)) < 0) {
      return code;
    }
  }

    /* fill in the server address */

#ifdef SOLARIS
  memset ((void *) &serverAddress, 0, sizeof (sockaddr_in));
  memcpy ((void *) &serverAddress.sin_addr, (void *) &server_ip, sizeof(UINT_32));
#else
  bzero ((char *) &serverAddress, sizeof(sockaddr_in));
  bcopy ((char *) &server_ip, (char *) &serverAddress.sin_addr, sizeof(UINT_32));
#endif
  serverAddress.sin_family = AF_INET;
  serverAddress.sin_port   = htons (port);

    /* mark the socket as nonblocking */

  mode = fcntl (sockfd, F_GETFL, 0);
  new_mode = mode | O_NONBLOCK;
  fcntl (sockfd, F_SETFL, new_mode);

    /* start the asynchronous connection */

  do 
  {
    code = connect (sockfd, (struct sockaddr *) &serverAddress, sizeof(sockaddr_in));
  } 
  while ((code < 0) && (errno == EINTR));

    /* wait for the connection to complete */

  if (code < 0) {
    if ((errno == EWOULDBLOCK) || (errno == EINPROGRESS)) {
      do 
      {
        FD_ZERO (&writeSet);
        FD_SET (sockfd, &writeSet);
        timeoutPtr = TIMERS::ticksToTimeval (stop, timeout);
        code = select (sockfd + 1, NULL, &writeSet, NULL, timeoutPtr);
      } 
      while ((code < 0) && (errno == EINTR));
    }

    if (code < 0) {
      code = TCPIP_UNABLE_TO_CONNECT;
    } else if (code == 0) {
      code = TCPIP_TIMEOUT;
    } else {
      code = 0;
    }
  }

    /* undo the mode change */

  fcntl (sockfd, F_SETFL, mode);
  return (code);
}

/* tcpip_unix_connect

   Purpose: Connect to a Unix domain socket

     Input: sockfd = socket descriptor
            name   = filename of the Unix domain socket

    Output: The procedure returns TCPIP_UNABLE_TO_CONNECT if it is unable to
	    connect sockfd to the Unix domain socket.  Otherwise the procedure
	    returns 0.
*/

int tcpip_unix_connect (int sockfd, char *name)
{
  unsigned len;
  int return_code;
  struct sockaddr_un unix_addr;

    /* determine the appropriate length */

#ifdef SUNLEN_EXISTS
  len = sizeof(unix_addr.sun_family) + sizeof(unix_addr.sun_len) + strlen(name) + 1;
#else
  len = sizeof(unix_addr.sun_family) + strlen(name);
#endif

    /* check the length */

  if (len >= sizeof(sockaddr_un)) {
    return TCPIP_UNABLE_TO_CONNECT;
  }

    /* fill in the sockaddr structure */

#ifdef SOLARIS
  memset ((void *) &unix_addr, 0, sizeof(sockaddr_un));
#else
  bzero ((char *) &unix_addr, sizeof(sockaddr_un));
#endif
#ifdef SUNLEN_EXISTS
  unix_addr.sun_len = len;
#endif
  unix_addr.sun_family = AF_UNIX;
  strcpy (unix_addr.sun_path, name);

    /* try to connect */

  do
  {
    return_code = connect (sockfd, (struct sockaddr *) &unix_addr, len);
  }
  while ((return_code < 0) && (errno == EINTR));

    /* error code */

  if (return_code < 0) {
    return TCPIP_UNABLE_TO_CONNECT;
  } else {
    return 0;
  }
}
 
/* tcpip_listen

   Purpose: Listen to a socket

     Input: sockfd = socket descriptor

    Output: The procedure returns TCPIP_UNABLE_TO_LISTEN on error.  Otherwise
            the procedure returns 0.
*/

int tcpip_listen (int sockfd)
{
  if (listen(sockfd, TCPIP_BACKLOG) < 0)
  {
    return TCPIP_UNABLE_TO_LISTEN;
  }
  else
  {
    return 0;
  }
}

/* tcpip_bind

   Purpose: Bind a socket to a local port

     Input: sockfd = socket descriptor
            port   = port number

    Output: The procedure returns TCPIP_UNABLE_TO_BIND if it can not bind to
            the port.  The procedure returns TCPIP_UNABLE_TO_GET_PORT if it
	    is unable to verify the port number of the bound port.  Otherwise
	    the procedure returns 0 and fills in *selected_port with the 
	    TCPIP/IP port of the socket.  The selected port will be equal to 
	    the requested port unless the requested port was TCPIP_ANY in 
	    which case the procedure lets the system choose a port.
*/

int tcpip_bind (int sockfd, int port, int *selected_port)
{
  struct sockaddr_in address;  
  int size = sizeof(sockaddr_in);

    /* bind the local address so thet a client can send to us */

#ifdef SOLARIS
  memset ((void *) &address, 0, sizeof(address));
#else
  bzero ((char *) &address, sizeof(address));
#endif
  address.sin_family      = AF_INET;
  address.sin_port        = htons (port);
  address.sin_addr.s_addr = htonl (INADDR_ANY);

  if (bind(sockfd, (struct sockaddr *) &address, size) < 0) {
    return TCPIP_UNABLE_TO_BIND;
  }

  if (getsockname(sockfd, (struct sockaddr *) &address, &size) < 0) {
    return TCPIP_UNABLE_TO_GET_PORT;
  }

  if (selected_port != NULL) {
    *selected_port = ntohs (address.sin_port);
  }
 
  return 0;
}

/* tcpip_unix_bind

   Purpose: Bind a Unix domain socket to a filename

     Input: sockfd = socket descriptor
            name   = filename  

    Output: The procedure returns TCPIP_UNABLE_TO_BIND if it is unable to
            bind the socket to the specified port.  Otherwise the procedure
            returns 0.
*/

int tcpip_unix_bind (int sockfd, char *name)
{
  unsigned len;
  struct sockaddr_un unix_addr;

    /* determine the appropriate length */

#ifdef SUNLEN_EXISTS
  len = sizeof(unix_addr.sun_family) + sizeof(unix_addr.sun_len) + strlen(name) + 1;
#else
  len = sizeof(unix_addr.sun_family) + strlen(name);
#endif

    /* check the length */

  if (len >= sizeof(unix_addr)) {
    return TCPIP_UNABLE_TO_BIND;
  }

    /* remove the filename if it exists */

  unlink (name);

    /* create the sockaddr structure */

#ifdef SOLARIS
  memset ((void *) &unix_addr, 0, sizeof(unix_addr));
#else
  bzero ((char *) &unix_addr, sizeof(unix_addr));
#endif
#ifdef SUNLEN_EXISTS
  unix_addr.sun_len = len;
#endif
  unix_addr.sun_family = AF_UNIX;
  strcpy (unix_addr.sun_path, name);

    /* bind to the filename */

  if (bind (sockfd, (struct sockaddr *) &unix_addr, len) < 0) {
    return TCPIP_UNABLE_TO_BIND;
  }

  return 0;
}

/* tcpip_setup

   Purpose: This function creates, binds and listens to a socket.

     Input: port = TCP/IP port for the socket

    Output: The procedure returns one of the error flags in tcpip.h on error.
            Otherwise the procedure fills in *selected__port with the TCP/IP
            port of the socket and returns the socket descriptor of the 
            socket.  The selected port will be equal to the requested port
            unless the desired port is TCPIP_ANY in which case the procedure 
            lets the system choose a port.
*/

int tcpip_setup (int port, int *selected_port)
{
  int sockfd;                  /* socket descriptor   */
  int return_code;             /* return code         */

    /* open the socket */

  if ((sockfd = tcpip_socket ()) < 0) {
    return sockfd;
  }

    /* bind the socket */

  if ((return_code = tcpip_bind (sockfd, port, selected_port)) < 0) {
    close (sockfd);
    return return_code;
  }

    /* listen at the sockt */

  if ((return_code = tcpip_listen (sockfd)) < 0) {
    close (sockfd);
    return return_code;
  }

  return sockfd;
}

/* tcpip_unix_setup

   Purpose: This function creates, binds and listens to a Unix domain socket.

     Input: name = filename

    Output: The procedure returns one of the error flags in tcpip.h on error.
            Otherwise the procedure returns the socket number.
*/

int tcpip_unix_setup (char *name)
{
  int sockfd;                  /* socket descriptor   */
  int return_code;             /* return code         */

    /* open the socket */

  if ((sockfd = tcpip_unix_socket ()) < 0) {
    return sockfd;
  }

    /* bind the socket */

  if ((return_code = tcpip_unix_bind (sockfd, name)) < 0) {
    close (sockfd);
    return return_code;
  }

    /* listen at the sockt */

  if ((return_code = tcpip_listen (sockfd)) < 0) {
    close (sockfd);
    return return_code;
  }

  return sockfd;
}

/* tcpip_accept

   Purpose: Accept a connection on a socket

     Input: sockfd = socket descriptor returned from tcpip_setup
 
    Output: The procedure returns TCPIP_UNABLE_TO_ACCEPT if it is unable
            to accept a connection.  Otherwise the procedure returns the
            NEW socket descriptor.
*/

int tcpip_accept (int sockfd)
{
  int flag = 1;                    /* flag for setsockopt        */
  int new_sockfd;                  /* new socket descriptor      */
  struct sockaddr_in address;      /* client address             */
  int size = sizeof(sockaddr_in);  /* size of the client address */

    /* accept a connection */

  do {
    new_sockfd = accept (sockfd, (struct sockaddr *) &address, &size);
  } while ((new_sockfd < 0) && (errno == EINTR));

    /* check for error */

  if (new_sockfd < 0) {
    return TCPIP_UNABLE_TO_ACCEPT;
  }

    /* make sure the TCP_NODELAY option is set for the new socket     */
    /* This should not be necessary but on some machines the accept() */
    /* function does not correctly propagate the TCP_NODELAY flag     */
    /* from sockfd to new_sockfd.                                     */

  setsockopt (new_sockfd, IPPROTO_TCP, TCP_NODELAY, (char *) &flag, sizeof(int));
  return new_sockfd;
}

/* tcpip_readn

   Purpose: This function is used in place of "read" when reading from a
            socket.

     Input: fd     = socket descriptor
            ptr    = buffer into which the characters are read
            nbytes = number of characters to read

    Output: The procedure returns an integer less than 0 on error.  
            Otherwise the procedure returns the number of characters
            read from the socket.
*/

int tcpip_readn (register int fd, register char *ptr, register size_t nbytes)
{
  register int nread;               /* number of bytes read         */   
  register size_t nleft = nbytes;   /* number of bytes left to read */

  while (nleft > 0) {

    do {
      nread = read (fd, ptr, nleft);
    } while ((nread < 0) && (errno == EINTR));

    if (nread < 0) {             /* ERROR */
      return (nread);
    } else if (nread == 0) {     /* EOF   */
      break;            
    }
 
    nleft -= nread;
    ptr   += nread;
  }

  return (nbytes - nleft);
}


/* tcpip_writen

   Purpose: This function is used in place of "write" when writing to a 
            socket.

     Input: fd     = file descriptor
            ptr    = buffer from which characters are written
            nbytes = number of bytes to write

    Output: The procedure returns an integer less than 0 on error. 
            The procedure returns 0 on EOF. 
            Otherwise the procedure returns the number of bytes written.
*/

int tcpip_writen (register int fd, register char *ptr, register size_t nbytes)
{
   register size_t nleft = nbytes;   /* number of bytes left to write  */
   register int nwritten;            /* number of bytes written        */
   
   nleft = nbytes;

   while (nleft > 0)
   {
     do
     {
       nwritten = write (fd, ptr, nleft);
     }
     while ((nwritten < 0) && (errno == EINTR));

     if (nwritten <= 0)         /* ERROR or EOF */
       return nwritten;

     nleft -= nwritten;
     ptr   += nwritten;
   }

   return (nbytes - nleft);
}

/* tcpip_error_to_string

   Purpose: Convert one of the error codes to a string

     Input: code = error code

    Output: The procedure returns a pointer to a STATIC string that describes
            the error code. 
*/

const char *tcpip_error_to_string (int code)
{
   switch (code)
   {
     case TCPIP_UNABLE_TO_CREATE:
       return "unable to open a socket";
     case TCPIP_UNABLE_TO_BIND:
       return "unable to bind sockfd to the port";
     case TCPIP_UNABLE_TO_LISTEN:
       return "unable to listen on sockfd "; 
     case TCPIP_UNABLE_TO_GET_PORT:
       return "unable to get the port number of sockfd";
     case TCPIP_UNABLE_TO_ACCEPT:
       return "unable to accept on sockfd";
     case TCPIP_UNABLE_TO_CONVERT_NAME:
       return "unable to convert server name to dotted IP";
     case TCPIP_UNABLE_TO_CONNECT:
       return "unable to connect sockfd to remote port";
     case TCPIP_TIMEOUT:
       return "timeout";
     default:
       return "unknown error";
    }
}

/* tcpip_read_* and tcpip_write_*
 
   Purpose: These procedures send and receive data on the socket.
*/

int tcpip_read_byte (int fd, UINT_8 *number)
{
  UINT_8 netChar;   /* char in network format */

  if (tcpip_readn (fd, (char *) &netChar, sizeof(UINT_8)) < (int) sizeof(UINT_8)) {
    return -1;
  }
 
  *number = netChar; 
  return 0;
}

int tcpip_read_short (int fd, UINT_16 *number)
{
  UINT_16 netShort;  /* short integer in network format */

  if (tcpip_readn (fd, (char *) &netShort, sizeof(UINT_16)) < (int) sizeof(UINT_16)) {
    return -1;
  }

  *number = ntohs (netShort); 
  return 0;
} 

int tcpip_read_long (int fd, UINT_32 *number)
{
  UINT_32 netLong;   /* long integer in network format */

  if (tcpip_readn (fd, (char *) &netLong, sizeof(UINT_32)) < (int) sizeof(UINT_32)) {
    return -1;
  }

  *number = ntohl (netLong);
  return 0;
}

char *tcpip_read_string (int fd)
{
  char *string;
  UINT_32 length;

  if (tcpip_read_long (fd, &length) < 0) {
    return NULL;
  }

  if (length == 0) {
    return NULL;
  }

  string = new char [length];

  if (tcpip_readn (fd, string, length) < (int) length) {
    delete string;
    return NULL;
  }

  string[length - 1] = '\0';
  return string;
}
 
int tcpip_write_byte (int fd, UINT_8 number)
{
  UINT_8 netChar = number;

  if (tcpip_writen (fd, (char *) &netChar, sizeof(UINT_8)) < 0)
    return -1;
 
  return 0;
}

int tcpip_write_short (int fd, UINT_16 number)
{
  UINT_16 netShort = htons (number);

  if (tcpip_writen (fd, (char *) &netShort, sizeof(UINT_16)) < 0)
    return -1;

  return 0;
}

int tcpip_write_long (int fd, UINT_32 number)
{
  UINT_32 netLong = htonl (number);

  if (tcpip_writen (fd, (char *) &netLong, sizeof(UINT_32)) < 0)
    return -1;

  return 0;
}

int tcpip_write_string (int fd, char *string)
{
  UINT_32 length = (string == NULL) ? 0 : strlen(string) + 1;

  if (tcpip_write_long (fd, length) < 0)
    return -1;

  if ((length > 0) && (tcpip_writen (fd, string, length) < 0))
    return -1;

  return 0;
}
