/* Agent Tcl
   Bob Gray
   28 Spetember 1995

   timers.cc

   This file implements a class that allows a process to define any number 
   of timers.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <sys/time.h>
#include <sys/times.h>
#include <assert.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "interrupt.h"
#include "timers.h"
#include "truefalse.h"

  /* static data members */

long   TIMERS::ticks      = 0;
int    TIMERS::timerMax   = 0;
int    TIMERS::timerCount = 0;
int    TIMERS::count      = 0;
TIMER *TIMERS::timerHeap  = NULL;

/* TIMER::TIMER

   Purpose: These procedures are the constructors for class TIMER.
*/

TIMER::TIMER (clock_t interval, TimerFunc function, TimerClientData data)
{
  TIMER::interval = interval;
  TIMER::stop     = interval + TIMERS::getWall ();
  TIMER::function = function;
  TIMER::data     = data;
}

TIMER::TIMER (const TIMER &timer)
{
  TIMER::interval = timer.interval;
  TIMER::stop     = timer.stop;
  TIMER::function = timer.function;
  TIMER::data     = timer.data;
}

TIMER::TIMER (void)
{
  interval = 0;
  stop     = 0;
  function = (TimerFunc) NULL;
  data     = (TimerClientData) NULL; 
}

TIMER &TIMER::operator= (TIMER &timer)
{
  interval = timer.interval;
  stop     = timer.stop;
  function = timer.function;
  data     = timer.data;
  return (*(this));
}

/* TIMERS::getTicks

   Purpose: Get the number of clock ticks per second

     Input: None

    Output: The procedure returns the number of clock ticks per second.
*/

clock_t TIMERS::getTicks (void)
{
  if (ticks == 0) {
    return (sysconf (_SC_CLK_TCK));
  } else {
    return (ticks);
  }
}

/* TIMERS::getWall

   Purpose: Get the total number of clock ticks that have elapsed since 
            system startup

     Input: None

    Output: The procedure returns the total number of clock ticks.
*/

clock_t TIMERS::getWall (void) 
{
#ifdef NO_GETTOD
  struct tms buffer;
#else
  long secs;
  long usecs;
  struct timeval buffer;
  static struct timeval startTime = {0, 0};
#endif

  if (ticks == 0) {
    ticks = sysconf (_SC_CLK_TCK);
  }

#ifdef NO_GETTOD

  return (times (&buffer));

#else

#ifdef HAVE_BSDGETTIMEOFDAY
#define gettimeofday BSDgettimeofday
#endif

  if (startTime.tv_sec == 0) {
    (void) gettimeofday (&startTime, (struct timezone *) NULL);
  }

  (void) gettimeofday (&buffer, (struct timezone *) NULL);

  secs  = buffer.tv_sec - startTime.tv_sec;
  usecs = buffer.tv_usec - startTime.tv_usec;   // O.K. if it's negative
  return (secs * ticks + usecs / (1000000 / ticks)); 

#endif
}
  
/* TIMERS::ticksToTimeval

   Purpose: Fill in a timeval structure with the time until a clock_t stop time

     Input: stop     = stop time (in clock ticks)
            duration = pointer to the timeval structure

    Output: The procedure returns NULL if stop is -1.  Otherwise the procedure
            fills in the timeval strucutre and returns its address.

      Note: The timeval structure will be filled in with zeros if the current
            time is past the given stop time.
*/

struct timeval *TIMERS::ticksToTimeval (clock_t stop, struct timeval &duration)
{
  clock_t current;
  clock_t interval;
  clock_t fraction;

    /* assertions */

  if (stop < 0) {
    return NULL;
  }

    /* finite */

  if (ticks == 0) {
    ticks = sysconf (_SC_CLK_TCK);
  }

  if ((current = getWall ()) >= stop) {
    duration.tv_sec  = 0;
    duration.tv_usec = 0;
  } else {
    interval         = stop - current;
    duration.tv_sec  = interval / ticks;
    fraction         = interval % ticks;
    duration.tv_usec = fraction * (1000000 / ticks) + fraction * (1000000 % ticks) / ticks;
  }
 
  return (&duration);
}

/* TIMERS::disposeTimers

   Purpose: Dispose of expired timers 

     Input: None

    Output: The procedure disposes of expired timers and calls the
            corresponding handlers.
 
      NOTE: The procedure assumes that SIGALRM is blocked.
*/

void TIMERS::disposeTimers (void)
{
  clock_t wall = 0;         /* wall time in clock ticks          */
  struct TIMER timer;	    /* timer that has expired            */
  struct itimerval itimer;  /* interval timer                    */
  clock_t interval;         /* ticks until next timer            */
  clock_t fraction;         /* ticks that make up partial second */

    /* dispose of expired timers */

  while ((timerCount > 0) && ((wall = getWall()) >= timerHeap[0].stop)) {

    remove (timer);
  
    if (timer.function != NULL) {
      (timer.function)(timer);
    }
  }

    /* set a SIGALRM for the next timer */

  if (timerCount > 0) {
    interval                   = timerHeap[0].stop - wall;
    itimer.it_value.tv_sec     = interval / ticks;
    fraction                   = interval % ticks;
    itimer.it_value.tv_usec    = fraction * (1000000 / ticks) + fraction * (1000000 % ticks) / ticks;
    itimer.it_interval.tv_sec  = 0;
    itimer.it_interval.tv_usec = 0;
    setitimer (ITIMER_REAL, &itimer, NULL);
  } 
}

/* TIMERS::bubble_up

   Purpose: Bubble an item up to its proper position in the heap

     Input: i = index of the item in the heap

    Output: The procedure bubbles the item up to its proper position.

      NOTE: The procedure assumes that SIGALRM is blocked.
*/

void TIMERS::bubble_up (int i)
{
  int child;           /* index of the child in the heap  */
  int parent;          /* index of the parent in the heap */

    /* we better have a valid heap element */

  assert (i < timerCount);

    /* bubble up the item */

  child  = i;
  parent = (child - 1) >> 1;

  while ((parent >= 0) && (timerHeap[child].stop < timerHeap[parent].stop)) {
    swap (child, parent);
    child  = parent;
    parent = (child - 1) >> 1;
  }
}

/* TIMERS::swap

   Purpose: Swap two elements of the heap

     Input: i = index of first element
            j = index of second element

    Output: The procedure swaps the two elements.

      NOTE: This procedure assumes that SIGALRM is blocked.
*/

void TIMERS::swap (int i, int j)
{
  TIMER temp   = timerHeap[i];
  timerHeap[i] = timerHeap[j];
  timerHeap[j] = temp;
}  

/* TIMERS::TIMERS

   Purpose: This procedure is the constructor for class TIMERS.
*/

TIMERS::TIMERS (void)
{
  sigset_t oldMask;    /* old interrupt mask      */
  sigset_t newMask;    /* new interrupt mask      */

    /* block SIGALRM */

  sigemptyset (&newMask);
  sigaddset   (&newMask, SIGALRM);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

    /* set up if this is the first instance */

  if (count == 0) {
    ticks      = sysconf (_SC_CLK_TCK);
    timerMax   = DEFAULT_TIMER_MAX;
    timerHeap  = new TIMER [timerMax];
    timerCount = 0; 
    install_signal_intr (SIGALRM, (void (*) (...)) sigalrm_handler, 0);
  }

  count += 1;

    /* unblock SIGALRM */

  sigprocmask (SIG_SETMASK, &oldMask, NULL);
}

/* TIMERS::~TIMERS

   Purpose: This procedure is the destructor for class TIMER.
*/

TIMERS::~TIMERS ()
{
  sigset_t oldMask;    /* old interrupt mask      */
  sigset_t newMask;    /* new interrupt mask      */

    /* block SIGALRM */

  sigemptyset (&newMask);
  sigaddset   (&newMask, SIGALRM);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

    /* cleanup if this is the last instance */

  count -= 1;

  if (count == 0) {
    delete timerHeap;
    install_signal_intr (SIGALRM, (void (*)(...)) SIG_IGN, 0);
  }

    /* unblock SIGALRM */

  sigprocmask (SIG_SETMASK, &oldMask, NULL);
}

/* TIMERS::add

   Purpose: Add a timer to the set of timers

     Input: timer = the timer

    Output: The procedure adds the timer to the set of timers.
*/

void TIMERS::add (TIMER timer)
{
  sigset_t oldMask;    /* old interrupt mask              */
  sigset_t newMask;    /* new interrupt mask              */
  TIMER *newHeap;      /* new timer                       */
  int newIndex;        /* index of the new heap element   */

     /* block SIGALRM */

  sigemptyset (&newMask);
  sigaddset   (&newMask, SIGALRM);
  sigprocmask (SIG_BLOCK, &newMask, &oldMask);

     /* make the heap bigger if necessary */

  if (timerCount == timerMax) {
    newHeap = new TIMER [timerMax << 1];
    
    for (int i = 0; i < timerMax; i++) {
      newHeap[i] = timerHeap[i];
    }

    delete timerHeap;
    timerMax = timerMax << 1;
    timerHeap = newHeap;
  }  

     /* add the timer to the heap and bubble up */

  newIndex            = timerCount;
  timerHeap[newIndex] = timer;
  timerCount         += 1;
  bubble_up (newIndex);

     /* dispose of any expired timers */

  disposeTimers ();
  
     /* unblock SIGALRM */

  sigprocmask (SIG_SETMASK, &oldMask, NULL);
}

/* TIMERS::remove

   Purpose: Remove the smallest timer from the heap

     Input: None

    Output: The procedure removes and returns the smallest timer. 

      NOTE: The procedure assumes that SIGALRM is blocked.
*/

void TIMERS::remove (TIMER &minTimer)
{
  int hole;      /* the hole that we are bubbling down */
  int left;      /* the left child of the hole         */
  int right;     /* the right child of the hole        */

    /* assert that there is one or more timers in the heap */

  assert (timerCount > 0);

    /* copy out the min element */

  minTimer = timerHeap[0];

    /* bubble the empty whole down */

  hole  = 0;
  left  = (hole << 1) + 1;
  right = (hole << 1) + 2;
 
  while (left < timerCount) {
    if ((right >= timerCount) || (timerHeap[left].stop < timerHeap[right].stop)) {
      swap (left, hole);
      hole = left;
    } else {
      swap (right, hole);
      hole = right;
    } 

    left  = (hole << 1) + 1;
    right = (hole << 1) + 2;
  }

    /* copy and bubble up in order to get a full tree again */

  timerCount -= 1;

  if (hole != timerCount) {
    timerHeap[hole] = timerHeap[timerCount];
    bubble_up (hole);
  }
}

/* sigalrm_handler

   Purpose: Handle SIGALRM signals

     Input: signo = SIGALRM

    Output: The procedure disposes of the expired timer.
*/

void sigalrm_handler (int signo)
{
    int oldErrno;

	/* dispose of timers (save and restore errno) */

    oldErrno = errno;
    TIMERS::disposeTimers ();
    errno = oldErrno;
}
