/* Agent Tcl
   Bob Gray
   11 July 1995

   transmit.cc

   This file implmenets the routines that send and receive messages.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef COSMO
#include <unistd.h>
#else
#include </usr/local/lib/gcc-lib/rs6000-ibm-aix3.2.5/2.6.3/include/unistd.h>
#endif
#include "agentId.h"
#include "message.h"
#include "my_alloc.h"
#include "my_strings.h"
#include "tcpip.h"
#include "transmit.h"
#include "truefalse.h"

/* breakout_long

   Purpose: Break a UINT_32 integer out of the incoming message

     Input: bp     = current position in the message buffer
            end_bp = last position in the message buffer

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns the new bp and sets value to the value of the UINT_32
            integer.
*/

inline char *breakout_long (char *bp, char *end_bp, UINT_32 &value)
{
    /* check bounds -- then get the value */

  if (bp + sizeof(UINT_32) > end_bp) {
    bp = NULL;
  } else {
    bp    = fast_src_memcpy ((char *) &value, bp, sizeof(UINT_32));
    value = ntohl (value);
  }

  return (bp);     
}

/* breakout_string

   Purpose: Break a string out of the incoming message

     Input: bp     = current position in the message buffer
            end_bp = last position in the message buffer
           
    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns the new bp and sets string to the position of the string
            within the message buffer.
*/

inline char *breakout_string (char *bp, char *end_bp, char *&string)
{
  UINT_32 length;   /* length of the string */

  if ((bp = breakout_long (bp, end_bp, length)) != NULL) {
    if (length == 0) {
      string = NULL;
    } else if (bp + length > end_bp) {
      bp = NULL;
    } else {
      string = bp;
      bp += length - 1;
      *bp = '\0';
      bp += 1;
    }
  }

  return (bp); 
} 

/* message_conn

   Purpose: Connect to the server

     Input: server    = name of the server
            server_ip = IP address of the server (or UNKNOWN_IP)
            stop      = wall time at which to issue a timeout (in clock ticks)
            async     = TRUE if we want SIGIO and SIGURG signals for the socket

    Output: The procedure returns -1 on error.  Otherwise the procedure
            returns a socket descriptor that is connected to the server.
*/

int message_conn (char *server, UINT_32 server_ip, clock_t stop, int async)
{
  int code;
  int sockfd;

    /* create the socket */

  if ((sockfd = tcpip_socket ()) < 0) {
    return MESSAGE_ERROR;
  } 
 
    /* turn on asynchronous I/O if desired */

  if (async) {
    if (tcpip_asynchronous (sockfd) < 0) {
      close (sockfd);
      return MESSAGE_ERROR;
    }
  }

    /* connect the socket */

  if ((code = tcpip_connect (sockfd, server, server_ip, SERVER_TCP_PORT, stop)) == TCPIP_TIMEOUT) { 
    close (sockfd);
    return MESSAGE_TIMEOUT;
  } else if (code < 0) {
    close (sockfd);
    return MESSAGE_ERROR;
  }

    /* return the socket descriptor */

  return (sockfd);
}

/* message_conn_and_send

   Purpose: Connect to the server over a TCP/IP socket, send a message and
            return the socket descriptor for further use.

     Input: server    = name of the server
            server-ip = IP address of the server (or UNKNOWN_IP) 
            message   = the message
            stop      = wall time at which to issue a timeout (in clock ticks)
            async     = TRUE if we want SIGIO and SIGURG signals for the socket

    Output: The procedure returns -1 on error.  Otherwise the procedure
            returns the socket descriptor of the connection over which
            the message was sent.
*/

int message_conn_and_send (char *server, UINT_32 server_ip, MESSAGE &message, clock_t stop, int async)
{
  int code;
  int sockfd;     

    /* connect the socket to the server */

  if ((sockfd = message_conn (server, server_ip, stop, async)) < 0) {
    return (sockfd);
  }

    /* send the message */

  if ((code = message_send (sockfd, message)) < 0) {
    close (sockfd);
    return (code);
  }

    /* return the sockfd */
 
  return (sockfd);
}

/* message_receive

   Purpose: Receive a message from a file descriptor

     Input: sockfd = socket descriptor
            types  = possible message types

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns a pointer to a dynamically allocated MESSAGE structure 
            that contains the transmitted message.   
*/

MESSAGE *message_receive (int sockfd, const MESSAGE_SET &set)
{
  UINT_32 j;
  ID_LIST *idList;
  NAME_LIST *nameList;
  char *string;
  int fd;
  UINT_32 size;   /* size of the message    */
  char *buffer;   /* buffer for the message */
  char *bp;
  char *end_bp;
  UINT_32 length;
  int i;                   
  UINT_8  flag;
  UINT_16 integer16;
  UINT_32 integer32;
  MESSAGE *message;

    /* get the size of the message */

  if ((tcpip_read_long (sockfd, &size) < 0) || (size == 0)) {
    return NULL;
  }
   
    /* read the whole message */

  buffer = new char [size];

  if (tcpip_readn (sockfd, buffer, size) < (int) size) {
    delete buffer;
    return NULL;
  }
 
    /* get the message flag */
  
  bp      = buffer;
  end_bp  = buffer + size;
  flag    = *bp++;

    /* check the flag */

  if (flag >= set.n) {
    delete buffer;
    return NULL;
  }

    /* read the message elements */
  
  message = new MESSAGE (set.messages[flag]);
  message -> buffer = buffer;

  for (i = 0; i < message -> n; i++) {

    if (message -> types[i] == MESG_TYPE_NAMES) {

      if ((bp = breakout_long (bp, end_bp, length)) == NULL) {
        delete message;
        return NULL;
      }

      nameList = new NAME_LIST (length);

      for (j = 0; j < length; j++) {
        if ((bp = breakout_string (bp, end_bp, string)) == NULL) {
          delete message;
          return NULL;
        }

        nameList -> fill (j, string);
      }

      message -> elements[i].string  = (char *) nameList;
      message -> elements[i].dealloc = TRUE;  

    } else if (message -> types[i] == MESG_TYPE_IDS) {

      if ((bp = breakout_long (bp, end_bp, length)) == NULL) {
        delete message;
        return NULL;
      }

      if (bp + length * sizeof(UINT_32) > end_bp) {
        delete message;
        return NULL;
      }

      idList = new ID_LIST (length);

      for (j = 0; j < length; j++) {
        bp = fast_src_memcpy ((char *) &integer32, bp, sizeof(UINT_32));
        idList -> ids[j] = ntohl (integer32);
      }

      message -> elements[i].string  = (char *) idList;
      message -> elements[i].dealloc = TRUE;  

    } else if (message -> types[i] == MESG_TYPE_BYTE) {

      if (bp + sizeof(UINT_8) > end_bp) {
        delete message;
        return NULL;
      }
 
      message -> elements[i].number = *bp;
      bp += sizeof (UINT_8);

    } else if (message -> types[i] == MESG_TYPE_SHORT) {

      if (bp + sizeof(UINT_16) > end_bp) {
        delete message;
        return NULL;
      }

      bp = fast_src_memcpy ((char *) &integer16, bp, sizeof(UINT_16));
      message -> elements[i].number = ntohs (integer16);

    } else if (message -> types[i] == MESG_TYPE_STR) {

      if ((bp = breakout_string (bp, end_bp, message -> elements[i].string)) == NULL) {
        delete message;
        return NULL;
      }

    } else if (message -> types[i] == MESG_TYPE_BINARY) {

      if ((bp = breakout_long (bp, end_bp, length)) == NULL) {
        delete message;
        return NULL;
      }

      if (bp + length > end_bp) {
        delete message;
        return NULL;
      }

        /* create the temporary file */

      message -> elements[i].string  = my_strcpy ("/tmp/cargoXXXXXX"); 
      message -> elements[i].dealloc = TRUE;

      if ((fd = mkstemp (message -> elements[i].string)) < 0) {
        delete message;
        return NULL;
      }

      fchmod (fd, 0600);

        /* copy the binary data into the file */

      if (length > 0) {
        if (tcpip_writen (fd, bp, length) < (int) length) {
          close (fd);
          delete message;
          return NULL; 
        }

        bp += length;
      }

      close (fd);
        
    } else { /* MESSAGE_TYPE_LONG or MESG_TYPE_IP */

      if (bp + sizeof(UINT_32) > end_bp) {
        delete message;
        return NULL;
      }

      bp = fast_src_memcpy ((char *) &integer32, bp, sizeof(UINT_32));

      if (message -> types[i] != MESG_TYPE_IP) {
        integer32 = ntohl (integer32);
      }

      message -> elements[i].number = integer32;
    }
  }

  message -> buffer = buffer;
  return message;
}

/* message_send

     Purpose: Write a message to a file descriptor

       Input: fd      = file descriptor
              message = the message

      Output: The procedure returns -1 on error.  Otherwise the procedure
              writes the message to the file descriptor and returns 0.
 
   ASSUMPTION: The procedure ASSUMES that there is at most one file per 
               message.
*/

struct TCPIP_VALUE
{
  UINT_8  net_byte;
  UINT_16 net_short;
  UINT_32 net_long;
  UINT_32 host_long;
};

int message_send (int sockfd, MESSAGE &message)
{
  int nread;                   /* number of bytes read from a file    */
  int fd = 0;                  /* file descriptor                     */
  char *fileBp;                /* pointer into the file buffer        */
  UINT_32 fileSize = 0;        /* size of the file                    */
  char *newBuffer = NULL;      /* used when expanding the file buffer */
  char *fileBuffer = NULL;     /* file buffer                         */ 
  UINT_32 bufferSize = 1024;   /* size of the file buffer             */

  register int i;
  register int j;
  char *bp;                      
  char *buffer;
  UINT_32 size = 0;
  UINT_32 net_size;
  UINT_32 net_long;
  ID_LIST *idList;
  NAME_LIST *nameList;
  int return_code = 0;
  static TCPIP_VALUE values[MAX_ELEMENTS];    /* buffers for elements */

    /* calculate the size of the buffer */

  size += sizeof(UINT_8);

  for (i = 0; i < message.n; i++) {

    if (message.types[i] == MESG_TYPE_BYTE) {

      values[i].net_byte = message.elements[i].number;
      size += sizeof (UINT_8);

    } else if (message.types[i] == MESG_TYPE_SHORT) {

      values[i].net_short = htons (message.elements[i].number);
      size += sizeof (UINT_16);

    } else if (message.types[i] == MESG_TYPE_LONG) {

      values[i].net_long = htonl (message.elements[i].number);
      size += sizeof (UINT_32);

    } else if (message.types[i] == MESG_TYPE_IP) {

      values[i].net_long = message.elements[i].number;
      size += sizeof (UINT_32);

    } else if (message.types[i] == MESG_TYPE_NAMES) {

      nameList = (NAME_LIST *) message.elements[i].string;
      values[i].host_long = nameList -> count;
      values[i].net_long  = htonl (values[i].host_long);

      size += (nameList -> count) * sizeof(UINT_32) + sizeof(UINT_32);

      for (j = 0; j < nameList -> count; j++) {
        size += nameList -> lengths[j];
      }

    } else if (message.types[i] == MESG_TYPE_IDS) {

      idList = (ID_LIST *) message.elements[i].string;
      values[i].host_long = idList -> count;
      values[i].net_long  = htonl (values[i].host_long);
      size += values[i].host_long * sizeof(UINT_32) + sizeof(UINT_32);

    } else if (message.types[i] == MESG_TYPE_BINARY) {
     
        /* get the fd */
 
      fd = (int) message.elements[i].number;

        /* copy into the fileBuffer */
  
      fileSize   = 0; 
      fileBuffer = new char [bufferSize];
      fileBp     = fileBuffer;

      while ((nread = tcpip_readn (fd, fileBp, bufferSize - fileSize)) > 0) {

          /* increment amount read */

        fileSize += nread;
        fileBp   += nread;

          /* increase buffer size if necessary */

        if (fileSize >= bufferSize) {
          bufferSize = bufferSize << 1;
          newBuffer = new char [bufferSize];
          fast_memcpy (newBuffer, fileBuffer, fileSize);
          delete fileBuffer;
          fileBuffer = newBuffer;
          fileBp = fileBuffer + fileSize;
        }
      }   
     
        /* bail on error */
 
      if (nread < 0) {
        delete fileBuffer;
        return MESSAGE_FILE;
      }
 
        /* increment the total size */

      values[i].net_long  = htonl (fileSize);
      size += fileSize + sizeof(UINT_32);

    } else if (message.elements[i].string == NULL) {

      values[i].host_long = 0;
      values[i].net_long  = 0;
      size += sizeof (UINT_32);

    } else {

      values[i].host_long = strlen(message.elements[i].string) + 1;
      values[i].net_long  = htonl (values[i].host_long);
      size += values[i].host_long;
      size += sizeof (UINT_32);
    }
  }

    /* copy into the buffer */

  net_size = htonl (size);       /* length excluding length field */
  size += sizeof(UINT_32);       /* room for the length           */

    /* allocate the buffer */

  buffer = new char [size];
  bp     = buffer;

    /* put in the length and the flag */

  bp = fast_memcpy (bp, (char *) &net_size, sizeof(UINT_32));
  bp = fast_memcpy (bp, (char *) &message.flag, sizeof(UINT_8));

    /* put in each component */

  for (i = 0; i < message.n; i++) {

    if (message.types[i] == MESG_TYPE_BYTE) {

      bp = fast_memcpy (bp, (char *) &values[i].net_byte, sizeof(UINT_8));

    } else if (message.types[i] == MESG_TYPE_SHORT) {

      bp = fast_memcpy (bp, (char *) &values[i].net_short, sizeof(UINT_16));

    } else if (message.types[i] == MESG_TYPE_STR) {

      bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));
      bp = fast_memcpy (bp, message.elements[i].string, values[i].host_long);

    } else if (message.types[i] == MESG_TYPE_LONG) {

      bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));

    } else if (message.types[i] == MESG_TYPE_IP) {
  
      bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));

    } else if (message.types[i] == MESG_TYPE_NAMES) {

      bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));

      nameList = (NAME_LIST *) message.elements[i].string;

      for (j = 0; j < nameList -> count; j++) {
        net_long = htonl (nameList -> lengths[j]);
        bp = fast_memcpy (bp, (char *) &net_long, sizeof(UINT_32));
        bp = fast_memcpy (bp, nameList -> names[j], nameList -> lengths[j]);
      }

    } else if (message.types[i] == MESG_TYPE_IDS) {
 
      bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));

      idList = (ID_LIST *) message.elements[i].string;

      for (j = 0; j < idList -> count; j++) {
        net_long = htonl (idList -> ids[j]);
        bp = fast_memcpy (bp, (char *) &net_long, sizeof(UINT_32));
      }

    } else {     /* message.types[i] == MESG_TYPE_BINARY */

      bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));
      bp = fast_memcpy (bp, fileBuffer, fileSize);
      delete fileBuffer;
    }
  }

    /* write the buffer */

  if (tcpip_writen (sockfd, buffer, size) < 0) {
    return_code = MESSAGE_ERROR;
  }

  delete buffer;
  return (return_code);
} 
