/* Agent Tcl
   Bob Gray
   10 January 1996

   tclAgent.cc

   This file reimplements class AGENT for use in the Tcl/Tk interpreters.

   Copyright (c) 1995-1999, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "suppDBuffer.h"
#include "genServerInterface.h"
#include "tclFactory.h"

/* ------------------------------------------------------------ */

#include "genEncrypt.h"
#include "tclAgent.h"
#include "tclLocation.h"
#include "tclMasks.h"
#include "tclMeeting.h"
#include "tclRestrict.h"
#include "tclSecurity.h"
#include "tclUtility.h"

    /* static member data */

POINTER_MAP *AGENT_TCL::interpMap 
	= NULL;

IncomingCallback_TCL *AGENT_TCL::incomingCallback
	= NULL;

/* IncomingCallback_TCL::IncomingCallback_TCL and IncomingCallback_TCL::~IncomingCallback_TCL

   Purpose: These procedures are the constructor and destructor for class
            IncomingCallback_TCL.
*/

IncomingCallback_TCL::IncomingCallback_TCL (void):
    IncomingCallback ()
{
    incomingAsync = Tcl_AsyncCreate (agentAsyncIncoming, (ClientData) this);
}

IncomingCallback_TCL::~IncomingCallback_TCL () 
{
    Tcl_AsyncDelete (incomingAsync);
}

/* AGENT_TCL::AGENT_TCL

   Purpose: These procedures are the constructors for class AGENT_TCL.
*/

AGENT_TCL::AGENT_TCL (const Version &version):
    AGENT (version, AgentFactory_Tcl(), AGENT::e_INCOMING_AGENT)
{
#ifdef FIX_LATER
	/* this initialization should be moved to avoid the race condition */
#endif

    if (incomingCallback == NULL) {
	incomingCallback = new IncomingCallback_TCL ();
    }

	/* Tcl can support both events and interrupts */

    masks -> turnOnInterrupts();
    masks -> turnOnEvents ();

	/* interpreters */

    masterInterp = NULL;
    interp       = NULL;
}

AGENT_TCL::AGENT_TCL (
        Tcl_Interp *newInterp, 
	Tcl_Interp *newMasterInterp, 
	const Version &version
    ):
    AGENT (version, AgentFactory_Tcl(), AGENT::e_LOCAL_AGENT)
{
	/* assertions on the parameters */

    assert (newInterp != NULL);

#ifdef FIX_LATER
	/* this initilization should be moved to avoid the race condition */
#endif

    if (incomingCallback == NULL) {
	incomingCallback = new IncomingCallback_TCL ();
    }

	/* Tcl can support both events and interrupts */

    masks -> turnOnInterrupts();
    masks -> turnOnEvents ();

	/* interpreters */

    interp       = newInterp;
    masterInterp = newMasterInterp;  

	/* setup the interpreters */

    (void) setupInterps ();
}

/* AGENT_TCL::setupInterps

   Purpose: Set the Tcl interpreters

     Input: None 

    Output: The procedure returns one of the following codes.

	    AGENT_OK        = setup was successful
	    AGENT_BAD_STATE = incoming state information was incorrect
*/

int AGENT_TCL::setupInterps (void)
{
	/* add the interpreter to the map */

    if (interpMap == NULL) {
	interpMap = new POINTER_MAP ();
    }

    interpMap -> add (interp, this);

    if (masterInterp != NULL) {
	interpMap -> add (masterInterp, this);
    }

	/* have each component set up the necessary interpreter state */

    ((LOCATION_TCL *) location) -> setInterp (interp, masterInterp);
    ((SECURITY_TCL *) security) -> setInterp (interp, masterInterp);
    ((MEETINGS_TCL *) meetings) -> setInterp (interp, masterInterp);
    ((RESTRICT_TCL *) restrict) -> setInterp (interp, masterInterp);
    ((MASKS_TCL    *) masks   ) -> setInterp (interp, masterInterp);

	/* done */

    return AGENT_OK;
}


/* AGENT_TCL::agentRun

   Purpose: This procedure runs a restored agent.

     Input: None

    Output: The procedure returns a standard Tcl result code.
*/

int AGENT_TCL::agentRun (void)
{
    int agentRc;
    int tclRc;
 
	/* assume success */

    agentRc = AGENT_OK;

	/* run the agent */

#ifdef TCL8

    assert (m_type == ServerInterface::e_SUB_SCRIPT);

    tclRc = Tcl_Eval (interp, m_state.value());

    if ((tclRc != TCL_REWIND) && (tclRc != TCL_OK)) {
	agentRc = AGENT_PROGRAM_ERROR;
    }

#else


    if (m_type == ServerInterface::e_SUB_SCRIPT) {

	Tcl_AllowStateCapture (interp);
	tclRc = Tcl_Eval (interp, m_state.value());

        if ((tclRc != TCL_REWIND) && (tclRc != TCL_OK)) {
	    agentRc = AGENT_PROGRAM_ERROR;
	}

    } else {

	tclRc = Tcl_EvalStack (interp, STACK_BOTTOM);

        if ((tclRc != TCL_REWIND) && (tclRc != TCL_OK)) {
	    agentRc = AGENT_PROGRAM_ERROR;
	}
    }

#endif

    return (agentRc);
}

/* AGENT_TCL::~AGENT_TCL

   Purpose: This procedure is the destructor for class AGENT_TCL.
*/

AGENT_TCL::~AGENT_TCL (void)
{
	/* remove the interpeters from the interpreter map */

    if (interpMap != NULL) {

	interpMap -> remove (interp);

	if (masterInterp != NULL) {
	    interpMap -> remove (masterInterp);
	}
    }
}

/* AGENT_TCL::restoreState

   Purpose: Restore the state of an incoming agent

     Input: type   = SUB_SCRIPT, SUB_FORK or SUB_JUMP 
		     (ServerInterface::SubmissionTypes)

	    state  = state information
		     (const DynamicBuffer&)

    Output: The procedure returns one of the following codes.

	    AGENT_OK        = success
            AGENT_BAD_STATE = unable to restore the state information
*/

int AGENT_TCL::restoreState (ServerInterface::SubmissionTypes type, const DynamicBuffer &state)
{
	/* make sure that state is not empty */

    if (state.isEmpty()) {
	return (AGENT_BAD_STATE);
    }

	/* remember the type of the incoming agent and the state */

    m_type  = type;
    m_state = state;

	/* make sure that the state ends in a NULL character */

    char *statePtr = m_state.value();
    statePtr[m_state.size() - 1] = '\0';
	
#ifdef TCL8

	/* Tcl 8 supports NEW agents only */

    if (type != ServerInterface::e_SUB_SCRIPT) {
	abort_with_message ("Tcl 8.0 interpreter does not support jump or fork");
    }

	/* create the interpreters */

    masterInterp = Tcl_CreateInterp();
    interp       = Tcl_CreateSlave (masterInterp, "slave", 1);

	/* setup the interpreters */

    if (setupInterps() != AGENT_OK) {
	Tcl_DeleteInterp (interp);
	Tcl_DeleteInterp (masterInterp);
	interp = masterInterp = (Tcl_Interp *) NULL;
	return (AGENT_BAD_STATE);
    }

#else

	/* create the interpreters and restore the state */

    if (type == ServerInterface::e_SUB_SCRIPT) {

	interp = Tcl_CreateInterp();

    } else {

        Tcl_DString error;
	Tcl_DStringInit (&error);
	interp = Tcl_CreateAndLoadInterp (m_state.value(), &error);
	Tcl_DStringFree (&error);
	m_state.empty();    // do not need to remember the state separately

	if (interp == NULL) {
	    return (AGENT_BAD_STATE);
	}
    }

	/* save the result string -- Note that (1) we do not need to save */
	/* the result string if this is a NEW child agent and (2) we use  */
	/* "malloc" since Tcl will later use "free" to free the saved     */
	/* result string.                                                 */

    char *result = NULL;
   
    if (type != ServerInterface::e_SUB_SCRIPT) {
        if (interp -> result == NULL) {
	    result = NULL;
        } else {
	    result = (char *) malloc ((unsigned) strlen(interp -> result) + 1);
	    strcpy (result, interp -> result);
        }
    }

	/* create the master interpreter */

    masterInterp = Tcl_CreateInterp ();

	/* interpreter is a slave of the master */

    if (Tcl_SetSlave (masterInterp, "slave", interp, 1) != TCL_OK) {
	Tcl_DeleteInterp (interp);
	Tcl_DeleteInterp (masterInterp);
	interp = masterInterp = (Tcl_Interp *) NULL;
	return (AGENT_BAD_STATE);
    }

	/* everything is non-interactive */

    Tcl_SetInteractive (masterInterp, 0);
    Tcl_SetInteractive (interp, 0);

	/* setup the interpreters */

    if (setupInterps() != AGENT_OK) {
	Tcl_DeleteInterp (interp);
	Tcl_DeleteInterp (masterInterp);
	interp = masterInterp = (Tcl_Interp *) NULL;
	return (AGENT_BAD_STATE);
    }

	/* restore the result string if this is NOT a new child agent */

    if (type != ServerInterface::e_SUB_SCRIPT) {
        Tcl_SetResult (interp, result, TCL_DYNAMIC);
    }

#endif

    return (AGENT_OK);     
} 

/* AGENT_TCL::captureState

   Purpose: Capture the current state of the agent

     Input: jumpType = e_FORKING or e_JUMPING
		      (AGENT::JumpType)

    Output: The procedure returns AGENT_BAD_STATE and sets the Tcl interpreter 
	    result to an appropriate error message if it is unable to capture 
	    the state.  Otherwise the procedure returns AGENT_OK and sets
	    "stateBuffer" to the state information.
*/

int AGENT_TCL::captureState (AGENT::JumpType jumpType, DynamicBuffer &stateBuffer)
{
#ifdef TCL8
	
    abort_with_message ("Tcl 8.0 does not support state capture");
    return (AGENT_BAD_STATE);

#else

    char *state;
    UINT_32 length;
    Tcl_DString error;
    Tcl_DString *dstate;

	/* capture the state */

    Tcl_DStringInit (&error);
    Tcl_SetResult   (interp, (char *) ((jumpType == e_FORKING) ? "CHILD" : "JUMPED"), TCL_STATIC);   

    if ((dstate = Tcl_SaveState (interp, &error)) == NULL) { 
	Tcl_DStringResult (interp, &error);
	Tcl_DStringFree (&error);
	return (AGENT_BAD_STATE);
    }

    Tcl_DStringFree (&error);
    Tcl_ResetResult (interp);

	/* extract the string out of the Tcl_DString */

    length = Tcl_DStringLength (dstate) + 1; 
    state  = Tcl_DStringExtract (dstate);

	/* construct the DynamicBuffer -- it assumes ownership of "state" */
	/* so state should NOT be deleted here                            */
   
    stateBuffer = DynamicBuffer (length, state, BufferOwnership::e_TAKE);
    state = NULL;

	/* return the string */

    Tcl_DStringFree (dstate);
    delete (dstate);
    return (AGENT_OK);

#endif
}

/* agentAsyncIncoming

   Purpose: This procedure is called when a SIGIO interrupt occurs.

     Input: clientData = the IncomingCallback_TCL structure
			 (class IncomingCallback_TCL *)

            interp     = the Tcl interpreter
			 (struct Tcl_Interp *)

            code       = the Tcl result code
			 (int)

    Output: The procedure receives any available items.
*/

int agentAsyncIncoming (ClientData clientData, Tcl_Interp *, int code)
{
    IncomingCallback_TCL *callback;

	/* callback */

    callback = (IncomingCallback_TCL *) clientData;
    assert (callback != NULL);

        /* check for incoming communication */

    callback -> handleIncomingItems();

	/* return the same code that was passed in */

    return (code);
}
