/* Agent Tcl
   Bob Gray
   10 January 1996

   tclLocation.cc

   This file reimplements class LOCATION for use in the Tcl/Tk interpreters.

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "mesgTcpip.h"		// tcpip_IPToString
#include "mesgMachId.h"	// MachineId
#include "suppDString.h"		// DynamicString
#include "tclMakeLists.h"	// Agent_IdToTclList

/* ------------------------------------------------------------------ */

#include "tcl.h"
#include "genConversion.h"
#include "tclLocation.h"
#include "truefalse.h"

static char *agentTrace (ClientData clientData, Tcl_Interp *interp, char *name1, char *name2, int flags);

/* LOCATION_TCL::LOCATION_TCL

   Purpose: This procedure is the constructor for class LOCATION_TCL.

     Input: p_agent = agent associated with this LOCATION_TCL instance
		      (class AGENT_TCL *)
*/

LOCATION_TCL::LOCATION_TCL (AGENT_TCL *p_agent): LOCATION (p_agent)
{
    interp       = NULL;
    masterInterp = NULL;
    modified     = FALSE;
}

/* LOCATION_TCL::setInterp

   Purpose: Set the Tcl interpreter

     Input: interp       = the Tcl interpreter associated with this instance
		           (struct Tcl_Interp *)

	    masterInterp = the first interpreter's master
			   (struct Tcl_Interp *)
*/

void LOCATION_TCL::setInterp (Tcl_Interp *newInterp, Tcl_Interp *newMasterInterp)
{
    int flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* assertions on the parameters */

    assert (interp == NULL);
    assert (masterInterp == NULL);
    assert (newInterp != NULL);

	/* remember the interpreter */

    interp       = newInterp;
    masterInterp = newMasterInterp;  

	/* load the Tcl array and turn on the traces */
 
    refresh ();

    Tcl_TraceVar2 (interp, "agent", NULL, flags, agentTrace, (ClientData) this);  
#ifndef TCL8
    Tcl_GlueVar2 (interp, "agent", NULL, TCL_GLOBAL_ONLY);
#endif

    if (masterInterp != NULL) {
	Tcl_TraceVar2 (masterInterp, "agent", NULL, flags, agentTrace, (ClientData) this);
#ifndef TCL8
	Tcl_GlueVar2 (masterInterp, "agent", NULL, TCL_GLOBAL_ONLY);
#endif
    }
}

/* LOCATION_TCL::~LOCATION_TCL

   Purpose: This procedure is the destructor for class LOCATION_TCL.
*/

LOCATION_TCL::~LOCATION_TCL ()
{
    int flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* turn off the variable trace */

    Tcl_UntraceVar2 (interp, "agent", (char *) NULL, flags, agentTrace, (ClientData) this);
    Tcl_UnsetVar2 (interp, "agent", (char *) NULL, TCL_GLOBAL_ONLY);

    if (masterInterp != NULL) {
	Tcl_UntraceVar2 (masterInterp, "agent", (char *) NULL, flags, agentTrace, (ClientData) this);
	Tcl_UnsetVar2 (masterInterp, "agent", (char *) NULL, TCL_GLOBAL_ONLY);
    }
}

/* LOCATION_TCL::refreshVariableString and LOCATION_TCL::refreshVariableObj

   Purpose: Reload an element of the "agent" array

     Input: name        = name of the array element
		          (char *)

            value       = new value for the array element
		          (char *)

		          OR

	    valueObject = new value for the array element
                          (Tcl_Obj *)

    Output: The procedure reloads the element. 
*/

void LOCATION_TCL::refreshVariableString (char *name, char *value)
{
	/* allow modification */

    modified = TRUE;

	/* do the agent interpreter */

    Tcl_SetVar2 (interp, (char *) "agent", name, (value == NULL) ? (char *) "" : value, TCL_GLOBAL_ONLY);

	/* do the master interpreter */

    if (masterInterp != NULL) {
	Tcl_SetVar2 (masterInterp, (char *) "agent", name, (value == NULL) ? (char *) "" : value, TCL_GLOBAL_ONLY);
    }

	/* disallow modification */

    modified = FALSE;
}

#ifdef TCL8

void LOCATION_TCL::refreshVariableObj (char *name, Tcl_Obj *valueObject)
{
    static Tcl_Obj *arrayNameObject = (Tcl_Obj *) NULL;

	// initialize the array name if necessary

    if (arrayNameObject == NULL) {
	arrayNameObject = Tcl_NewStringObj ("agent", -1);
	Tcl_IncrRefCount (arrayNameObject);
    }

	// construct the element-name object

#ifdef FIX_LATER
	/* 
	 * If we wanted to be really (overly?) efficient, we could maintain
	 * a static array of element-name objects, thus constructing each
	 * element-name object only once.
	 */
#endif

   Tcl_Obj *elementNameObject = Tcl_NewStringObj (name, -1);

	// allow modification

    modified = TRUE;

	// set the array element in the interpreter

    Tcl_ObjSetVar2 (
	interp, arrayNameObject, elementNameObject, valueObject, 
	TCL_GLOBAL_ONLY);

	/* do the master interpreter */

    if (masterInterp != NULL) {
	Tcl_ObjSetVar2 (
	    masterInterp, arrayNameObject, elementNameObject, valueObject,
	    TCL_GLOBAL_ONLY);
    }

	// disallow modification

    modified = FALSE;
}

#endif

/* LOCATION_TCL::refresh_version

   Purpose: Refresh the version section of the "agent" array

     Input: None

    Output: The procedure refreshes the version section of the "agent" array.
*/

void LOCATION_TCL::refresh_version (void)
{
    char temp[32];
    Version version;

	/* reload */

    version = agent -> getCoreVersion ();
    sprintf (temp, "%d %d %s", version.m_major, version.m_minor, (version.m_type == Version::e_ALPHA) ? "alpha" : (version.m_type == Version::e_BETA) ?  "beta" : "production");
    refreshVariableString ("core-version", temp); 

    version = agent -> getInterpVersion ();
    sprintf (temp, "%d %d %s", version.m_major, version.m_minor, (version.m_type == Version::e_ALPHA) ? "alpha" : (version.m_type == Version::e_BETA) ?  "beta" : "production");
    refreshVariableString ("version", temp); 
    refreshVariableString ("language", version.m_name.value());
}

/* LOCATION_TCL::refresh

   Purpose: Reload the "agent" array

     Input: None

    Output: The procedure reloads the agent array.
*/

void LOCATION_TCL::refresh (void)
{
    char *string;
    char temp[32];

	/* reload */

    if (!(isRegistered())) {

	refreshVariableString ("root-server"    , (char *) NULL);
	refreshVariableString ("root-ip"        , (char *) NULL);
	refreshVariableString ("root-name"      , (char *) NULL);
	refreshVariableString ("root-id"        , (char *) NULL);
	refreshVariableString ("root"           , (char *) NULL);
        refreshVariableString ("roothome-server", (char *) NULL);
        refreshVariableString ("roothome-ip"    , (char *) NULL);
        refreshVariableString ("roothome-name"  , (char *) NULL);
        refreshVariableString ("roothome-id"    , (char *) NULL);
        refreshVariableString ("roothome"       , (char *) NULL);
	refreshVariableString ("local-server"   , (char *) NULL);
	refreshVariableString ("local-ip"       , (char *) NULL);
	refreshVariableString ("local-name"     , (char *) NULL);
	refreshVariableString ("local-id"       , (char *) NULL);
	refreshVariableString ("local"          , (char *) NULL);
	refreshVariableString ("home-server"    , (char *) NULL);
	refreshVariableString ("home-ip"        , (char *) NULL);
	refreshVariableString ("home-name"      , (char *) NULL);
	refreshVariableString ("home-id"        , (char *) NULL);
	refreshVariableString ("home"           , (char *) NULL);

    } else {

	refreshVariableString ("root-server", rootId.getServerName().value());
        refreshVariableString ("roothome-server", rootHomeId.getServerName().value());
	refreshVariableString ("home-server", homeId.getServerName().value());
	refreshVariableString ("local-server", localId.getServerName().value());
	refreshVariableString ("root-name", rootId.getName().value());
        refreshVariableString ("roothome-name", rootHomeId.getName().value());
	refreshVariableString ("home-name", homeId.getName().value());
	refreshVariableString ("local-name", localId.getName().value());
    
	sprintf (temp, "%u", rootId.getId()); 
	refreshVariableString ("root-id", temp);

        sprintf (temp, "%u", rootHomeId.getId());
 	refreshVariableString ("roothome-id", temp);

	sprintf (temp, "%u", homeId.getId());
	refreshVariableString ("home-id", temp);

	sprintf (temp, "%u", localId.getId()); 
	refreshVariableString ("local-id", temp);

        string = tcpip_IPToString (rootId.getServerIp());
	refreshVariableString ("root-ip", string);
	delete string;

        string = tcpip_IPToString (rootHomeId.getServerIp());
	refreshVariableString ("roothome-ip", string);
	delete [] string;

	string = tcpip_IPToString (homeId.getServerIp());
	refreshVariableString ("home-ip", string);
	delete [] string;

        string = tcpip_IPToString (localId.getServerIp());
	refreshVariableString ("local-ip", string);
	delete [] string;

	Tcl_DString dstring;
	Tcl_DStringInit (&dstring);
	Agent_IdToTclList (&rootId, &dstring);
	refreshVariableString ("root", Tcl_DStringValue(&dstring));
	Tcl_DStringFree (&dstring);

	Tcl_DStringInit (&dstring);
	Agent_IdToTclList (&rootHomeId, &dstring);
	refreshVariableString ("roothome", Tcl_DStringValue(&dstring));
	Tcl_DStringFree (&dstring);

	Tcl_DStringInit (&dstring);
	Agent_IdToTclList (&homeId, &dstring);
	refreshVariableString ("home", Tcl_DStringValue(&dstring));
	Tcl_DStringFree (&dstring);

#ifdef TCL8

	Tcl_Obj *listObject = Agent_IdToTclListObj (&localId);
	refreshVariableObj ("local", listObject);

#else

	Tcl_DStringInit (&dstring);
	Agent_IdToTclList (&localId, &dstring);
	refreshVariableString ("local", Tcl_DStringValue(&dstring));
	Tcl_DStringFree (&dstring);

#endif
    }

	/* reload the temp directory, actual machine and the registration flags */

    refreshVariableString ("temp-directory", tempDirectory.value());
    refreshVariableString ("actual-server", machine.getName().value());

    string = tcpip_IPToString (machine.getIp());
    refreshVariableString ("actual-ip", string);
    delete [] string;

    sprintf (temp, "%d", (registered == AGENT::e_REGISTERED) ? 1 : 0);
    refreshVariableString ("registered", temp);

    sprintf (temp, "%d", (root == AGENT::e_ROOT_AGENT) ? 1 : 0);
    refreshVariableString ("isRoot", temp);

    sprintf (temp, "%d", (home == AGENT::e_AT_HOME) ? 1 : 0);
    refreshVariableString ("isHome", temp);

    sprintf (temp, "%d", (server == AGENT::e_INCOMING_AGENT) ? 1 : 0);
    refreshVariableString ("server", temp);

    sprintf (temp, "%d", TCL_MAX_PREC);
    refreshVariableString ("maxprecision", temp);

	/* refresh the version information */

    refresh_version ();
}

/* LOCATION_TCL::changeNotification

   Purpose: Callback for when the agent's location information has changed
*/

void LOCATION_TCL::changeNotification (void)
{
    refresh ();
}

/* agentTrace

   Purpose: This procedure traces the "agent" array and prevents writes and
	    unsets.
*/

static char *agentTrace (ClientData clientData, Tcl_Interp *interp, char *, char *, int flags)
{
    int tflags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* LOCATION instance */

    LOCATION_TCL *location = (LOCATION_TCL *) clientData;
    assert (location != NULL);

	/* allow the operation if modified flag is SET */

    if (location -> allowModification()) {
      return ((char *) NULL);
    }

	/* restore original values on a write */

    if (flags & TCL_TRACE_WRITES) {
	Tcl_UnsetVar2 (interp, "agent", NULL, TCL_GLOBAL_ONLY);
	location -> refresh ();
	return "can not overwrite agent identification";
    }

	/* otherwise we are trying to unset                 */

    if (!(flags & TCL_INTERP_DESTROYED)) {

	location -> refresh ();

	if (flags & TCL_TRACE_DESTROYED) {
	    Tcl_TraceVar2 (interp, "agent", NULL, tflags, agentTrace, (ClientData) location);  
	}

#ifndef TCL8
	Tcl_GlueVar2 (interp, "agent", NULL, TCL_GLOBAL_ONLY);
#endif
	return "can not unset agent identification";
    }

    return ((char *) NULL);
}
