/* Agent Tcl
   Bob Gray
   10 January 1996

   tclMakeLists.cc

   This file implements various conversion routines.

   Copyright (c) 1995-1999, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "mesgTcpip.h"		// tpcip_iptostring
#include "agentAgentId.h"	// AgentId
#include "agentSecure.h"
#include "tcl.h"
#include "tclMakeLists.h"

/* Agent_IdToTclList

   Purpose: Convert an AgentId to a Tcl list

     Input: agentId = the agent identification
                      (AgentId *)
  
    Output: The four parts of the agent identification are appended to
            dstring (Tcl_DString *), each one as a list element.
*/ 

void Agent_IdToTclList (AgentId *agentId, Tcl_DString *dstring)
{
	// assertions

    assert (agentId != NULL);
    assert (dstring != NULL);

	// server name, server ip, agent name and agent id

    Tcl_DStringAppendElement (dstring, agentId -> getServerName().value());
    
    if (agentId -> getServerIp() == UNKNOWN_IP) {
	Tcl_DStringAppendElement (dstring, "");
    } else {
	char *ipString = tcpip_IPToString (agentId -> getServerIp());
	Tcl_DStringAppendElement (dstring, ipString);
	delete [] ipString;
    }

    Tcl_DStringAppendElement (dstring, agentId -> getName().value());

    char idString[16];
    sprintf (idString, "%d", agentId -> getId());
    Tcl_DStringAppendElement (dstring, idString);
}

#ifdef TCL8

/* Agent_IdToTclListObj

   Purpose: Convert an AgentId to a Tcl list OBJECT

     Input: agentId = the agent identification
                      (AgentId *)
  
    Output: The procedure constructs and returns a dynamically allocated
	    list object (Tcl_Obj *) that contains the agent identification.
*/ 

Tcl_Obj *Agent_IdToTclListObj (AgentId *agentId)
{
    Tcl_Obj *listObject;
    Tcl_Obj *listElements[4];

	// construct the list elements -- server name, server IP address,
	// agent name and agent id number

    listElements[0] = Tcl_NewStringObj (
	agentId -> getServerName().value(),
	agentId -> getServerName().length()
    );

    if (agentId -> getServerIp() == UNKNOWN_IP) {
	listElements[1] = Tcl_NewStringObj ("", 0);
    } else {
	char *ipString = tcpip_IPToString (agentId -> getServerIp());
	listElements[1] = Tcl_NewStringObj (ipString, -1);
	delete [] ipString;
    }

    listElements[2] = Tcl_NewStringObj (
	agentId -> getName().value(),
	agentId -> getName().length()
    );

    listElements[3] = Tcl_NewIntObj (agentId -> getId());

	// construct and return the list

    listObject = Tcl_NewListObj (4, listElements);
    return (listObject);
}

#endif

/* Agent_SecurityToTclList

   Purpose: Convert a BASE_SECURITY into a Tcl list

     Input: security = security information
                       (BASE_SECURITY *)

    Output: The security information is converted into list form and appended 
            to the end of dstring (Tcl_DString *).
*/
 
void Agent_SecurityToTclList (BASE_SECURITY *security, Tcl_DString *dstring)
{
	// assertions

    assert (security != NULL);
    assert (dstring != NULL);

	// get the security pieces

    SEC_IDENTITY owner = security -> getOwner();
    SEC_IDENTITY machine = security -> getMachine();
    BOOLEAN ownerSigned = security -> isOwnerSigned();
    BOOLEAN agentAuthenticated = security -> isAgentAuthenticated();

	// construct the Tcl list

    Tcl_DStringStartSublist (dstring);
    Tcl_DStringAppendElement (dstring, owner.m_keyname.value());
    Tcl_DStringAppendElement (dstring, (char *) (owner.m_authenticated ? "owner-auth" : "owner-not-auth"));
    Tcl_DStringEndSublist (dstring);
    Tcl_DStringStartSublist (dstring);
    Tcl_DStringAppendElement (dstring, machine.m_keyname.value());
    Tcl_DStringAppendElement (dstring, (char *) (machine.m_authenticated ? "machine-auth" : "machine-not-auth"));
    Tcl_DStringEndSublist (dstring);
    Tcl_DStringAppendElement (dstring, (char *) (ownerSigned ? "owner-signed" : "machine-signed"));
    Tcl_DStringAppendElement (dstring, (char *) (agentAuthenticated ? "agent-auth" : "agent-not-auth"));
}

#ifdef TCL8

/* Agent_SecurityToTclListObj

   Purpose: Convert a BASE_SECURITY into a Tcl list OBJECT

     Input: security = security information
                       (BASE_SECURITY *)

    Output: The procedure constructs and returns a dynamically allocated
	    list object (Tcl_Obj *) that contains the security information.
*/
 
Tcl_Obj *Agent_SecurityToTclListObj (BASE_SECURITY *security)
{
    Tcl_Obj *listObject;
    Tcl_Obj *ownerSublistElements[2];
    Tcl_Obj *machineSublistElements[2];
    Tcl_Obj *listElements[4];

	// assertions

    assert (security != NULL);

	// get the security pieces

    SEC_IDENTITY owner = security -> getOwner();
    SEC_IDENTITY machine = security -> getMachine();
    BOOLEAN ownerSigned = security -> isOwnerSigned();
    BOOLEAN agentAuthenticated = security -> isAgentAuthenticated();

	// construct the sublists

    ownerSublistElements[0] = Tcl_NewStringObj 
	(owner.m_keyname.value(), owner.m_keyname.length());
    ownerSublistElements[1] = Tcl_NewStringObj
	(owner.m_authenticated ? "owner-auth" : "owner-not-auth", -1);

    machineSublistElements[0] = Tcl_NewStringObj
	(machine.m_keyname.value(), machine.m_keyname.length());
    machineSublistElements[1] = Tcl_NewStringObj
	(machine.m_authenticated ? "machine-auth" : "machine-not-auth", -1);

	// construct and return the main list

    listElements[0] = Tcl_NewListObj (2, ownerSublistElements);
    listElements[1] = Tcl_NewListObj (2, machineSublistElements);
    listElements[2] = Tcl_NewStringObj
	(ownerSigned ? "owner-signed" : "machine-signed", -1);
    listElements[3] = Tcl_NewStringObj
	(agentAuthenticated ? "agent-auth" : "agent-not-auth", -1);

    listObject = Tcl_NewListObj (4, listElements);
    return (listObject);
}

#endif
