/* Agent Tcl
   Bob Gray
   10 January 1996

   tclAgentMasks.cc

   This file reimplements class MASKS for use in the Tcl/Tk interpreters.

   Copyright (c) 1995-1998, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "suppStrings.h"
#include "agentAgentId.h"	// AgentId
#include "genAgentIO.h"		// AgentIO
#include "tclMakeLists.h"	// Agent_IdToTclList

#include "genAgentConn.h"
#include "genAgentConv.h"
#include "genConversion.h"
#include "genLocation.h"
#include "genMasks.h"
#include "genManager.h"
#include "genVersion.h"
#include "tclAgent.h"
#include "tclMasks.h"

    /* forward declarations */ 

static void maskNotifierSetup
	(ClientData clientData, int flags);

#ifdef TCL8

static void dummyFileProc 
	(ClientData clientData, int flags);

#else

static void maskNotifierCheck
	(ClientData clientData, int flags);

#endif

    /* symbolic names for the masks */

static char *maskNames[] = {
	"meeting",
	"message",
	"event"
};

/* MASK_HANDLER_TCL::getPackedSize

   Purpose: Calculate the packed size of the *language-specific* portion of
            the mask.
*/

UINT_32 MASK_HANDLER_TCL::getPackedSize (void) const
{
    UINT_32 length;
    UINT_32 size = 0;

    	/* length field */

    size += sizeof(UINT_32);

	/* actual string */

    if ((length = name.length()) > 0) {
	size += length;
    }

	/* done */

    return (size); 
}

/* MASK_HANDLER_TCL::pack

   Purpose: Pack up the language-specific portion of the the mask.
*/

char *MASK_HANDLER_TCL::pack (char *bp) const
{
    UINT_32 length;
    UINT_32 netLong;
    
	/* length of name */

    length  = name.length();
    netLong = htonl (length);
    bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));

	/* name */

    if (length != 0) {
	bp = fast_memcpy (bp, name.value(), length);
    }

	/* done */

    return (bp);
}

/* MASK_HANDLER_TCL::unpack 

   Purpose: Unpack the language-specific portion of the mask
*/

char *MASK_HANDLER_TCL::unpack (char *bp, char *end_bp)
{
    UINT_32 length;

	/* get the length of the name */

    if ((bp = breakout_long (bp, end_bp, length)) == NULL) {
	return ((char *) NULL);
    }

	/* done if the length was zero */

    if (length == 0) {
	return (bp);
    }

	/* make sure that we have all the name */

    if (bp + length > end_bp) {
	return ((char *) NULL);
    }

	/* remember the name */

    name.append (bp, length);
    bp += length;
    return (bp);
}

/* MASK_HANDLER_TCL::getStringRepresentation

   Purpose: Get a string representation of the mask handler

     Input: None

    Output: The procedure returns a dynamically allocated string that
            contains a representation of the mask handler.
*/

DynamicString MASK_HANDLER_TCL::getStringRepresentation (void) const
{
    return (name);
}

/* MASK_HANDLER_TCL::MASK_HANDLER_TCL

   Purpose: These procedures are the constructors for class MASK_HANDLER_TCL.

     Input: type = EVENT_HANDLER or INTERRUPT_HANDLER
		   (AgentMaskHandler::HandlerTypes)

	    name = name of the Tcl procedure
		   (DynamicString)
*/

MASK_HANDLER_TCL::MASK_HANDLER_TCL (AgentMaskHandler::HandlerTypes p_type):
    AgentMaskHandler (p_type),
    name ()
{
    // empty
}

MASK_HANDLER_TCL::MASK_HANDLER_TCL (AgentMaskHandler::HandlerTypes p_type, DynamicString p_name): 
    AgentMaskHandler (p_type),
    name (p_name)
{
    // empty
}

/* MASK_HANDLER_TCL::~MASK_HANDLER_TCL

   Purpose: This procedure is the destructor for class MASK_HANDLER_TCL.
*/

MASK_HANDLER_TCL::~MASK_HANDLER_TCL ()
{
    // empty
}

/* MASK_HANDLER_TCL::compare

   Purpose: Compare two handlers

     Input: handler = the second handler
		      (class AgentMaskHandler * ==> class MASK_HANDLER_TCL *)

    Output: The procedure returns 0 if the handlers are different and 1 if they
	    are the same.
*/

int MASK_HANDLER_TCL::compare (const AgentMaskHandler &handler) const
{
    const MASK_HANDLER_TCL *cast;

	/* cast the handler to MASK_HANDLER_TCL */

    cast = (MASK_HANDLER_TCL *) &handler;

	/* compare the two handlers */

    if (type != cast -> type) {
	return 0;
    }

    if (name != cast -> name) {
	return 0;
    }

    return 1;
}

/* MASK_HANDLER_TCL::clone

   Purpose: Clone the handler

     Input: None

    Output: The procedure returns a pointer to a dynamically allocated
            AgentMaskHandler structure that is a clone of the current instance.
*/

AgentMaskHandler *MASK_HANDLER_TCL::clone (void) const
{
    MASK_HANDLER_TCL *handler = new MASK_HANDLER_TCL (type, name);
    return ((AgentMaskHandler *) handler);
}
  
/* MASKS_TCL::MASKS_TCL

   Purpose: This procedure is the constructor for class MASKS_TCL.
 
     Input: p_agent = agent associated with this MASKS_TCL instance
		      (class AGENT_TCL *) 
*/

MASKS_TCL::MASKS_TCL (AGENT_TCL *p_agent): 
    AgentMasks 	 (p_agent),
    interp	 (NULL),
    masterInterp (NULL),
    modified	 (FALSE)
{
	/* create the Tcl interrupt handler */

    handlerAsync = Tcl_AsyncCreate (maskAsyncHandler, (ClientData) this);
}

/* MASKS_TCL::setInterp

   Purpose: Set up the Tcl interpreter

     Input: interp       = Tcl interpreter associated with this instance
		           (struct Tcl_Interp *)

	    masterInterp = master of that interpreter
			   (struct Tcl_Interp *)

    Output: The procedure sets up the Tcl interpreter.
*/

void MASKS_TCL::setInterp (Tcl_Interp *newInterp, Tcl_Interp *newMasterInterp)
{
    int flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* assertions on the parameters */

    assert (interp == NULL);
    assert (masterInterp == NULL);
    assert (newInterp != NULL);

	/* remember the interpreter */

    interp       = newInterp;
    masterInterp = newMasterInterp;

	/* set up the event-handling routines */

    AgentIO *io = agent -> get_io();
    assert (io != NULL);
#ifdef TCL8
    Tcl_CreateEventSource (maskNotifierSetup, (Tcl_EventCheckProc *) NULL, (ClientData) io);
#else
    Tcl_CreateEventSource (maskNotifierSetup, maskNotifierCheck, (ClientData) io);
#endif

	/* load the Tcl "mask" array and turn on the trace */

    refresh ();

    Tcl_TraceVar2 (interp, "mask", (char *) NULL, flags, maskTrace, (ClientData) this);
#ifndef TCL8
    Tcl_GlueVar2 (interp, "mask", (char *) NULL, TCL_GLOBAL_ONLY);    
#endif

    if (masterInterp != NULL) {
	Tcl_TraceVar2 (masterInterp, "mask", (char *) NULL, flags, maskTrace, (ClientData) this);
#ifndef TCL8
	Tcl_GlueVar2 (masterInterp, "mask", (char *) NULL, TCL_GLOBAL_ONLY);    
#endif
    }
}

/* MASKS_TCL::~MASKS_TCL

   Purpose: This procedure is the destructor for class ~MASKS_TCL.
*/

MASKS_TCL::~MASKS_TCL ()
{
	/* delete the Tcl asynchronous handler */

    Tcl_AsyncDelete (handlerAsync);

	/* nothing else to do if we do not have an interp */ 

    if (interp != NULL) {

	int flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	    /* delete the event-handling routines */

#ifdef TCL8
	Tcl_DeleteEventSource (maskNotifierSetup, (Tcl_EventCheckProc *) NULL, (ClientData) this);
#else
	Tcl_DeleteEventSource (maskNotifierSetup, maskNotifierCheck, (ClientData) this);
#endif

	    /* turn off the trace and delete the "mask" array */

	Tcl_UntraceVar2 (interp, "mask", (char *) NULL, flags, maskTrace, (ClientData) this);
	Tcl_UnsetVar2 (interp, "mask", (char *) NULL, TCL_GLOBAL_ONLY);

	if (masterInterp != NULL) {
	    Tcl_UntraceVar2 (masterInterp, "mask", (char *) NULL, flags, maskTrace, (ClientData) this);
	    Tcl_UnsetVar2 (masterInterp, "mask", (char *) NULL, TCL_GLOBAL_ONLY);
        } 
    }
}

/* MASKS::refresh

   Purpose: Refresh the Tcl array "mask" 

     Input: None

    Output: The procedure loads the current mask information into the Tcl
             array "mask".
*/

void MASKS_TCL::refresh (void)
{
    char temp[16];

	/* reload the array */

    modified = TRUE;

    for (int i = 0; i < NUM_ITEMS; i++) {

	sprintf (temp, "%d", getHandle(i));

	Tcl_SetVar2 (interp, "mask", maskNames[i], temp, TCL_GLOBAL_ONLY);

	if (masterInterp != NULL) {
	    Tcl_SetVar2 (masterInterp, "mask", maskNames[i], temp, TCL_GLOBAL_ONLY);
	} 
    }

    modified = FALSE;
}

/* maskTrace

   Purpose: This is the Tcl trace procedure that guards access to the "mask"
	    array.
*/

char *maskTrace (ClientData clientData, Tcl_Interp *interp, char *name1, char *name2, int flags)
{
    int i;
    int handle;
    char *value;
    MASKS_TCL *masks = (MASKS_TCL *) clientData;

    if (masks -> modified) {
	return NULL;
    }

	/* check mask handle on a WRITE */

    if (flags & TCL_TRACE_WRITES) {

	value = Tcl_GetVar2 (interp, name1, name2, TCL_GLOBAL_ONLY);

	if ((Tcl_GetInt (interp, value, &handle) != TCL_OK) || (handle < 0)) {
	    Tcl_ResetResult (interp);
	    Tcl_UnsetVar2 (interp, name1, name2, TCL_GLOBAL_ONLY);
	    masks -> refresh ();
	    return "invalid handle";
        }

	for (i = 0; i < NUM_ITEMS; i++) {
	    if (!strcmp(name2, maskNames[i])) {
		break;
	    }
	}

	if (i == NUM_ITEMS) {
	    Tcl_UnsetVar2 (interp, name1, name2, TCL_GLOBAL_ONLY);
	    return "invalid array element";
	}

	if (masks -> updateHandle (i, handle) != MASK_OK) {
	    Tcl_UnsetVar2 (interp, name1, name2, TCL_GLOBAL_ONLY);
	    masks -> refresh ();
	    return "no mask with that handle";
        }

	return NULL;
    }

	/* restore the array on an UNSET */

    if (!(flags & TCL_INTERP_DESTROYED)) {

	masks -> refresh ();
#ifndef TCL8
	Tcl_GlueVar2 (interp, "mask", NULL, TCL_GLOBAL_ONLY);
#endif

	if (flags & TCL_TRACE_DESTROYED) {
	    int traceFlags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;
	    Tcl_TraceVar2 (interp, "mask", NULL, traceFlags, maskTrace, (ClientData) masks);
	}   
    }

    return NULL;
}

/* MASKS_TCL::updateHandle

   Purpose: Change one of the three distinguished masks

     Input: which  = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM
		     (int)

            handle = mask handle 
		     (UINT_32)

    Output: The procedure returns one of the following codes.

            MASK_OK             = success
            MASK_INVALID_HANDLE = no mask with the specified handle
*/

int MASKS_TCL::updateHandle (int which, UINT_32 handle)
{
    int code;

	/* assertions on the parameters */

    assert ((which == MESSAGE_ITEM) || (which == MEETING_ITEM) || (which == EVENT_ITEM));

	/* update and refresh */

    if ((code = AgentMasks::updateHandle (which, handle)) == MASK_OK) {
	refresh ();
    }

    return (code);
}

/* MASKS_TCL::callTclBroken

   Purpose: Throw a background error that indicates a broken server connection

     Input: None

    Output: The procedure throws the background error.
*/

void MASKS_TCL::callTclBroken (void)
{
    char *infoSave = NULL;
    char *codeSave = NULL; 
    char *resultSave = NULL;

	/* save the result, errorCode and errorInfo */

    resultSave = strcpyWithAlloc (interp -> result);
    infoSave   = strcpyWithAlloc (Tcl_GetVar2 (interp, "errorInfo", (char *) NULL, TCL_GLOBAL_ONLY));
    codeSave   = strcpyWithAlloc (Tcl_GetVar2 (interp, "errorCode", (char *) NULL, TCL_GLOBAL_ONLY));

	/* call tkerror */

    Tcl_SetResult (interp, "broken connection to the server\n", TCL_STATIC);
    Tcl_BackgroundError (interp);

	/* restore the result, errorCode and errorInfo */

    Tcl_SetResult (interp, resultSave, TCL_DYNAMIC);

    if (codeSave == NULL) {
	Tcl_UnsetVar2 (interp, "errorCode", (char *) NULL, TCL_GLOBAL_ONLY);
    } else {
	Tcl_SetVar2 (interp, "errorCode", (char *) NULL, codeSave, TCL_GLOBAL_ONLY);
	delete (codeSave);   
    }

    if (infoSave == NULL) {
	Tcl_UnsetVar2 (interp, "errorInfo", (char *) NULL, TCL_GLOBAL_ONLY);
    } else {
	Tcl_SetVar2 (interp, "errorInfo", (char *) NULL, infoSave, TCL_GLOBAL_ONLY);
	delete (infoSave);   
    }
}

/* MASKS_TCL::callTclHandler

   Purpose: Call a specific Tcl interrupt or event handler

     Input: eventData = description of the event or interrupt
			(class EventData *)

    Output: The procedure calls the interrupt or event handler.
*/

void MASKS_TCL::callTclHandler (EventData *eventData)
{
    int code;
    DynamicString name; 
    INCOMING *item;
    char number[32];
    Tcl_DString script;
    Tcl_DString arguments;
    char *infoSave = NULL;
    char *codeSave = NULL; 
    char *resultSave = NULL;

	/* assertions on the parameters */

    assert (eventData != NULL);
    assert (eventData -> m_item != NULL);
    assert (eventData -> m_item -> id != NULL);
    assert ((eventData -> m_eventType == AgentMaskHandler::INTERRUPT_HANDLER) || (eventData -> m_eventType == AgentMaskHandler::EVENT_HANDLER));
    assert ((eventData -> m_itemType == MESSAGE_ITEM) || (eventData -> m_itemType == MEETING_ITEM) || (eventData -> m_itemType == EVENT_ITEM));

	/* initialize */ 

    item = eventData -> m_item;
    Tcl_DStringInit (&arguments);

	/* construct the argument list -- first the sender identification */

    Tcl_DStringStartSublist (&arguments);
    Agent_IdToTclList (item -> id, &arguments);
    Tcl_DStringEndSublist (&arguments);

	/* -- then the security information */

    Tcl_DStringStartSublist (&arguments);
    Agent_SecurityToTclList (item -> security, &arguments);
    Tcl_DStringEndSublist (&arguments);

	/* -- then different things depending on the item type */

    if (eventData -> m_itemType == MESSAGE_ITEM) {
	
	INCOMING_MESSAGE *message = (INCOMING_MESSAGE *) item;
	sprintf (number, "%d", message -> code);
	Tcl_DStringAppendElement (&arguments, number);
	Tcl_DStringAppendElement (&arguments, message -> string);

    } else if (eventData -> m_itemType == MEETING_ITEM) {

	INCOMING_MEETING *meeting = (INCOMING_MEETING *) item;
	sprintf (number, "%u", meeting -> localHandle);
	Tcl_DStringAppendElement (&arguments, number);

    } else {

	INCOMING_EVENT *event = (INCOMING_EVENT *) item;
	Tcl_DStringAppendElement (&arguments, event -> tag);
	Tcl_DStringAppendElement (&arguments, event -> string);
    }

	/* save result, errorCode and errorInfo if this is an interrupt */

    if (eventData -> m_eventType == AgentMaskHandler::INTERRUPT_HANDLER) {
	resultSave = strcpyWithAlloc (interp -> result);
	infoSave   = strcpyWithAlloc (Tcl_GetVar2 (interp, "errorInfo", (char *) NULL, TCL_GLOBAL_ONLY));
        codeSave   = strcpyWithAlloc (Tcl_GetVar2 (interp, "errorCode", (char *) NULL, TCL_GLOBAL_ONLY));
    }

	/* loop through the individual handlers */

    for (int i = 0; i < eventData -> m_count; i++) {

	    /* assert that we have a handler */

	assert (eventData -> m_handlers[i] != NULL);

	    /* get the handler name */

	name = ((MASK_HANDLER_TCL *) eventData -> m_handlers[i]) -> copy_name ();

	    /* construct the appropriate script */

	Tcl_DStringInit (&script);
	Tcl_DStringAppend (&script, name.value(), -1);
	Tcl_DStringAppend (&script, " ", -1);
	Tcl_DStringAppend (&script, Tcl_DStringValue(&arguments), -1);

	    /* execute the script */

	Tcl_AllowExceptions (interp);

	if (eventData -> m_eventType == AgentMaskHandler::INTERRUPT_HANDLER) {
	    code = Tcl_Eval (interp, Tcl_DStringValue(&script));
	} else {
	    code = Tcl_GlobalEval (interp, Tcl_DStringValue(&script));
        }

	    /* done with the script */

	Tcl_DStringFree (&script);

	    /* handle the result code */

	if ((code != TCL_OK) && (code != TCL_CONTINUE)) {

	    if (code == TCL_BREAK) {
		break;
	    }

	    Tcl_AddErrorInfo (interp, "\n    (handler for an incoming item)");
	    Tcl_BackgroundError (interp);
	    break;
	}
    }

	/* restore result, errorCode and errorInfo if this is an interrupt */

    if (eventData -> m_eventType == AgentMaskHandler::INTERRUPT_HANDLER) {

	Tcl_SetResult (interp, resultSave, TCL_DYNAMIC);

	if (codeSave == NULL) {
	    Tcl_UnsetVar2 (interp, "errorCode", (char *) NULL, TCL_GLOBAL_ONLY);
	} else {
	    Tcl_SetVar2 (interp, "errorCode", (char *) NULL, codeSave, TCL_GLOBAL_ONLY);
	    delete (codeSave);   
	}

	if (infoSave == NULL) {
	    Tcl_UnsetVar2 (interp, "errorInfo", (char *) NULL, TCL_GLOBAL_ONLY);
	} else {
	    Tcl_SetVar2 (interp, "errorInfo", (char *) NULL, infoSave, TCL_GLOBAL_ONLY);
	    delete (infoSave);   
	}
    }

	/* cleanup */

    Tcl_DStringFree (&arguments);
}

/* MASKS_TCL::callTclInterrupts

   Purpose: Call the Tcl handlers associated with any pending interrupts

     Input: None

    Output: The procedure calls the Tcl handlers.
*/

void MASKS_TCL::callTclInterrupts (void)
{
    EventData *eventData;

	/* loop through all of the pending interrupts */

    while ((eventData = getNextInterrupt ()) != NULL) {
	callTclHandler (eventData);
	delete (eventData);
    }
}

/* MASKS_TCL::callTclEvents

   Purpose: Call the Tcl handlers associated with any pending events

     Input: None

    Output: The procedure calls the Tcl handlers.
*/

void MASKS_TCL::callTclEvents (void)
{
    EventData *eventData;

	/* loop through all the pending events */

    while ((eventData = getNextEvent ()) != NULL) {
	callTclHandler (eventData);
	delete (eventData);
    }
}

void maskEventHandler (ClientData clientData)
{
    MASKS_TCL *masks = (MASKS_TCL *) clientData;
    assert (masks != NULL);
    masks -> callTclEvents ();     
}

/* maskAsyncHandler

    Purpose: This procedure is called when event or interrupt handlers need to
 	     be fired. 

      Input: clientData = the MASKS_TCL structure
			  (class MASKS_TCL *)

	     interp     = the Tcl interpreter
			  (struct Tcl_Interp *)

	     code       = the Tcl result code
			  (int)

     Output: The procedure fires off the event or interrupt handlers.
*/

int maskAsyncHandler (ClientData clientData, Tcl_Interp *, int code)
{
    MASKS_TCL *masks;

	/* MASKS_TCL structure */

    masks = (MASKS_TCL *) clientData;
    assert (masks != NULL);

	/* call the Tcl interrupt handlers */

    if (masks -> pendingInterrupts()) {
	masks -> callTclInterrupts (); 
    }

	/* fire off a Tcl event so that we call the event */
	/* handlers when we are next in the event loop    */

    if (masks -> pendingEvents()) {
	Tcl_CreateTimerHandler (0, maskEventHandler, (ClientData) masks);
    } 

	/* return the same code that was passed in */

    return (code);
}

/* maskNotifierSetup and maskNotifierCheck 

   Purpose: These procedures are the two event-handling procedures that
            make the Tcl event loop recognize incoming communication from
            other agents.  The Tcl event loop calls both procedures
            automatically.  maskNotifierSetup is called just before the
            Tcl event loop drops into its timed wait -- maskNotifierSetup
            tells the Tcl core to watch the file descriptor of the background
            connection between the agent and the server.  maskNotifierCheck
            is called right after the Tcl event loop returns from its timed
            wait -- maskNotifierCheck currently does nothing since the
	    incoming communication is actually processed by a Tcl interrupt 
            handler.
*/

void maskNotifierSetup (ClientData clientData, int flags)
{
    int connectionFd;
    AgentConnection *connection;

	/* return immediately if we are not looking for file events */

    if (!((flags & TCL_ALL_EVENTS) || (flags & TCL_FILE_EVENTS))) {
	return;
    }

	/* get the MASKS_TCL structure */

    AgentIO *io = (AgentIO *) clientData;
    assert (io != NULL);

	/* get the fd of the background connection */ 

    if ((connection = io -> getBackgroundConnection()) == NULL) {
	return;
    }

    connectionFd = connection -> getFd ();
    assert (connectionFd > 0);

	/* tell the Tcl notifier to watch the given file */

#ifdef TCL8
    Tcl_CreateFileHandler (connectionFd, TCL_READABLE, dummyFileProc, (ClientData) NULL);
#else
    Tcl_WatchFile (Tcl_GetFile ((ClientData) connectionFd, TCL_UNIX_FD), TCL_READABLE);
#endif
}

#ifdef TCL8

void dummyFileProc (ClientData, int)
{
#ifdef FIX
	/*
	 * Arrivals are actually handled using a Tcl interrupt handler
	 * (i.e., agentAsyncIncoming in tclAgent.cc ) so we do not do
	 * anything here since the interrupt handler has already been fired
	 * and will be called on the next go through the event loop.
	 * Eventually it would be good to add sufficient logic so that we
	 * call *either* the interrupt handler *or* this routine but not both.
	 * In that case this routine would do much the same thing that
	 * agentAsyncIncoming does now.  (We use interrupt handlers since we
	 * want to get incoming communication off the background connection as 
	 * quickly as possible even if the agent never enters an event loop.)
	 */
#endif
}

#else

void maskNotifierCheck (ClientData, int)
{
#ifdef FIX
	/*
	 * Arrivals are actually handled using a Tcl interrupt handler
	 * (i.e., agentAsyncIncoming in tclAgent.cc ) so we do not do
	 * anything here since the interrupt handler has already been fired
	 * and will be called on the next go through the event loop.
	 * Eventually it would be good to add sufficient logic so that we
	 * call *either* the interrupt handler *or* this routine but not both.
	 * In that case this routine would do much the same thing that
	 * agentAsyncIncoming does now.  (We use interrupt handlers since we
	 * want to get incoming communication off the background connection as 
	 * quickly as possible, even if the agent never enters an event loop.)
	 */
#endif
}

#endif
