/* Agent Tcl
   Bob Gray
   16 June 1996

   tclMeetings.cc

   This file reimplements class MEETINGS for use in the Tcl/Tk interpreters.

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "genConversion.h"
#include "tclAgent.h"
#include "tclMeeting.h"
#include "tclMakeLists.h"
#include "genUtility.h"

    /* symbolic name for the meeting array */

static char *const arrayName = "meetings";

    /* forward declarations */

static char *meetingTrace
        (ClientData clientData, Tcl_Interp *interp, char *name1, char *name2,
         int flags);

static void meetingNotifierSetup
	(ClientData clientData, int flags);

#ifdef TCL8

static void dummyFileProc 
	(ClientData clientData, int flags);

#else

static void meetingNotifierCheck
	(ClientData clientData, int flags);

#endif

/* meetingTrace

   Purpose: This is the variable trace routine for the "meeting" array.

     Input: clientData = the associated MEETINGS_TCL structure
			 (ClientData ==> class MEETINGS_TCL *)

	    interp     = the Tcl interpreter 
		         (struct Tcl_Interp *)

	    name1      = array portion of the variable name
			 (char *)

            name2      = element portion of the variable name
			 (char *)

            flags      = flags that indicate type of access
			 (int)

    Output: The procedures puts the appropriate value into the array
            variable on a READ and prevents UNSETS and WRITES.
*/
 
static char *meetingTrace (ClientData clientData, Tcl_Interp *interp, char *, char *name2, int flags)
{
    UINT_32 count;
    int handle;
    char buffer[24];
    Tcl_DString dstring;
    MEETING_INFO *meeting;
    int tflags = TCL_GLOBAL_ONLY | TCL_TRACE_READS | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* meetings */

    MEETINGS_TCL *meetings = (MEETINGS_TCL *) clientData;
    assert (meetings != NULL);

	/* allow the operation if modified flag is SET */

    if (meetings -> allowModification()) {
      return ((char *) NULL);
    }

	/* set the variable appropriately on a read */

    if (flags & TCL_TRACE_READS) {

	    /* either "count" or a meeting handle -- check for "count" */

	if (!strcmp(name2, "count")) {
            count = meetings -> getCount ();
            fastIntegerToAscii (count, buffer);
            Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
            return ((char *) NULL);
	}

	    /* get the meeting information */

	if (Tcl_GetInt (interp, name2, &handle) != TCL_OK) {
	    return "does not exist";
	}

	if ((meeting = meetings -> lookupMeeting ((UINT_32) handle)) == NULL) {
	    return "does not exist";
	}
	
	fastIntegerToAscii (meeting -> localHandle, buffer);
    	char *staString = NULL;
 
	if (meeting -> messageReady) {
	    staString = Agent_MeetingStatusToString (MEET_MESSAGE_READY);
	} else if (meeting -> fileReady) {
	    staString = Agent_MeetingStatusToString (MEET_FILE_READY);
	} else {
            staString = Agent_MeetingStatusToString (meeting -> status);
	}

        Tcl_DStringInit (&dstring);
	Tcl_DStringAppendElement (&dstring, buffer);
	Tcl_DStringAppendElement (&dstring, staString);

	Tcl_DStringStartSublist (&dstring);
	Agent_IdToTclList (&(meeting -> id), &dstring);
	Tcl_DStringEndSublist (&dstring);

	Tcl_DStringStartSublist (&dstring);
	Agent_SecurityToTclList (&(meeting -> security), &dstring);
	Tcl_DStringEndSublist (&dstring);

	Tcl_SetVar2 (interp, arrayName, name2, Tcl_DStringValue (&dstring), TCL_GLOBAL_ONLY);

        Tcl_DStringFree (&dstring);
	delete (staString);
	delete (meeting);
	return ((char *) NULL);
    }
     
	/* restore original values on a write */

    if (flags & TCL_TRACE_WRITES) {
	Tcl_UnsetVar2 (interp, arrayName, name2, TCL_GLOBAL_ONLY);
	meetings -> refresh ();
	return "can not overwrite meetings";
    }

	/* otherwise we are trying to unset                 */

    if (!(flags & TCL_INTERP_DESTROYED)) {

	meetings -> refresh ();

	if (flags & TCL_TRACE_DESTROYED) {
	    Tcl_TraceVar2 (interp, arrayName, NULL, tflags, meetingTrace, (ClientData) meetings);  
	}

#ifndef TCL8
	Tcl_GlueVar2 (interp, arrayName, NULL, TCL_GLOBAL_ONLY);
#endif
	return "can not unset meetings";
    }

    return ((char *) NULL);
}

/* MEETINGS_TCL::MEETINGS_TCL

   Purpose: This procedure is the constructor for class MEETINGS_TCL.
 
     Input: p_agent = agent associated with this MEETINGS_TCL instance
		      (class AGENT_TCL *) 
*/

MEETINGS_TCL::MEETINGS_TCL (AGENT_TCL *p_agent): 
    AgentMeetings (p_agent),
    modified      (e_FALSE),
    interp	  (NULL),
    masterInterp  (NULL)
{
    // empty
}

/* MEETINGS_TCL::~MEETINGS_TCL

   Purpose: This procedure is the destructor for class MEETINGS_TCL.
*/

MEETINGS_TCL::~MEETINGS_TCL ()
{
	/* nothing to do if we do not have an interpreter */

    if (interp != NULL) {
 
	int flags = TCL_GLOBAL_ONLY | TCL_TRACE_READS | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	    /* delete the event-handling routines */

#ifdef TCL8
	Tcl_DeleteEventSource (meetingNotifierSetup, (Tcl_EventCheckProc *) NULL, (ClientData) this);
#else
	Tcl_DeleteEventSource (meetingNotifierSetup, meetingNotifierCheck, (ClientData) this);
#endif

	    /* turn off the variable traces */

	Tcl_UntraceVar2 (interp, arrayName, (char *) NULL, flags, meetingTrace, (ClientData) this);
	Tcl_UnsetVar2 (interp, arrayName, (char *) NULL, TCL_GLOBAL_ONLY);

	if (masterInterp != NULL) {
	    Tcl_UntraceVar2 (masterInterp, arrayName, (char *) NULL, flags, meetingTrace, (ClientData) this);
	    Tcl_UnsetVar2 (masterInterp, arrayName, (char *) NULL, TCL_GLOBAL_ONLY);
	}
    }
}

/* MEETINGS_TCL::setInterp

   Purpose: Set the Tcl interpreter

     Input: interp       = Tcl interpreter associated with this instance
		           (struct Tcl_Interp *)

	    masterInterp = master of that interpreter
			   (struct Tcl_Interp *)
*/

void MEETINGS_TCL::setInterp (Tcl_Interp *newInterp, Tcl_Interp *newMasterInterp)
{
    int flags = TCL_GLOBAL_ONLY | TCL_TRACE_READS | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* assertions on the parameters */

    assert (interp == NULL);
    assert (masterInterp == NULL);
    assert (newInterp != NULL);

	/* remember the interpreter */

    interp       = newInterp;
    masterInterp = newMasterInterp;

	/* set up the event-handling routines */

#ifdef TCL8
    Tcl_CreateEventSource (meetingNotifierSetup, (Tcl_EventCheckProc *) NULL, (ClientData) this);
#else
    Tcl_CreateEventSource (meetingNotifierSetup, meetingNotifierCheck, (ClientData) this);
#endif

        /* load the Tcl array and turn on the traces */

    Tcl_SetVar2 (interp, arrayName, "count", "", TCL_GLOBAL_ONLY);
    Tcl_TraceVar2 (interp, arrayName, NULL, flags, meetingTrace, (ClientData) this);
#ifndef TCL8
    Tcl_GlueVar2 (interp, arrayName, NULL, TCL_GLOBAL_ONLY);
#endif

    if (masterInterp != NULL) {
	Tcl_SetVar2 (masterInterp, arrayName, "count", "", TCL_GLOBAL_ONLY);
        Tcl_TraceVar2 (masterInterp, arrayName, NULL, flags, meetingTrace, (ClientData) this);
#ifndef TCL8
        Tcl_GlueVar2 (masterInterp, arrayName, NULL, TCL_GLOBAL_ONLY);
#endif
    }
} 

/* MEETINGS_TCL::new_handle

   Purpose: Callback when a new meeting handle is created

     Input: handle = the new handle
		     (UINT_32)

    Output: The procedure adds the handle to the "meeting" array.
*/

void MEETINGS_TCL::new_handle (UINT_32 handle)
{
    char buffer[16];

	/* convert the handle to a string */

    fastIntegerToAscii (handle, buffer);
    set_string (buffer);
}

/* MEETINGS_TCL::remove_handle

   Purpose: Callback when a meeting handle is removed

     Input: handle = the new handle
		     (UINT_32)

    Output: The procedure removes the handle from the "meeting" array.
*/

void MEETINGS_TCL::remove_handle (UINT_32 handle)
{
    char buffer[16];

	/* convert the handle to a string */

    fastIntegerToAscii (handle, buffer);
    unset_string (buffer);
}

/* MEETINGS_TCL::refresh

   Purpose: Refresh the meeting array

     Input: None

    Output: The procedure refreshes the meeting array.
*/

void MEETINGS_TCL::refresh (void)
{
	/* set the meeting count */ 

    set_string ("count");

	/* load all the meeting handles */

    for (int which  = 0; which < NUM_MEET_STATES; ++which) {

	MEETING *meeting;

	AgentMeetingsIterator iterator (*this, which);

	while ((meeting = iterator.next()) != NULL) {
	    new_handle (meeting -> localHandle);
	}
    }
}

/* MEETINGS_TCL::set_string

   Purpose: Set an element of the "restrict" array to the empty string 

     Input: name   = name of the array element
		     (char *)

    Output: The procedure sets the element to the empty string.
*/

void MEETINGS_TCL::set_string (char *name)
{
	/* allow modification */

    modified = e_TRUE;

	/* do the agent interpreter */

    if (interp != NULL) {

	Tcl_SetVar2 (interp, arrayName, name, "", TCL_GLOBAL_ONLY);

	    /* do the master interpreter */

	if (masterInterp != NULL) {
	    Tcl_SetVar2 (masterInterp, arrayName, name, "", TCL_GLOBAL_ONLY);
	}
    }

	/* disallow modification */

    modified = e_FALSE;
}

/* MEETINGS_TCL::unset_string

   Purpose: Unset an element of the "restrict" array 

     Input: name   = name of the array element
		     (char *)

    Output: The procedure unsets the element.
*/

void MEETINGS_TCL::unset_string (char *name)
{
	/* allow modification */

    modified = e_TRUE;

	/* do the agent interpreter */

    if (interp != NULL) {

	Tcl_UnsetVar2 (interp, arrayName, name, TCL_GLOBAL_ONLY);

	    /* do the master interpreter */

	if (masterInterp != NULL) {
	    Tcl_UnsetVar2 (masterInterp, arrayName, name, TCL_GLOBAL_ONLY);
	}
    }
   
	/* disallow modification */

    modified = e_FALSE;
}

/* meetingNotifierSetup and meetingNotifierCheck 

   Purpose: These procedures are the two event-handling procedures that
            make the Tcl event loop recognize incoming communication from 
            other agents.  The Tcl event loop calls both procedures
            automatically.  meetingNotifierSetup is called just before the
            Tcl event loop drops into its timed wait -- meetingNotifierSetup
            tells the Tcl core to watch the file descriptor of each meeting
            that is in the ACCEPTING, CONNECTING or COMPLETE state.
            meetingNotifierCheck is called right after the Tcl event loop 
            returns from its timed wait -- meetingNotifierCheck currently does 
            nothing since the incoming communication is actually processed by 
            a Tcl interrupt handler.
*/

void meetingNotifierSetup (ClientData clientData, int flags)
{
    int sockfd;
    MEETING *meeting;

	/* return immediately if we are not looking for file events */

    if (!((flags & TCL_ALL_EVENTS) || (flags & TCL_FILE_EVENTS))) {
	return;
    }

	/* get the MEETINGS_TCL structure */

    MEETINGS_TCL *meetings = (MEETINGS_TCL *) clientData;
    assert (meetings != NULL);

	/* loop through all meetings that are in the ACCEPTING, CONNECTING */
	/* or COMPLETE states and add the fd of each meeting to Tcl's list */
	/* of watched file                                                 */

#ifdef FIX_LATER
	/* this is quite inefficient */
#endif

    AgentMeetingsIterator iterator (*meetings, MEET_ACCEPTING);

    while ((meeting = iterator.next()) != NULL) {
	sockfd = meeting -> getSockfd ();
#ifdef TCL8
	Tcl_CreateFileHandler (sockfd, TCL_READABLE, dummyFileProc, (ClientData) NULL);
#else
	Tcl_WatchFile (Tcl_GetFile ((ClientData) sockfd, TCL_UNIX_FD), TCL_READABLE);
#endif
    }

    iterator.reset (MEET_CONNECTING);
    
    while ((meeting = iterator.next()) != NULL) {
	sockfd = meeting -> getSockfd ();
#ifdef TCL8
	Tcl_CreateFileHandler (sockfd, TCL_WRITABLE, dummyFileProc, (ClientData) NULL);
#else
	Tcl_WatchFile (Tcl_GetFile ((ClientData) sockfd, TCL_UNIX_FD), TCL_WRITABLE);
#endif
    }

    iterator.reset (MEET_COMPLETE);

    while ((meeting = iterator.next()) != NULL) {
	sockfd = meeting -> getSockfd ();
#ifdef TCL8
	Tcl_CreateFileHandler (sockfd, TCL_READABLE, dummyFileProc, (ClientData) NULL);
#else
	Tcl_WatchFile (Tcl_GetFile ((ClientData) sockfd, TCL_UNIX_FD), TCL_READABLE);
#endif
    }
}

#ifdef TCL8

void dummyFileProc (ClientData, int)
{
#ifdef FIX
	/*
	 * Meeting status changes are actually handled using a Tcl interrupt 
         * handler (e.g., agentAsyncIncoming in tclAgent.cc) so we do not do
	 * anything here since the interrupt handler has already been fired
	 * and will be called on the next go through the event loop.
	 * Eventually it would be good to add sufficient logic so that we
	 * call *either* the interrupt handler *or* this routine but not both.
	 * In that case this routine would do much the same thing that
	 * agentAsyncIncoming does now.  (We use interrupt handlers since we
	 * want to get incoming communication off the background connection as 
	 * quickly as possible, even if the agent never enters an event loop.)
	 */
#endif
}

#else

void meetingNotifierCheck (ClientData, int)
{
#ifdef FIX
	/*
	 * Meeting status changes are actually handled using a Tcl interrupt 
         * handler (e.g., agentAsyncIncoming in tclAgent.cc) so we do not do
	 * anything here since the interrupt handler has already been fired
	 * and will be called on the next go through the event loop.
	 * Eventually it would be good to add sufficient logic so that we
	 * call *either* the interrupt handler *or* this routine but not both.
	 * In that case this routine would do much the same thing that
	 * agentAsyncIncoming does now.  (We use interrupt handlers since we
	 * want to get incoming communication off the background connection as 
	 * quickly as possible, even if the agent never enters an event loop.)
	 */
#endif
}

#endif
