/* Agent Tcl
   Bob Gray
   10 January 1996

   tclRestrict.cc

   This file reimplements class RESTRICT for use in the Tcl/Tk interpreters.

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.

   RCS information:

   $Id: tclRestrict.cc,v 1.9 1998/12/14 21:43:48 rgray Exp $

   $Log: tclRestrict.cc,v $
   Revision 1.9  1998/12/14 21:43:48  rgray
   many performance enhancements
   Mon Dec 14 16:43:48 EST 1998

   Revision 1.8  1998/08/10 14:57:05  rgray
   updated messaging system
   Mon Aug 10 10:57:05 EDT 1998

   Revision 1.7  1998/04/27 21:17:47  rgray
   fixed bugs and merged in Eric's code
   Mon Apr 27 17:17:47 EDT 1998

   Revision 1.6  1997/12/20 08:38:20  rgray
   checkin
   Sat Dec 20 08:38:20 GMT 1997

   Revision 1.5  1997/12/09 21:49:17  rgray
   eliminated the need for RESTRICT_TRAP
   Tue Dec  9 21:49:17 GMT 1997

   Revision 1.4  1997/12/09 10:02:25  rgray
   code cleanup
   Tue Dec  9 10:02:25 GMT 1997

   Revision 1.3  1997/12/06 13:51:29  rgray
   server now resolves IP addresses after it forks (so that we do not delay the machine startup process)
   Sat Dec  6 13:51:29 GMT 1997

   Revision 1.2  1997/10/30 16:37:26  rgray
   added MaskPacker and MaskUnpacker to genMasks.h and fixed some assertions
   Thu Oct 30 11:37:26 EST 1997

   Revision 1.1  1997/10/17 06:02:06  rgray
   Initial revision

   Revision 1.1  1997/10/17 05:51:26  rgray
   Initial revision

   Revision 1.1  1997/10/17 05:43:36  rgray
   Initial revision

   Revision 1.12  1997/05/09 12:09:39  rgray
   performance improvements

   Revision 1.11  1997/05/03 18:26:15  rgray
   fixed pragmas

// Revision 1.10  1997/04/10  16:55:59  rgray
// lots
//
   Revision 1.9  1997/03/29 23:22:12  rgray
   added SIGTERM handling to the server

   Revision 1.8  1997/03/28 16:41:44  rgray
   continued our quest for global permits

   Revision 1.7  1997/03/25 18:14:00  rgray
   test of RCS comments

   Revision 1.6  1997/03/25 18:12:01  rgray
   added RCS comments

*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platSystem.h"
#include "platTimers.h"
#include "platTimeval.h"
#include "agentPermit.h"	// class Permit
#include "genManager.h"
#include "genUtility.h"
#include "genVersion.h"
#include "tclRestrict.h"

    /* symbolic name for the restrict array */

static char *const arrayName = "restrict";
   
    /* forward declarations */

static char *restrictTrace
	(ClientData clientData, Tcl_Interp *interp, char *name1, char *name2,
	 int flags);

static int restrictAsyncProc 
	(ClientData data, Tcl_Interp *interp, int code);

/* restrictTrace

   Purpose: This is the variable trace routine for the "restrict" array.

     Input: clientData = the associated RESTRICT_TCL structure
			 (ClientData ==> class RESTRICT_TCL *)

	    interp     = the Tcl interpreter 
		         (struct Tcl_Interp *)

	    name1      = array portion of the variable name
			 (char *)

            name2      = element portion of the variable name
			 (char *)

            flags      = flags that indicate type of access
			 (int)

    Output: The procedures puts the appropriate value into the array
            variable on a READ and prevents UNSETS and WRITES.
*/

static void addIntegerElement (Tcl_DString &dstring, char *string, int flag, int thisFlag, UINT_32 duration, UINT_32 threshold, UINT_32 minimum, UINT_32 current)
{
    char buffer[128];

    Tcl_DStringStartSublist (&dstring);
    Tcl_DStringAppendElement (&dstring, string);

    if (!thisFlag) {
	Tcl_DStringAppendElement (&dstring, "none");
    } else {
	sprintf (buffer, "%u", duration);
	Tcl_DStringAppendElement (&dstring, buffer);
    }

    if (!thisFlag) {
	Tcl_DStringAppendElement (&dstring, "none");
    } else {
	sprintf (buffer, "%u", threshold);
	Tcl_DStringAppendElement (&dstring, buffer);
    }

    if (!flag) {
	Tcl_DStringAppendElement (&dstring, "none");
	Tcl_DStringAppendElement (&dstring, "none");
    } else {
	sprintf (buffer, "%u", minimum);
	Tcl_DStringAppendElement (&dstring, buffer);
	sprintf (buffer, "%u", minimum - current);
	Tcl_DStringAppendElement (&dstring, buffer);
    }

    Tcl_DStringEndSublist (&dstring);
}

static void addDoubleElement (Tcl_DString &dstring, char *string, int flag, int thisFlag, struct timeval duration, struct timeval threshold, struct timeval minimum, struct timeval current)
{
    char buffer[128];

    Tcl_DStringStartSublist (&dstring);
    Tcl_DStringAppendElement (&dstring, string);

    if (!thisFlag) {
	Tcl_DStringAppendElement (&dstring, "none");
    } else {
	sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (duration));
	Tcl_DStringAppendElement (&dstring, buffer);
    }

    if (!thisFlag) {
	Tcl_DStringAppendElement (&dstring, "none");
    } else {
	sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (threshold));
	Tcl_DStringAppendElement (&dstring, buffer);
    }

    if (!flag) {
	Tcl_DStringAppendElement (&dstring, "none");
	Tcl_DStringAppendElement (&dstring, "none");
    } else {
	sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (minimum));
	Tcl_DStringAppendElement (&dstring, buffer);
	sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (TimevalUtil::subTimevals (minimum, current)));
	Tcl_DStringAppendElement (&dstring, buffer);
    }

    Tcl_DStringEndSublist (&dstring);
}

static char *restrictTrace (ClientData clientData, Tcl_Interp *interp, char *, char *name2, int flags)
{
    int level;
    char buffer[24];
    PERMIT_SET *permit;
    Tcl_DString dstring;
    int tflags = TCL_GLOBAL_ONLY | TCL_TRACE_READS | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* restrictions */ 

    RESTRICT_TCL *restrict = (RESTRICT_TCL *) clientData;
    assert (restrict!= NULL);

	/* allow the operation if modified flag is SET */

    if (restrict -> allowModification()) {
      return ((char *) NULL);
    }

	/* set the variable appropriately on a read */

    if (flags & TCL_TRACE_READS) {

	    /* make sure that the variable exists */

	if (Tcl_GetVar2 (interp, arrayName, name2, TCL_GLOBAL_ONLY) == NULL) {
	    return "does not exist";
	}

	    /* remember the first character */

	char firstChar = *name2;

	    /* several cases */

	if ((firstChar == 'l') && !strcmp (name2, "levels")) {

	    level = restrict -> getLevel ();
	    fastIntegerToAscii (level, buffer);
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL);

	} else if ((firstChar == 'w') && !strcmp (name2, "wall-start")) {

	    struct timeval tv = restrict -> getStartTimeval ();
	    sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (tv));
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'w') && !strcmp (name2, "wall-current")) {

	    struct timeval tv = SystemUtil::getCurrentWall ();
	    sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (tv));
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

        } else if ((firstChar == 'w') && !strcmp (name2, "wall-elapsed")) {

	    struct timeval tv = TimevalUtil::subTimevals (SystemUtil::getCurrentWall(), restrict -> getStartTimeval());
	    sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (tv));
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'c') && !strcmp (name2, "cpu-elapsed")) {

	    struct timeval tv = SystemUtil::getCurrentCpu ();
	    sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (tv));
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 
	
	} else if ((firstChar == 'j') && !strcmp (name2, "jumps")) {

	    sprintf (buffer, "%u", restrict -> getTotalJumps());
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'c') && !strcmp (name2, "children")) {

	    sprintf (buffer, "%u", restrict -> getTotalChildren());
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'd') && !strcmp (name2, "depth")) {

	    sprintf (buffer, "%u", restrict -> getChildDepth());
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'g') && !strcmp (name2, "group-wall-start")) {

	    struct timeval tv = restrict -> getGroupStartTimeval ();
	    sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (tv));
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 
	    
	} else if ((firstChar == 'g') && !strcmp (name2, "group-wall-elapsed")) {

	    struct timeval tv = TimevalUtil::subTimevals (SystemUtil::getCurrentWall(), restrict -> getGroupStartTimeval());
	    sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (tv));
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'g') && !strcmp (name2, "group-cpu-start")) {

	    struct timeval tv = restrict -> getGroupStartCpu ();
	    sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (tv));
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'g') && !strcmp (name2, "group-cpu-elapsed")) {

	    struct timeval tv = TimevalUtil::subTimevals (SystemUtil::getCurrentCpu(), restrict -> getGroupStartCpu());
	    sprintf (buffer, "%0.2f", TimevalUtil::timevalToSeconds (tv));
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'g') && !strcmp (name2, "group-jumps-start")) {

	    sprintf (buffer, "%u", restrict -> getGroupStartJumps());
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 
	    
	} else if ((firstChar == 'g') && !strcmp (name2, "group-jumps")) {

	    sprintf (buffer, "%u", restrict -> getTotalJumps() - restrict -> getGroupStartJumps());
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL);

	} else if ((firstChar == 'g') && !strcmp (name2, "group-children-start")) {

	    sprintf (buffer, "%u", restrict -> getGroupStartChildren());
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'g') && !strcmp (name2, "group-children")) {

	    sprintf (buffer, "%u", restrict -> getTotalChildren() - restrict -> getGroupStartChildren());
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'g') && !strcmp (name2, "group-depth-start")) {

	    sprintf (buffer, "%u", restrict -> getGroupStartDepth());
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else if ((firstChar == 'g') && !strcmp (name2, "group-depth")) {

	    sprintf (buffer, "%u", restrict -> getChildDepth() - restrict -> getGroupStartDepth());
	    Tcl_SetVar2 (interp, arrayName, name2, buffer, TCL_GLOBAL_ONLY);
	    return ((char *) NULL); 

	} else {

	    (void) Tcl_GetInt (interp, name2, &level);
	    permit = restrict -> getPermits (level);

	    Tcl_DStringInit (&dstring);

	    addDoubleElement 
		(dstring, 
		 "wall", 
		 permit -> minimum.haveWallLimit(), 
		 permit -> threshold.haveWallLimit(), 
		 permit -> duration.getWallLimit(), 
		 permit -> threshold.getWallLimit(), 
		 permit -> minimum.getWallLimit(), 
		 SystemUtil::getCurrentWall()
		);
	
	    addDoubleElement 
		(dstring, 
		 "cpu", 
		 permit -> minimum.haveCpuLimit(),
		 permit -> threshold.haveCpuLimit(),
		 permit -> duration.getCpuLimit(),
		 permit -> threshold.getCpuLimit(),
		 permit -> minimum.getCpuLimit(),
		 SystemUtil::getCurrentCpu()
		);

	    addIntegerElement 
		(dstring, 
		 "jumps", 
		 permit -> minimum.haveJumpLimit(),
		 permit -> threshold.haveJumpLimit(),
		 permit -> duration.getJumpLimit(),
		 permit -> threshold.getJumpLimit(),
		 permit -> minimum.getJumpLimit(),
		 restrict -> getTotalJumps()
		);

            addIntegerElement 
		(dstring, 
		 "children", 
		 permit -> minimum.haveChildrenLimit(),
		 permit -> threshold.haveChildrenLimit(),
		 permit -> duration.getChildrenLimit(),
		 permit -> threshold.getChildrenLimit(),
		 permit -> minimum.getChildrenLimit(),
		 restrict -> getTotalChildren()
		);

	    addIntegerElement 
		(dstring, 
		 "depth", 
		 permit -> minimum.haveDepthLimit(),
		 permit -> threshold.haveDepthLimit(),
		 permit -> duration.getDepthLimit(),
		 permit -> threshold.getDepthLimit(),
		 permit -> minimum.getDepthLimit(),
		 restrict -> getChildDepth()
		);

	    Tcl_SetVar2 (interp, arrayName, name2, Tcl_DStringValue (&dstring), TCL_GLOBAL_ONLY);
	    Tcl_DStringFree (&dstring);

	    delete (permit);
	    return ((char *) NULL);
	}
    }
     
	/* restore original values on a write */

    if (flags & TCL_TRACE_WRITES) {
	Tcl_UnsetVar2 (interp, arrayName, name2, TCL_GLOBAL_ONLY);
	restrict -> refresh ();
	return "can not overwrite restrictions";
    }

	/* otherwise we are trying to unset                 */

    if (!(flags & TCL_INTERP_DESTROYED)) {

	restrict -> refresh ();

	if (flags & TCL_TRACE_DESTROYED) {
	    Tcl_TraceVar2 (interp, arrayName, NULL, tflags, restrictTrace, (ClientData) restrict);  
	}

#ifndef TCL8
	Tcl_GlueVar2 (interp, arrayName, NULL, TCL_GLOBAL_ONLY);
#endif
	return "can not unset restrictions";
    }

    return ((char *) NULL);
}

/* RESTRICT_TCL::RESTRICT_TCL

   Purpose: This procedure is the constructor for class RESTRICT_TCL.

     Input: p_agent = agent associated with this RESTRICT_TCL structure
		      (class AGENT_TCL *)
*/

RESTRICT_TCL::RESTRICT_TCL (AGENT_TCL *p_agent): 
    RESTRICT (p_agent),
    m_interp (NULL), 
    m_masterInterp (NULL),
    m_pendingAlarm (e_FALSE),
    m_modified (e_FALSE)
{
    /* empty */
}

/* RESTRICT_TCL::~RESTRICT_TCL

   Purpose: This procedure is the destructor for class RESTRICT_TCL.
*/

RESTRICT_TCL::~RESTRICT_TCL ()
{
    int flags = TCL_GLOBAL_ONLY | TCL_TRACE_READS | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* nothing to do if we do not have a Tcl interpreter yet */

    if (m_interp != NULL) {

	    /* turn off the asynchronous Tcl handler */

        Tcl_AsyncDelete (m_handler);

	    /* turn off the variable trace */

        Tcl_UntraceVar2 (m_interp, arrayName, (char *) NULL, flags, restrictTrace, (ClientData) this);
        Tcl_UnsetVar2 (m_interp, arrayName, (char *) NULL, TCL_GLOBAL_ONLY);

        if (m_masterInterp != NULL) {
	    Tcl_UntraceVar2 (m_masterInterp, arrayName, (char *) NULL, flags, restrictTrace, (ClientData) this);
	    Tcl_UnsetVar2 (m_masterInterp, arrayName, (char *) NULL, TCL_GLOBAL_ONLY);
        }
    }
}

/* RESTRICT_TCL::setAllResultCodes

   Purpose: Set the result code of the main user interpreter and all its slave
            interpreters to the given code

     Input: code = a normal Tcl return code
                   (int)

    Output: The procedure sets the result code of the interpreters.

      Note: This procedure is necessary since there is no easy way to tell
            which interpreter is currently "active" -- i.e., restrictAsyncProc
            can not tell which interpreter should have its result code set to
            TCL_PERMIT so it simply sets the result code of all the interpreters
            to TCL_PERMIT.  To prevent the TCL_PERMIT from being incorrectly
            propagated all the way up the call stack (when that call stack
            spans multiple interpreters), the "restrict" command that catches 
	    the permit violation must set all the result codes to TCL_OK.  In
	    other words, this method is called at two locations, in  
            restrictAsyncProc to set the result codes to TCL_PERMIT and in
            Restrict_RestrictBody to set the result codes to TCL_OK.
*/

#ifndef TCL8

static void recursiveSetAllResultCodes (Tcl_Interp *interp, int code)
{
    Tcl_Interp **ptr;
    Tcl_Interp **interpList;

    interp -> resultCode = code;

    if ((interpList = Tcl_GetAllSlaves (interp)) == NULL) {
	return;
    }

    for (ptr = interpList; *ptr != NULL; ptr++) {
	recursiveSetAllResultCodes (*ptr, code);
    }

    delete (interpList);
}

#endif

void RESTRICT_TCL::setAllResultCodes (int code)
{
#ifndef TCL8
    recursiveSetAllResultCodes (m_interp, code);
#endif
}

/* RESTRICT_TCL::setInterp

   Purpose: Set up the Tcl interpreter

     Input: interp       = Tcl intepreter associated with this instance
		           (struct Tcl_Interp *)

	    masterInterp = master of that interpreter
			   (struct Tcl_Interp *)

    Output: The procedure sets up the Tcl interpreter.
*/

void RESTRICT_TCL::setInterp (Tcl_Interp *newInterp, Tcl_Interp *newMasterInterp)
{
    int flags = TCL_GLOBAL_ONLY | TCL_TRACE_READS | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

 	/* assertions on the parameters */

    assert (m_interp == NULL);
    assert (m_masterInterp == NULL);
    assert (newInterp != NULL);

	/* remember the interpreter */

    m_interp       = newInterp;
    m_masterInterp = newMasterInterp;

	/* create the Tcl asynchronous handler and see if we already need to call it */

    m_handler = Tcl_AsyncCreate (restrictAsyncProc, (ClientData) this);

    if (m_pendingAlarm) {
	Tcl_AsyncMark (m_handler);
	m_pendingAlarm = e_FALSE;
    }

	/* load the Tcl array and turn on the traces */

    refresh ();
    Tcl_TraceVar2 (m_interp, arrayName, NULL, flags, restrictTrace, (ClientData) this);  
#ifndef TCL8
    Tcl_GlueVar2 (m_interp, arrayName, NULL, TCL_GLOBAL_ONLY);
#endif

    if (m_masterInterp != NULL) {
	Tcl_TraceVar2 (m_masterInterp, arrayName, NULL, flags, restrictTrace, (ClientData) this);
#ifndef TCL8
	Tcl_GlueVar2 (m_masterInterp, arrayName, NULL, TCL_GLOBAL_ONLY);
#endif
    }
}

/* RESTRICT_TCL::set_level

   Purpose: Add the entries for an added permit to the "restrict" array
*/

void RESTRICT_TCL::set_level (void)
{
    int level;
    char buffer[16];
  
    level = getLevel (); 
    fastIntegerToAscii (level, buffer);
    set_string (buffer);
}

/* RESTRICT_TCL::unset_level

   Purpose: Remove the entries for a deleted permit to the "restrict" array
*/

void RESTRICT_TCL::unset_level (void)
{
    int level;
    char buffer[16];
 
    level = getLevel (); 
    fastIntegerToAscii (level + 1, buffer);
    unset_string (buffer);
}

/* RESTRICT_TCL::set_string

   Purpose: Set an element of the "restrict" array to the empty string 

     Input: name   = name of the array element
		     (char *)

    Output: The procedure sets the element to the empty string.
*/

void RESTRICT_TCL::set_string (char *name)
{
	/* allow modification */

    m_modified = e_TRUE;

	/* do the agent interpreter */

    if (m_interp != NULL) {

	Tcl_SetVar2 (m_interp, arrayName, name, "", TCL_GLOBAL_ONLY);

	    /* do the master interpreter */

	if (m_masterInterp != NULL) {
	    Tcl_SetVar2 (m_masterInterp, arrayName, name, "", TCL_GLOBAL_ONLY);
	}
    }

	/* disallow modification */

    m_modified = e_FALSE;
}

/* RESTRICT_TCL::unset_string

   Purpose: Unset an element of the "restrict" array 

     Input: name   = name of the array element
		     (char *)

    Output: The procedure unsets the element.
*/

void RESTRICT_TCL::unset_string (char *name)
{
	/* allow modification */

    m_modified = e_TRUE;

	/* do the agent interpreter */

    if (m_interp != NULL) {

	Tcl_UnsetVar2 (m_interp, arrayName, name, TCL_GLOBAL_ONLY);

	    /* do the master interpreter */

	if (m_masterInterp != NULL) {
	    Tcl_UnsetVar2 (m_masterInterp, arrayName, name, TCL_GLOBAL_ONLY);
	}
    }
   
	/* disallow modification */

    m_modified = e_FALSE;
}

/* RESTRICT_TCL::refresh

   Purpose: Refresh the elements of the "restrict" array

     Input: None

    Output: The procedure refreshes the elements of the "restrict" array.
*/

void RESTRICT_TCL::refresh (void)
{
    set_string ("levels");
    set_string ("wall-start");
    set_string ("wall-current");
    set_string ("wall-elapsed");
    set_string ("cpu-elapsed");
    set_string ("jumps");
    set_string ("children");
    set_string ("depth");
    set_string ("group-wall-start");
    set_string ("group-wall-elapsed");
    set_string ("group-cpu-start");
    set_string ("group-cpu-elapsed");
    set_string ("group-jumps-start");
    set_string ("group-jumps");
    set_string ("group-children-start");
    set_string ("group-children");
    set_string ("group-depth-start");
    set_string ("group-depth");

    char buffer[16];

    for (UINT_32 i = 0; i <= getLevel(); i++) {
	fastIntegerToAscii (i, buffer);
	set_string (buffer);
    }
}

/* restrictAsyncProc

   Purpose: This procedure is called whenever a POTENTITAL permit violation
	    has occurred.  The procedure is only called when the Tcl
	    interpreter is in a "safe" state.

     Input: data          = RESTRICT_TCL structure
			    (ClientData ==> RESTRICT_TCL *)

	    callingInterp = interpreter that called Tcl_AsyncInvoke
			    (Tcl_Interp *)

	    code          = current interpreter result code 
			    (int)

    Output: The procedure returns TCL_PERMIT if a permit violation has
	    occurred.  Otherwise the procedure passes through the current
	    result code.
*/

static int restrictAsyncProc (ClientData data, Tcl_Interp *callingInterp, int code)
{
    RESTRICT_TCL *restrict;

	/* restrictions */

    restrict = (RESTRICT_TCL *) data;
    assert (restrict != NULL);

	/* check the restrictions */

    if (restrict -> checkPermits() > 0) {

#ifdef TCL8

	abort_with_message ("Tcl8.0 does not yet handle restrictions");
	return (code);

#else	

	restrict -> setAllResultCodes (TCL_PERMIT);

#ifdef FIX_LATER
	    /* this is a hack to prevent a permit violation from firing   */
	    /* in the master interpreter and it depends on the master     */
	    /* interpreter never entering an event loop -- what is needed */
	    /* instead is a mechanism to turn checking for user-specified */
	    /* permits on and off                                         */
#endif

	Tcl_Interp *masterInterp = restrict -> get_master_interp ();

	if ((callingInterp != NULL) && (callingInterp == masterInterp)) {
	    return (code);
	} 

	return TCL_PERMIT;

#endif
    }

    return (code);
}

/* RESTRICT_TCL::pendingPermitViolation

   Purpose: Callback when there is a potential user-defined permit violation
*/

void RESTRICT_TCL::pendingPermitViolation (void)
{
    if (m_interp != NULL) {
	Tcl_AsyncMark (m_handler);
    } else {
	m_pendingAlarm = e_TRUE;
    }
}
