/* Agent Tcl
   Bob Gray
   17 April 1996

   tclSecurity.cc

   This file reimplements class SECURITY for use in the Tcl/Tk interpreters.
 
   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "genDefaults.h"
#include "tcl.h"
#include "tclAgent.h"
#include "tclSecurity.h"
#include "truefalse.h"

    /* forward declarations */

static char *securityTrace 
	(ClientData clientData, Tcl_Interp *interp, char *name1, char *name2, 
	 int flags);

/* SECURITY_TCL::SECURITY_TCL

   Purpose: This procedure is the constructor for class SECURITY_TCL.

     Input: p_agent = agent associated with this SECURITY_TCL instance
		      (class AGENT_TCL *)
*/

SECURITY_TCL::SECURITY_TCL (AGENT_TCL *p_agent):

	SECURITY	(p_agent),
	interp		((Tcl_Interp *) NULL),
	masterInterp	((Tcl_Interp *) NULL),
	modified	(e_FALSE)

{
	/* empty */
}

/* SECURITY_TCL::setInterp

   Purpose: Associate a Tcl interpeter with this instance

     Input: p_interp       = Tcl interpeter associated with this instance
		             (struct Tcl_Interp *)

	    p_masterInterp = master interpreter for that Tcl interpreter
			     (struct Tcl_Interp *)
*/

void SECURITY_TCL::setInterp (Tcl_Interp *p_interp, Tcl_Interp *p_masterInterp)
{
    int flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* assertions on the parameters */

    assert (interp == NULL);
    assert (masterInterp == NULL);
    assert (p_interp != NULL);

	/* remember the interpreter */

    interp       = p_interp;
    masterInterp = p_masterInterp;  

	/* load the Tcl array and turn on the traces */
 
    refresh ();

    Tcl_TraceVar2 (interp, "security", NULL, flags, securityTrace, (ClientData) this);  
#ifndef TCL8
    Tcl_GlueVar2 (interp, "security", NULL, TCL_GLOBAL_ONLY);
#endif

    if (masterInterp != NULL) {
	Tcl_TraceVar2 (masterInterp, "security", NULL, flags, securityTrace, (ClientData) this);
#ifndef TCL8
	Tcl_GlueVar2 (masterInterp, "security", NULL, TCL_GLOBAL_ONLY);
#endif
    }
}

/* SECURITY_TCL::~SECURITY_TCL

   Purpose: This procedure is the destructor for class SECURITY_TCL.
*/

SECURITY_TCL::~SECURITY_TCL ()
{
    int flags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* turn off the variable trace */

    Tcl_UntraceVar2 (interp, "security", (char *) NULL, flags, securityTrace, (ClientData) this);
    Tcl_UnsetVar2 (interp, "security", (char *) NULL, TCL_GLOBAL_ONLY);

    if (masterInterp != NULL) {
	Tcl_UntraceVar2 (masterInterp, "security", (char *) NULL, flags, securityTrace, (ClientData) this);
	Tcl_UnsetVar2 (masterInterp, "security", (char *) NULL, TCL_GLOBAL_ONLY);
    }
}

/* SECURITY_TCL::refresh_string

   Purpose: Reload a string element of the agent array

     Input: name   = name of the array element
		     (char *)

	    value   = new value for the element
		      (char *)

    Output: The procedure reloads the element.
*/

void SECURITY_TCL::refresh_string (char *name, char *value)
{
	/* assertions on the parameters */

    assert (name != NULL);

	/* allow modification */

    modified = e_TRUE;

	/* NULL is represented as the empty string */

    if (value == NULL) {
	value = "";
    } 

	/* reload in the interpreter and the master interpreter if present */

    Tcl_SetVar2 (interp, "security", name, value, TCL_GLOBAL_ONLY);

    if (masterInterp != NULL) {
	Tcl_SetVar2 (masterInterp, "security", name, value, TCL_GLOBAL_ONLY);
    }
	
	/* disallow modification */

    modified = e_FALSE;
} 
 
/* SECURITY_TCL::refresh_boolean

   Purpose: Reload a BOOLEAN element of the agent array

     Input: name   = name of the array element
		     (char *)

	    value  = true or false
		     (enum BOOLEAN)

	    format = desired format
		     (enum BOOL_FORMAT)

    Output: The procedure reloads the element.
*/

void SECURITY_TCL::refresh_boolean (char *name, BOOLEAN value, BOOL_FORMAT format)
{
    char *string = BooleanToString (value, format); 
    refresh_string (name, string);
} 

/* SECURITY_TCL::refresh

   Purpose: Reload the "security" array

     Input: None

    Output: The procedure reloads the "security" array.
*/

void SECURITY_TCL::refresh (void)
{
    refresh_boolean ("encryption", m_encryption, e_ON_OFF);
    refresh_boolean ("signatures", m_signatures, e_ON_OFF);
    refresh_boolean ("available", available, e_YES_NO);
    refresh_boolean ("owner-auth", m_owner.m_authenticated, e_YES_NO);
    refresh_boolean ("machine-auth", m_machine.m_authenticated, e_YES_NO);
    refresh_string ("owner", m_owner.m_keyname.value());
    refresh_string ("machine", m_machine.m_keyname.value());

    if (m_ownerSigned) {
	refresh_string ("signer", m_owner.m_keyname.value());
        refresh_boolean ("signer-auth", m_owner.m_authenticated, e_YES_NO);
    } else {
	refresh_string ("signer", m_machine.m_keyname.value());
        refresh_boolean ("signer-auth", m_machine.m_authenticated, e_YES_NO);
    }
}

/* securityTrace

   Purpose: This procedure traces the "security" array and prevents writes 
	    and unsets.
*/

static char *securityTrace (ClientData clientData, Tcl_Interp *interp, char *, char *name2, int flags)
{
    int tflags = TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS;

	/* SECURITY instace */

    SECURITY_TCL *security = (SECURITY_TCL *) clientData;
    assert (security != NULL);

	/* allow the operation if modified flag is SET */

    if (security -> allowModification()) {
      return ((char *) NULL); 
    }

	/* restore original values on a write */

    if (flags & TCL_TRACE_WRITES) {
	Tcl_UnsetVar2 (interp, "security", name2, TCL_GLOBAL_ONLY);
	security -> refresh ();
	return "can not overwrite security information";
    }

	/* otherwise we are trying to unset                 */

    if (!(flags & TCL_INTERP_DESTROYED)) {

	security -> refresh ();

	if (flags & TCL_TRACE_DESTROYED) {
	    Tcl_TraceVar2 (interp, "security", NULL, tflags, securityTrace, (ClientData) security);  
	}

#ifndef TCL8
	Tcl_GlueVar2 (interp, "security", NULL, TCL_GLOBAL_ONLY);
#endif
	return "can not unset security information";
    }

    return ((char *) NULL);
}

/* SECURITY_TCL::recordServerKeyname

   Purpose: Record the keyname of the agent's new server

     Input: id = new local identification
		 (const AgentId &)

    Output: The procedure records the server keyname.
*/

void SECURITY_TCL::recordServerKeyname (const AgentId &id)
{
    SECURITY::recordServerKeyname (id);
    refresh ();
}

/* SECURITY::throwServerKeyname

   Purpose: Discard the server keyname when the agent is no longer registered

     Input: None 

    Output: The procedure discards the server keyname.
*/

void SECURITY_TCL::throwServerKeyname (void)
{
    SECURITY::throwServerKeyname ();
    refresh ();
}

