/* Agent Tcl
   Bob Gray
   10 January 1996

   tclUtility.cc

   This file implements various utility routines.

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "suppStrings.h"
#include "tclInt.h"
#include "tclUtility.h"

/* Tcl_TransferResult

   Purpose: Transfer the result string, errorCode and errorInfo from one
	    interpreter to another.

     Input: dest = the destination interpreter
		   (struct Tcl_Interp *)

	    source = the source interpreter
		     (struct Tcl_Interp *)

    Output: The procedure copies the result string, errorCode and errorInfo
	    from the source interpreter into the destination interpreter.
*/

void Tcl_TransferResult (Tcl_Interp *dest, Tcl_Interp *source)
{
    char *string;

	/*
	 * Assertions on the parameters.
	 */
 
    assert (dest != NULL);
    assert (source != NULL);

	/*
	 * Copy the result code, result string, errorCode and errorInfo.
	 */

#ifndef TCL8
    dest -> resultCode = source -> resultCode;
#endif
    Tcl_SetResult (dest, source -> result, TCL_VOLATILE);

    if ((string = Tcl_GetVar2 (source, "errorInfo", (char *) NULL, TCL_GLOBAL_ONLY)) != NULL) {
	Tcl_SetVar2 (dest, "errorInfo", (char *) NULL, string, TCL_GLOBAL_ONLY);
    } else {
	Tcl_UnsetVar2 (dest, "errorInfo", (char *) NULL, TCL_GLOBAL_ONLY);
    }

    if ((string = Tcl_GetVar2 (source, "errorCode", (char *) NULL, TCL_GLOBAL_ONLY)) != NULL) {
	Tcl_SetVar2 (dest, "errorCode", (char *) NULL, string, TCL_GLOBAL_ONLY);
    } else {
	Tcl_UnsetVar2 (dest, "errorCode", (char *) NULL, TCL_GLOBAL_ONLY);
    }

	/* Copy the flags */

    ((Interp *) dest) -> flags |= (((Interp *) source) -> flags) & (ERR_ALREADY_LOGGED | ERR_IN_PROGRESS | ERROR_CODE_SET);
}

/* Tcl_DStringExtract 

   Purpose: Extract the contents of a dynamic Tcl string

     Input: dstring = the dynamic Tcl string
		      (Tcl_DString *)

    Output: The procedure returns a pointer to a dynamically allocated
	    C string that contains the contents of the dynamic Tcl string.
	    The dynamic Tcl string is reset to empty.
*/
 
char * Tcl_DStringExtract (Tcl_DString *dstring)
{
    char *string;

	/* extract the string out of the Tcl_DString */

    if (dstring -> string == dstring -> staticSpace) {
	string = strcpyWithAlloc (dstring -> string);
    } else {
	string                   = dstring -> string;
	dstring -> string         = dstring -> staticSpace;
	dstring -> staticSpace[0] = '\0';
        dstring -> length         = 0;
        dstring -> spaceAvl       = TCL_DSTRING_STATIC_SIZE;
    }

    return (string);
}
