/* Agent Tcl
   Bob Gray
   10 January 1996

   genConversion.cc

   This file implements various conversion routines.

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "suppStrings.h"
#include "suppDString.h"
#include "mesgTcpip.h"
#include "mesgMachId.h"		// MachineId class
#include "agentAgentId.h"	// AgentId class
#include "genUtility.h"

#include "genConversion.h"
#include "genMeeting.h"

/* Agent_IdToString

   Purpose: Convert an agent identification into a string

     Input: id     = agent identification
		     (struct AgentId *)

    Output: The procedure returns NULL on error or a pointer to a dynamically
            allocated string.
*/
 
char *Agent_IdToString (const struct AgentId *id)
{
    char temp[16];
    char *string;
    unsigned length = 0;
    char *ipString = NULL;

	/* calculate the length */ 

    DynamicString serverName = id -> getServerName();

    if (serverName.length() == 0) {
	length += 3;
    } else {
	length += serverName.length() + 1;
    }

    if (id -> getServerIp() == UNKNOWN_IP) {
	length += 3;
    } else {
        ipString = tcpip_IPToString (id -> getServerIp());
	length += strlen(ipString) + 1;
    }

    DynamicString name = id -> getName();

    if (name.length() == 0) {
	length += 3;
    } else {
	length += name.length() + 1;
    }
 
    sprintf (temp, "%d", id -> getId());
    length += strlen(temp);

	/* construct the string */

    string = new char [length + 1];
    strcpy (string, (serverName.length()  == 0) ? "{}" : serverName.value());
    strcat (string, " ");
    strcat (string, (id -> getServerIp() == UNKNOWN_IP) ? "{}" : ipString);
    strcat (string, " ");
    strcat (string, (name.length() == 0) ? "{}" : name.value());
    strcat (string, " ");
    strcat (string, temp);

	/* done */

    DELETE_ARRAY_OBJECT(ipString);
    return (string);
}

/* Agent_MachineToString

   Purpose: Convert a machine identification into a string

     Input: id = machine identification
	         (struct MachineId *)

    Output: The procedure returns NULL on error or a pointer to a dynamically
            allocated string.
*/

char *Agent_MachineToString (const MachineId *id)
{
    char *string;
    int length = 0;
    char *ipString = NULL;

	/* assertions on the parameters */

    assert (id != NULL);

	/* calculate the length */

    DynamicString name =  id -> getName();
    UINT_32 ip = id -> getIp();

    if (name.isEmpty()) {
	length += 3;
    } else {
	length += strlen (name.value()) + 1;
    }

    if (ip == UNKNOWN_IP) {
	length += 3;
    } else {
        ipString = tcpip_IPToString (ip);
	length += strlen (ipString) + 1;
    }

	/* construct the string */

    string = new char [length + 1];
    strcpy (string, name.isEmpty() ? "{}" : name.value());
    strcat (string, " ");
    strcat (string, (ip == UNKNOWN_IP) ? "{}" : ipString);

	/* done */

    DELETE_ARRAY_OBJECT(ipString);
    return (string);
}

/* Agent_SecurityToString

   Purpose: Convert base security information into a string

     Input: security = base security information
	               (struct BASE_SECURITY *)

    Output: The procedure returns NULL on error or a pointer to a dynamically
            allocated string.

      Note: The format for the string is

	    {{keyname} owner-auth/owner-not-auth} {{keyname} machine-auth/machine-not-auth} owner-encrypt/machine-encrypt agent-auth/agent-not-auth
*/

char *Agent_SecurityToString (const struct BASE_SECURITY *security)
{
    char *string;
    int length = 0;
    SEC_IDENTITY owner;
    SEC_IDENTITY machine;
    BOOLEAN ownerSigned;
    BOOLEAN agentAuthenticated;

	/* get the security pieces */

    owner = security -> getOwner ();
    machine = security -> getMachine ();
    ownerSigned = security -> isOwnerSigned ();
    agentAuthenticated = security -> isAgentAuthenticated ();

	/* calculate the length */

    length += owner.m_keyname.length();
    length += (owner.m_authenticated) ?  10 : 14;
    length += machine.m_keyname.length();
    length += (machine.m_authenticated) ?  12 : 16;
    length += (ownerSigned) ? 12 : 14;
    length += (agentAuthenticated) ? 10 : 14;
    length += 14;

	/* construct the string */

    string = new char [length + 1];
    strcpy (string, "{{");
    strcat (string, owner.m_keyname.value());
    strcat (string, "} ");
    strcat (string, (owner.m_authenticated) ? "owner-auth" : "owner-not-auth");
    strcat (string, "} {{");
    strcat (string, machine.m_keyname.value());
    strcat (string, "} ");
    strcat (string, (machine.m_authenticated) ? "machine-auth" : "machine-not-auth");
    strcat (string, "} ");
    strcat (string, (ownerSigned) ? "owner-signed" : "machine-signed");
    strcat (string, " ");
    strcat (string, (agentAuthenticated) ? "agent-auth" : "agent-not-auth");

	/* done */

    return (string);
}

/* Agent_MeetingStatusToString

   Purpose: Convert a meeting status to a string

     Input: status = the status
		     (UINT_8)

    Output: The procedure returns a dynamically allocated string. 
*/

char *Agent_MeetingStatusToString (const UINT_8 status)
{
    char *string;

    switch (status) {

	case MEET_REQUESTING:

	    string = "requesting";
	    break;

	case MEET_ACCEPTING:

	    string = "accepting";
	    break;

	case MEET_CONNECTING:

	    string = "connecting";
	    break;

	case MEET_REQUESTED:

	    string = "requested";
	    break;

	case MEET_COMPLETE:

	    string = "complete";
	    break;

	case MEET_REJECTED:

	    string = "rejected";
	    break;

	case MEET_FAILED:

	    string = "failed";
	    break;

	case MEET_TERMINATED:

	    string = "terminated";
	    break;

	case MEET_BROKEN:

	    string = "broken";
	    break;

	case MEET_REJECTING:

	    string = "rejecting";
	    break;

	case MEET_FILE_READY:
	
	    string = "file-ready";
	    break;

	case MEET_MESSAGE_READY:

	    string = "message-ready";
	    break;

	default:

	    string = "unknown";
    }
	
    return (strcpyWithAlloc(string));
}
